/**
 * fshows.com
 * Copyright (C) 2013-2021 All Rights Reserved.
 */
package
        com.fshows.ark.spring.boot.starter.core.mq.rocketmq.producer;

import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.AnnotatedBeanDefinition;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.BeanDefinitionHolder;
import org.springframework.beans.factory.config.RuntimeBeanReference;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.GenericBeanDefinition;
import org.springframework.context.annotation.ClassPathBeanDefinitionScanner;
import org.springframework.core.type.filter.AnnotationTypeFilter;
import org.springframework.core.type.filter.AssignableTypeFilter;
import org.springframework.util.StringUtils;

import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.Set;

/**
 * Rocket MQ 生产者注解扫描器
 *
 * @author liluqing
 * @version ClassPathRocketScanner.java, v 0.1 2021-08-14 10:24
 */
@Slf4j
public class ClassPathProducerScanner extends ClassPathBeanDefinitionScanner {

    /**
     * 需要扫描的注解
     */
    private Class<? extends Annotation> annotationClass;

    /**
     * 生产者bean
     */
    private Class<? extends RocketProducerFactoryBean> producerFactoryBeanClass = RocketProducerFactoryBean.class;

    // 1. 通过实现BeanDefinitionRegistryPostProcessor接口将注解扫描器scanner注册到spring容器中。
    // 2. 通过继承ClassPathBeanDefinitionScanner实现一个自定义注解扫描器。

    /**
     * 初始化bean扫描器
     *
     * @param registry
     */
    public ClassPathProducerScanner(BeanDefinitionRegistry registry) {
        super(registry, false);
    }

    /**
     * 注册注解扫描过滤器
     */
    public void registerFilters() {
        addIncludeFilter(new AnnotationTypeFilter(this.annotationClass,true, true));

        // exclude package-info.java
        addExcludeFilter((metadataReader, metadataReaderFactory) -> {
            String className = metadataReader.getClassMetadata().getClassName();
            return className.endsWith("package-info");
        });
    }

    /**
     * 执行bean扫描，并将所有bean的实现替换为producerFactoryBeanClass
     */
    @Override
    public Set<BeanDefinitionHolder> doScan(String... basePackages) {
        // 执行扫描操作
        Set<BeanDefinitionHolder> beanDefinitions = super.doScan(basePackages);
        // 如果为空则直接返回
        if (beanDefinitions.isEmpty()) {
            log.warn("Rocket MQ 生产者注解配置扫描 >> 没有从指定的basePackages目录扫描到MQ生产者配置!basePackages={}", basePackages);
            return beanDefinitions;
        }
        // 初始化rocket MQ生产者代理配的beanDefinition
        processBeanDefinitions(beanDefinitions);
        return beanDefinitions;
    }

    /**
     * rocket MQ生产者beanDefinition自定义处理
     *
     * @param beanDefinitions
     */
    private void processBeanDefinitions(Set<BeanDefinitionHolder> beanDefinitions) {
        // 遍历所有的扫描到的bean定义,将实际实现替换为producerFactoryBeanClass
        for (BeanDefinitionHolder holder : beanDefinitions) {
            GenericBeanDefinition definition = (GenericBeanDefinition) holder.getBeanDefinition();
            String beanClassName = definition.getBeanClassName();
            log.debug("创建RocketProducerFactoryBean >> beanName={}, beanClass={}" , holder.getBeanName(), beanClassName);

            // 将bean的实现替换为producerFactoryBean
            definition.getConstructorArgumentValues().addGenericArgumentValue(beanClassName);
            definition.setBeanClass(this.producerFactoryBeanClass);
            definition.setAutowireMode(AbstractBeanDefinition.AUTOWIRE_BY_TYPE);
        }
    }

    /**
     * 判断打了目标注解的类是否为接口，且是独立的非内部类
     *
     * {@inheritDoc}
     */
    @Override
    protected boolean isCandidateComponent(AnnotatedBeanDefinition beanDefinition) {
        return beanDefinition.getMetadata().isInterface() && beanDefinition.getMetadata().isIndependent();
    }

    /**
     * 检查bean名称是否合法
     *
     * {@inheritDoc}
     */
    @Override
    protected boolean checkCandidate(String beanName, BeanDefinition beanDefinition) {
        // 校验bean
        if (!super.checkCandidate(beanName, beanDefinition)) {
            log.warn("RocketProducerFactoryBean创建>> Spring容器中已存在相同的名称的bean,本次将跳过该bean!  beanName={}, beanClass={}", beanName, beanDefinition.getBeanClassName());
            return false;
        }
        return true;
    }

    public void setAnnotationClass(Class<? extends Annotation> annotationClass) {
        this.annotationClass = annotationClass;
    }

    public void setProducerFactoryBeanClass(Class<? extends RocketProducerFactoryBean> producerFactoryBeanClass) {
        this.producerFactoryBeanClass = producerFactoryBeanClass;
    }
}