/**
 * fshows.com
 * Copyright (C) 2013-2021 All Rights Reserved.
 */
package
        com.fshows.ark.spring.boot.starter.core.mq.rocketmq.producer;

import com.aliyun.openservices.ons.api.Message;
import com.aliyun.openservices.ons.api.SendResult;
import com.aliyun.openservices.ons.api.bean.ProducerBean;
import com.fshows.ark.spring.boot.starter.core.mq.base.FsMessage;
import com.fshows.ark.spring.boot.starter.core.mq.base.FsMessageConvert;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsMessageSendContext;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsSendErrorCallback;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsSendResult;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsSendSuccessCallback;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.IFshowsProducer;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.TransactionCommitCallback;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.TransactionMessageManage;
import com.fshows.ark.spring.boot.starter.core.mq.rocketmq.interceptor.ProducerInterceptorActuator;
import com.fshows.ark.spring.boot.starter.enums.ProducerSendTypeEnum;
import com.fshows.ark.spring.boot.starter.enums.ProducerTypeEnum;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.support.TransactionSynchronizationManager;

/**
 * 阿里云MQSDK producerImpl
 *
 * @author liluqing
 * @version AliyunFshowsProducer.java, v 0.1 2021-08-20 16:19
 */
@Slf4j
public class AliyunFshowsProducerImpl implements IFshowsProducer {

    /**
     * 实际的生产者bean
     */
    private ProducerBean producerBean;

    /**
     * 消息内容字符集编码
     */
    private String charsetName;

    /**
     * 拦截器执行器
     */
    private ProducerInterceptorActuator producerInterceptorActuator;

    /**
     * 事务管理器
     */
    private TransactionMessageManage transactionMessageManage;

    AliyunFshowsProducerImpl(ProducerBean producerBean, String charsetName) {
        this.producerBean = producerBean;
        this.charsetName = charsetName;
    }

    AliyunFshowsProducerImpl(ProducerBean producerBean, String charsetName
            , ProducerInterceptorActuator producerInterceptorActuator
            , TransactionMessageManage transactionMessageManage) {
        this.producerBean = producerBean;
        this.charsetName = charsetName;
        this.producerInterceptorActuator = producerInterceptorActuator;
        this.transactionMessageManage = transactionMessageManage;
    }

    @Override
    public void start() {
        producerBean.start();
    }

    @Override
    public void shutdown() {
        producerBean.shutdown();
    }

    @Override
    public FsSendResult send(FsMessage fsMessage) {
        FsMessageSendContext context = this.buildMsgContext(fsMessage, ProducerSendTypeEnum.SYNC,
                null, null);
        producerInterceptorActuator.doBeforeSend(context);
        FsSendResult fsSendResult = doSend(fsMessage);
        producerInterceptorActuator.doAfterSend(context, fsSendResult);
        return fsSendResult;
    }

    /**
     * 执行实际消息发送处理
     *
     * @param fsMessage
     * @return
     */
    private FsSendResult doSend(FsMessage fsMessage) {
        // 执行消息发送
        Message message = FsMessageConvert.convertMessage(fsMessage, this.charsetName);
        SendResult sendResult = producerBean.send(message);
        FsSendResult fsSendResult = FsMessageConvert.convertFsSendResult(sendResult);
        return fsSendResult;
    }

    @Override
    public void sendOneway(FsMessage fsMessage) {
        FsMessageSendContext context = this.buildMsgContext(fsMessage, ProducerSendTypeEnum.ONE_WAY,
                null, null);
        // 拦截器前置处理
        producerInterceptorActuator.doBeforeSend(context);

        doSendOneway(fsMessage);

        // 执行后置处理,oneway方式没有发送结果
        producerInterceptorActuator.doAfterSend(context, null);
    }

    /**
     * 执行Oneway消息发送处理
     *
     * @param fsMessage
     */
    private void doSendOneway(FsMessage fsMessage) {
        // 执行消息发送
        Message message = FsMessageConvert.convertMessage(fsMessage, this.charsetName);
        producerBean.sendOneway(message);
    }

    @Override
    public void sendAsync(FsMessage fsMessage, FsSendSuccessCallback sendSuccessCallback, FsSendErrorCallback fsSendErrorCallback) {
        FsMessageSendContext context = this.buildMsgContext(fsMessage, ProducerSendTypeEnum.ASYNC,
                sendSuccessCallback, fsSendErrorCallback);
        // 拦截器前置处理
        producerInterceptorActuator.doBeforeSend(context);

        this.doSendAsync(context, sendSuccessCallback, fsSendErrorCallback, producerInterceptorActuator);
    }

    /**
     * 构建消息上下文
     */
    private FsMessageSendContext buildMsgContext(FsMessage fsMessage, ProducerSendTypeEnum sendTypeEnum, FsSendSuccessCallback successCallback, FsSendErrorCallback errorCallback) {
        FsMessageSendContext context = new FsMessageSendContext();
        context.setProducerSendTypeEnum(sendTypeEnum);
        context.setProducerTypeEnum(ProducerTypeEnum.DEFAULT_NOMAL_PRODUCER);
        context.setSuccessCallback(successCallback);
        context.setErrorCallback(errorCallback);
        context.setFsMessage(fsMessage);
        return context;
    }

    /**
     * 执行消息异步发送
     *
     * @param context
     * @param sendSuccessCallback
     * @param fsSendErrorCallback
     */
    private void doSendAsync(FsMessageSendContext context, FsSendSuccessCallback sendSuccessCallback, FsSendErrorCallback fsSendErrorCallback, ProducerInterceptorActuator producerInterceptorActuator) {
        // 执行异步消息发送, 并且在消息回调中进行拦截器后置处理
        Message message = FsMessageConvert.convertMessage(context.getFsMessage(), this.charsetName);
        try {
            producerBean.sendAsync(message, new AliyunSendCallBackWrap(sendSuccessCallback, fsSendErrorCallback, context, producerInterceptorActuator));
        } catch (Exception e) {
            // 拦截器后置处理器
            producerInterceptorActuator.doAfterSend(context, new FsSendResult(e));
            throw e;
        }
    }

    @Override
    public void sendAsync(FsMessage fsMessage, FsSendSuccessCallback sendSuccessCallback) {
        FsMessageSendContext context = this.buildMsgContext(fsMessage, ProducerSendTypeEnum.ASYNC,
                sendSuccessCallback, null);
        // 拦截器前置处理
        producerInterceptorActuator.doBeforeSend(context);
        this.doSendAsync(context, sendSuccessCallback, null, producerInterceptorActuator);
    }

    @Override
    public void sendAsync(FsMessage fsMessage, FsSendErrorCallback fsSendErrorCallback) {
        FsMessageSendContext context = this.buildMsgContext(fsMessage, ProducerSendTypeEnum.ASYNC,
                null, fsSendErrorCallback);
        // 拦截器前置处理
        producerInterceptorActuator.doBeforeSend(context);
        this.doSendAsync(context, null, fsSendErrorCallback, producerInterceptorActuator);
    }

    @Override
    public FsSendResult send(FsMessageSendContext context) {
        if (ProducerTypeEnum.TRANSACTION_MESSAGE_PRODUCER.equals(context.getProducerTypeEnum())) {
            // 判断当前是否处于事务中，如果在事务中,则按照事务消息逻辑进行处理,否则按照普通消息正常发送
            if (transactionMessageManage.isInTransaction()) {
                return sendTransactionMsg(context);
            }
        }
        return sendNormalMsg(context);
    }

    /**
     * 发送事务消息
     *
     * @param context
     * @return
     */
    private FsSendResult sendTransactionMsg(FsMessageSendContext context) {
        // 拦截器前置处理
        producerInterceptorActuator.doBeforeSend(context);
        // 创建提交成功同步回调函数
        TransactionCommitCallback commitCallback =  (msgctx,fsSendResult)  -> {
            try {
                if (!fsSendResult.isSuccess()) {
                    return true;
                }
                // 执行拦截器后置处理
                producerInterceptorActuator.doAfterSend(context, fsSendResult);
                // 执行同步消息发送
                FsSendResult sendResult = doSend(context.getFsMessage());
                // 如果消息发送成功则更新库中的消息发送状态
                if (sendResult.isSuccess()) {
                    transactionMessageManage.updateMsgSendStatusToSuccess(msgctx);
                }
            } catch (Exception e) {
                log.error("ark-spring-boot-starter >> 事务消息后置同步处理异常！message={}", e, context.getFsMessage());
                return false;
            }
            return true;
        };
       return transactionMessageManage.sendTransactionMsg(context, commitCallback);
    }

    /**
     * 发送普通消息
     *
     * @param context
     * @return
     */
    private FsSendResult sendNormalMsg(FsMessageSendContext context) {
        FsSendResult fsSendResult = null;
        // 拦截器前置处理
        producerInterceptorActuator.doBeforeSend(context);
        if (ProducerSendTypeEnum.SYNC.equals(context.getProducerSendTypeEnum())) {
            try {
                fsSendResult = this.doSend(context.getFsMessage());
            } catch (Exception ex) {
                fsSendResult = new FsSendResult(ex);
                if (context.getRocketSendMessageMethodModel() != null
                        && context.getRocketSendMessageMethodModel().isSendErrorThrowEx()) {
                    throw ex;
                }
            } finally {
                producerInterceptorActuator.doAfterSend(context, fsSendResult);
            }
        }
        if (ProducerSendTypeEnum.ASYNC.equals(context.getProducerSendTypeEnum())) {
            this.doSendAsync(context, context.getSuccessCallback(), context.getErrorCallback(), producerInterceptorActuator);
        }
        if (ProducerSendTypeEnum.ONE_WAY.equals(context.getProducerSendTypeEnum())) {
            this.doSendOneway(context.getFsMessage());
            producerInterceptorActuator.doAfterSend(context, null);
        }
        return fsSendResult;
    }
}