/**
 * fshows.com
 * Copyright (C) 2013-2019 All Rights Reserved.
 */
package com.fshows.ark.spring.boot.starter.core.mq.rocketmq.consumer;

import com.aliyun.openservices.ons.api.Action;
import com.aliyun.openservices.ons.api.ConsumeContext;
import com.aliyun.openservices.ons.api.Message;
import com.aliyun.openservices.ons.api.MessageListener;
import com.fshows.ark.spring.boot.starter.core.mq.base.FsMessage;
import com.fshows.ark.spring.boot.starter.core.mq.base.consumer.FsConsumerModel;
import com.fshows.ark.spring.boot.starter.core.mq.rocketmq.interceptor.consumer.IConsumerInterceptorManagement;
import com.fshows.ark.spring.boot.starter.enums.ConsumerParamTypeEnum;
import com.fshows.ark.spring.boot.starter.enums.ConsumerReturnTypeEnum;
import com.fshows.ark.spring.boot.starter.exception.MQConsumerException;
import com.fshows.ark.spring.boot.starter.util.LogUtil;
import lombok.extern.slf4j.Slf4j;

import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * 消费者监听代理类
 *
 * @author Liluqing
 * @version RocketListenerProxy.java, v 0.1 2019-08-15 10:09
 */
@Slf4j
public class RocketListenerProxy implements MessageListener {

    /**
     * 消费者拦截器管理器
     */
    private IConsumerInterceptorManagement consumerInterceptorManagement;

    /**
     * 消费者元信息
     */
    private FsConsumerModel consumerModel;
    /**
     * 消费函数
     */
    private Method m;

    /**
     * 目标类
     */
    private Object target;

    public RocketListenerProxy(FsConsumerModel consumerModel, IConsumerInterceptorManagement consumerInterceptorManagement) {
        this.m = consumerModel.getListenerMethod();
        this.target = consumerModel.getTarget();
        this.consumerModel = consumerModel;
        this.consumerInterceptorManagement = consumerInterceptorManagement;
    }

    @Override
    public Action consume(Message message, ConsumeContext context) {
        Action action = Action.ReconsumeLater;
        // 消息格式转换
        FsMessage fsMessage = convertMessage(message);
        try {
            // 执行消费者拦截器前置处理
            if (consumerInterceptorManagement != null) {
                consumerInterceptorManagement.doBeforeConsume(fsMessage);
            }
            // 构建反射调用参数
            Object[] methodParam = createMethodParam(message, fsMessage);
            // 反射调用实际方法
            if (ConsumerReturnTypeEnum.VOID.equals(consumerModel.getReturnTypeEnum())) {
                m.invoke(target, methodParam);
                action = Action.CommitMessage;
            } else if (ConsumerReturnTypeEnum.ALIYUN_ACTION.equals(consumerModel.getReturnTypeEnum())) {
                Object o = m.invoke(target, methodParam);
                action = (Action)o;
            } else if (ConsumerReturnTypeEnum.BOOLEAN.equals(consumerModel.getReturnTypeEnum())) {
                Boolean bo = (Boolean)m.invoke(target, methodParam);
                action = Boolean.TRUE.equals(bo) ? Action.CommitMessage : Action.ReconsumeLater;
            }
        } catch (IllegalAccessException e) {
            LogUtil.error(log, "ark-spring-boot-starter >> 消费者方法调用失败，无访问权限！class={},m={}", e, target.getClass().getName(), m.getName());
        } catch (InvocationTargetException e) {
            LogUtil.error(log, "ark-spring-boot-starter >> 消费者业务异常！class={},m={}", e.getTargetException(), target.getClass().getName(), m.getName());
        } catch (Throwable e) {
            LogUtil.error(log, "ark-spring-boot-starter >> 消费者业务异常！class={},m={}", e, target.getClass().getName(), m.getName());
        } finally {
            // 消费后置处理
            if (consumerInterceptorManagement != null) {
                consumerInterceptorManagement.doAfterConsume(fsMessage,  Action.ReconsumeLater.equals(action));
            }
        }
        return action;
    }

    /**
     * 获取方法入参
     *
     * @param message
     * @return
     */
    private Object[] createMethodParam(Message message, FsMessage fsMessage) {
        // 入参是阿里云原始消息对象
        if (ConsumerParamTypeEnum.ALIYUN_MESSAGE.equals(consumerModel.getParamTypeEnum())) {
            return new Object[]{message};
        }
        // 入参是自定义消息对象
        if (ConsumerParamTypeEnum.FS_MESSAGE.equals(consumerModel.getParamTypeEnum())) {
            return new Object[]{fsMessage};
        }
        // 入参是字符串
        if (ConsumerParamTypeEnum.MESSAGE_BODY_STRING.equals(consumerModel.getParamTypeEnum())) {
            return new Object[]{fsMessage.getContent()};
        }
        return null;
    }

    /**
     * 阿里云message 转 FsMessage
     *
     * @param message
     * @return
     */
    private FsMessage convertMessage(Message message) {
        FsMessage fsMessage = new FsMessage();
        fsMessage.setTag(message.getTag());
        fsMessage.setTopic(message.getTopic());
        fsMessage.setReconsumeTimes(message.getReconsumeTimes());
        fsMessage.setSendTime(message.getStartDeliverTime() != 0L
                ? message.getStartDeliverTime() : message.getBornTimestamp());
        try {
            fsMessage.setContent(new String(message.getBody(), "UTF-8"));
        } catch (UnsupportedEncodingException e) {
            LogUtil.error(log, "ark-spring-boot-starter >> 消费者方法调用失败，消息内容提取失败！message={},m={}.{}()", e, message, target.getClass().getName(), m.getName());
            throw new MQConsumerException("ark-spring-boot-starter >> 消费者方法调用失败，消息内容提取失败");
        }
        fsMessage.setExtendParam(message.getUserProperties());
        return fsMessage;
    }
}