package cn.dalgen.mybatis.gen.model.repository;

import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import cn.dalgen.mybatis.gen.model.Config;
import cn.dalgen.mybatis.gen.model.config.CfEncrypt;
import cn.dalgen.mybatis.gen.model.config.CfEncryptColumn;
import cn.dalgen.mybatis.gen.model.config.CfOperation;
import cn.dalgen.mybatis.gen.model.config.CfTable;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import cn.dalgen.mybatis.gen.enums.MultiplicityEnum;
import cn.dalgen.mybatis.gen.enums.PagingCntTypeEnum;
import cn.dalgen.mybatis.gen.enums.ParamTypeEnum;
import cn.dalgen.mybatis.gen.exception.DalgenException;
import cn.dalgen.mybatis.gen.model.config.CfAssociation;
import cn.dalgen.mybatis.gen.model.config.CfCollection;
import cn.dalgen.mybatis.gen.model.config.CfColumn;
import cn.dalgen.mybatis.gen.model.config.CfResultMap;
import cn.dalgen.mybatis.gen.utils.CamelCaseUtils;
import org.dom4j.Attribute;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.tree.DefaultElement;

/**
 * SQL模板 文件解析
 *
 * @author bangis.wangdf
 * @date 15/12/6
 */
public class CfTableRepository {
    /**
     * The constant PARAM_PATTERN. 匹配格式: #{bbb,jdbcType=VARCHAR}
     */
    private static final Pattern PARAM_PATTERN = Pattern.compile("#\\{(.*?)\\}");

    /**
     * The constant PARAM_PATTERN.
     * 匹配格式:
     * aaa = #{bbb,jdbcType=VARCHAR}
     * item.aaa = #{bbb,jdbcType=VARCHAR}
     * aaa != #{bbb,jdbcType=VARCHAR}
     * aaa <> #{bbb,jdbcType=VARCHAR}
     */
    private static final Pattern PARAM_PATTERN2 = Pattern.compile("(\\w+)(\\.\\w+)?\\s*(=|!=|<>)(\\s*?)(.*?)#\\{(.*?)}");

    /**
     * The constant PARAM_PATTERN.
     * 匹配格式:
     * aaa LIKE CONCAT('%', #{bbb,jdbcType=VARCHAR},'%')
     * aaa LIKE CONCAT(#{bbb,jdbcType=VARCHAR},'%')
     * aaa LIKE CONCAT('%', #{bbb,jdbcType=VARCHAR})
     *
     * aaa NOT LIKE CONCAT('%', #{bbb,jdbcType=VARCHAR})
     * aaa NOT LIKE('%', #{bbb,jdbcType=VARCHAR})
     *
     * aaa LIKE "%"#{bbb,jdbcType=VARCHAR}"%"
     * aaa LIKE("%"#{bbb,jdbcType=VARCHAR}"%")
     */
    private static final Pattern PARAM_PATTERN3 = Pattern.compile("(?i)(\\w+)(\\.\\w+)?\\s*(like|not like|like concat|not like concat)(\\(?)(\\s*)(('%'|\"%\")\\s*,?\\s*)?#\\{(.*?)}");

    /**
     * The constant PARAM_PATTERN.
     * 匹配格式:
     * aaa in
     * <foreach collection="list" item="item" separator="," open="(" close=")">
     * #{item, jdbcType=VARCHAR}
     * </foreach>
     */
    private static final Pattern PARAM_PATTERN4 = Pattern.compile("(?i)(\\w+)(\\.\\w+)?\\s*(in|not in)\\s*(\\(?)\\s*(<foreach).*(\\s*)#\\{(.*?)}");

    /**
     * The constant STAR_BRACKET. 将 select * from 替换的正则
     */
    private static final Pattern STAR_BRACKET = Pattern.compile("\\((\\s*\\*\\s*)\\)");

    /**
     * 匹配?参数
     */
    private static final Pattern QUESTION_MARK_PATTERN = Pattern.compile("\\w+\\s*=\\s*\\?");

    private static final String JAVA_TYPE = "javatype\\s*=\\s*\".*?\"";

    private static final String LIKE_CONCAT = "(?i)\\s*(like|not like|like concat|not like concat)(\\(?)(\\s*)(('%'|\"%\")\\s*,?\\s*)?#";

    private static final String IN_FOREACH_CONCAT = "(?i)\\s*(in|not in)\\s*(\\(?)\\s*(<foreach).*(\\s*)#";

    /**
     * 从?参数中获取 column参数
     */
    private static final Pattern QUESTION_COLUMN_PATTERN = Pattern.compile("\\w{1,}");

    /**
     * The constant FOR_DESC_SQL_P. 为mapper.java的方法准备注释
     */
    private static final String FOR_DESC_SQL_P  = "\\s*<.*>\\s*";
    /**
     * The constant FOR_DESC_SQL_PN.
     */
    private static final String FOR_DESC_SQL_PN = "\\s{2,}";

    private static final String COUNT_X = "count\\(\\s*\\*\\s*\\)";

    /**
     * The constant ORDER_BY_PATTERN.
     */
    private static final String  ORDER_BY_PATTERN    = "[o|O][r|R][d|D][e|E][r|R]\\s+[b|B][y|Y]\\s+";
    /**
     * The constant SELECT_FROM_PATTERN. 正则表达式,贪婪匹配,勉强匹配 .* 贪婪 .*? 勉强,之匹配最近一个
     */
    private static final Pattern SELECT_FROM_PATTERN = Pattern
        .compile("[s|S][e|E][l|L][e|E][c|C][t|T]\\s+[\\s\\S]*?\\s+[f|F][r|R][o|O][m|M]");

    /**
     * Gain cf table cf table.
     *
     * @param table the table file
     * @return the cf table
     * @throws DocumentException the document exception
     */
    public CfTable gainCfTable(DefaultElement table, Map<String, Element> cfSqlMap, Config config) throws DocumentException {
        CfTable cfTable = new CfTable();

        cfTable.setSqlname(attr(table, "sqlname"));
        cfTable.setPhysicalName(attr(table, "physicalName"));
        cfTable.setRemark(attr(table, "remark"));
        cfTable.setSequence(attr(table, "sequence"));
        cfTable.setOrdinalEffectiveDay(attr(table, "ordinalEffectiveDay"));
        cfTable.setOrdinalMaxPosition(attrLong(table, "ordinalMaxPosition"));
        fillColumns(cfTable, table);

        fillEncryptKv(cfTable, table, config);

        fillResultMap(cfTable, table);

        fillOperation(cfTable, table, cfSqlMap);

        fillSql(cfTable, table);

        return cfTable;
    }

    private void fillSql(CfTable cfTable, DefaultElement table) {
        List<Element> elements = table.elements("sql");
        if (CollectionUtils.isNotEmpty(elements)) {
            for (Element element : elements) {
                cfTable.addSqlMap(element.attributeValue("id"), element.asXML());
            }
        }
    }

    /**
     * Fill result map.
     *
     * @param cfTable the cf table
     * @param table   the table
     */
    private void fillResultMap(CfTable cfTable, Element table) {
        List<Element> elements = table.elements("resultmap");
        if (CollectionUtils.isNotEmpty(elements)) {
            for (Element e : elements) {
                CfResultMap cfResultMap = new CfResultMap();
                cfResultMap.setName(attr(e, "name"));
                cfResultMap.setType(attr(e, "type"));
                cfResultMap.setRemark(attr(e, "remark"));
                final String extend = attr(e, "extend");
                if (StringUtils.isNotBlank(extend)) {
                    cfResultMap.setExtend(extend);
                }
                List<Element> ers = e.elements();
                for (Element er : ers) {
                    if (StringUtils.equals(er.getName(), "column")) {
                        CfColumn cfColumn = new CfColumn();
                        cfColumn.setName(attr(er, "name"));
                        cfColumn.setJavatype(attr(er, "javatype"));
                        cfColumn.setSqlType(attr(er, "jdbctype"));
                        cfColumn.setEncryptType(attr(er, "encryptType"));
                        cfColumn.setRemark(attr(er, "remark"));
                        cfColumn.setRelatedColumn(attr(er, "relatedColumn"));

                        cfResultMap.addColumn(cfColumn);
                    } else if (StringUtils.equals(er.getName(), "association")) {
                        CfAssociation cfAssociation = new CfAssociation();
                        cfAssociation.setColumn(attr(er, "column"));
                        cfAssociation.setProperty(attr(er, "property"));
                        cfAssociation.setSelect(attr(er, "select"));
                        cfAssociation.setRemark(attr(er, "remark"));
                        cfResultMap.addAssociation(cfAssociation);
                    } else if (StringUtils.equals(er.getName(), "collection")) {
                        CfCollection cfCollection = new CfCollection();
                        cfCollection.setColumn(attr(er, "column"));
                        cfCollection.setProperty(attr(er, "property"));
                        cfCollection.setSelect(attr(er, "select"));
                        cfCollection.setRemark(attr(er, "remark"));
                        cfResultMap.addCollection(cfCollection);
                    } else if (StringUtils.equals(er.getName(), "import")) {
                        cfResultMap.assImport(attr(er, "class"));
                    }
                }
                cfTable.addResultMap(cfResultMap);
            }
        }
    }

    /**
     * Fill operation.
     *
     * @param cfTable the cf table
     * @param table   the table
     */
    private void fillOperation(CfTable cfTable, DefaultElement table, Map<String, Element> cfSqlMap) {
        List<Element> operations = table.elements("operation");
        for (Element operation : operations) {

            CfOperation cfOperation = new CfOperation();
            cfOperation.setRemark(attr(operation, "remark"));
            cfOperation.setName(attr(operation, "name"));
            cfOperation.setMultiplicity(MultiplicityEnum.getByCode(attr(operation, "multiplicity")));
            cfOperation.setPagingCntType(PagingCntTypeEnum.getByCode(attr(operation, "pagingCntType")));
            cfOperation.setPaging(attr(operation, "paging"));
            cfOperation.setPagingCntOperation(attr(operation, "pagingCntOperation"));
            if (cfOperation.getMultiplicity() == MultiplicityEnum.paging) {
                Validate.notEmpty(cfOperation.getPaging(), "需要设置paging,用来生成分页类");
            }
            if (cfOperation.getPagingCntType() == PagingCntTypeEnum.pagingCustom) {
                Validate.notEmpty(cfOperation.getPagingCntOperation(),
                    "需要设置pagingCntSql,获取分页Operation");
                Validate.notEmpty(cfOperation.getPaging(), "需要设置paging,用来生成分页类");
            }
            cfOperation.setParamType(ParamTypeEnum.getByCode(attr(operation, "paramtype")));
            cfOperation.setResultmap(attr(operation, "resultmap"));
            cfOperation.setResulttype(attr(operation, "resulttype"));
            cfOperation.setKvMap(attr(operation, "kvmap"));
            cfOperation.setMapK(attr(operation, "mapK"));
            cfOperation.setMapV(attr(operation, "mapV"));
            if (!StringUtils.equals(cfOperation.getKvMap(), "false")) {
                Validate.isTrue(cfOperation.getMultiplicity() == MultiplicityEnum.many,
                    "转KvMap要求返回结果必须是many");
                Validate.isTrue(StringUtils.isNotBlank(cfOperation.getMapK()), "转KvMap要求必填mapK");
            }

            Map<String, String> encryptKv = cfTable.getEncryptKv();
            //sql内容
            setCfOperationCdata(cfTable, operation, table, cfOperation, cfSqlMap, encryptKv);

            cfTable.addOperation(cfOperation);
        }
    }

    /**
     * Sets cf operation cdata.
     *
     * @param cfTable     the cf table
     * @param oElement    the e
     * @param cfOperation the cf operation
     */
    private void setCfOperationCdata(CfTable cfTable, Element oElement, DefaultElement table, CfOperation cfOperation,
                                     Map<String, Element> cfSqlMap, Map<String, String> encryptKv4Cdata) {
        final Element extraparams = oElement.element("extraparams");
        oElement.remove(extraparams);

        List<Element> optimizePagings = oElement.elements("optimizePaging");

        String cXml = oElement.asXML();
        String[] lines = StringUtils.split(cXml, "\n");
        StringBuilder sb = new StringBuilder();
        for (int i = 1; i < lines.length - 1; i++) {
            if (i > 1) {
                sb.append("\n");
            }
            sb.append(lines[i]);
        }
        String cdata = sb.toString();
        //SQlDESC
        String sqlDesc = cdata.replaceAll(FOR_DESC_SQL_P, " ");
        sqlDesc = sqlDesc.replaceAll(FOR_DESC_SQL_PN, " ");
        cfOperation.setSqlDesc(sqlDesc);
        String text = oElement.getTextTrim();
        if (StringUtils.indexOf(text, "*") > 0) {

            cdata = StringUtils.replace(cdata, "sf.*", "<include refid=\"Base_SF_Column_List\" />");
            cdata = StringUtils.replace(cdata, "SF.*", "<include refid=\"Base_SF_Column_List\" />");
            cdata = StringUtils.replace(cdata, "sF.*", "<include refid=\"Base_SF_Column_List\" />");
            cdata = StringUtils.replace(cdata, "Sf.*", "<include refid=\"Base_SF_Column_List\" />");

            cdata = StringUtils.replace(cdata, ".*", "—-dotStar--");
            cdata = StringUtils.replace(cdata, "/*", "--/Star--");
            cdata = StringUtils.replace(cdata, "*/", "--Star/--");

            cdata = cdata.replaceAll(COUNT_X, "count(!!!COUNT!!!)");

            cdata = StringUtils.replace(cdata, " *", " <include refid=\"Base_Column_List\" />");
            cdata = StringUtils.replace(cdata, "    *",
                "   <include refid=\"Base_Column_List\" />");
            cdata = StringUtils.replace(cdata, "\n*", "\n<include refid=\"Base_Column_List\" />");

            cdata = StringUtils.replace(cdata, "—-dotStar--", ".*");
            cdata = StringUtils.replace(cdata, "--/Star--", "/*");
            cdata = StringUtils.replace(cdata, "--Star/--", "*/");

            cdata = cdata.replaceAll("count\\(!!!COUNT!!!\\)", " count(*) ");

        }

        cdata = cdata.replaceAll(JAVA_TYPE, " ");
        ParamTypeEnum paramType = cfOperation.getParamType();
        // 原生态参数场景，降低正则匹配的复杂度： 1.比如"a.`id`"转为"a.id"处理 2.去除无意义的空格
        if (ParamTypeEnum.primitive.equals(paramType)) {
            cdata = cdata.replace(".`", ".").replace("`", " ");
            cdata = cdata.replaceAll("\\s*\\(\\s*", "\\(")
                    .replaceAll("\\s+\\)",")")
                    .replaceAll("\\s+=", "=")
                    .replaceAll("\\s+#", " #") // 考虑需留空格：LIMIT #{startRowNum,jdbcType=INTEGER},#{pageSize,jdbcType=INTEGER}
                    .replaceAll("= #", "=#")
                    .replaceAll(" *, *", ",")
                    .replaceAll("(?i)not(\\s+)in", "not in")
                    .replaceAll("(?i)not(\\s+)like", "not like")
                    .replaceAll("(?i)like(\\s+)concat", "like concat");
        }

        //填充cdate中参数的typeHandler，目前仅只有加解密处理器
//        if (!encryptKv4Cdata.isEmpty()) {
//            cdata = fillCdataTypeHandler(cdata, cfOperation, encryptKv4Cdata);
//        }

        // 处理cdata中参数的encryptType
        cdata = dealCdataEncryptType(cdata, cfOperation);
        //pageCount添加
        setCfOperationPageCdata(cdata, cfOperation, optimizePagings);

        //分页配置
        if (cfOperation.getMultiplicity() == MultiplicityEnum.paging) {
            //其他扩展
            if (cfOperation.getPagingCntType() == PagingCntTypeEnum.pagingCustom) {
                int indexOf = cdata.indexOf("{pageLimit}");
                Validate.isTrue(indexOf > 0, "pagingCustom 需要使用 {pageLimit} 指定分页");
                cdata = cdata.replace("{pageLimit}", "limit #{startRow},#{limit}");
            }
            if (cfOperation.getPagingCntType() == PagingCntTypeEnum.optimize) {
                cdata = cdata.replaceAll("<optimizePaging>", "");
                cdata = cdata.replaceAll("</optimizePaging>", " limit #{startRow},#{limit} ");
            }
        }
        //? 参数替换 不指定类型
        cdata = delQuestionMarkParam(cdata, cfOperation, cfTable);

        //添加sql注释,以便于DBA 分析top sql 定位
        cfOperation.setCdata(addSqlAnnotation(cdata, cfOperation.getName(), cfTable.getSqlname()));

        //处理扩展参数
        dalExtraparams(extraparams, cfOperation);

        //获取参数
        fillOperationParams(oElement, table, cfOperation, cfSqlMap);
    }

    /**
     * 处理cdata中参数的encryptType，需要删掉
     *
     * @param cdata
     * @param cfOperation
     * @return
     */
    private String dealCdataEncryptType(String cdata, CfOperation cfOperation) {
        Matcher m = PARAM_PATTERN.matcher(cdata);
        List<String> params = Lists.newArrayList();
        while (m.find()) {
            params.add(m.group(1));
        }
        for (String _p : params) {
            // 统一处理为（无空格，无换行）
            String p = _p.replaceAll(" ", "").replaceAll("\n", "");
            cdata = cdata.replace(_p, p);

            for (String s : StringUtils.split(p, ",")) {
                if (s.contains("=")) {
                    if (StringUtils.startsWithIgnoreCase(s, "encryptType")) {
                        cdata = StringUtils.replace(cdata, "," + s, StringUtils.EMPTY);
                    }
                }
            }
        }
        cfOperation.setCdata(cdata);
        return cdata;
    }

    /**
     * 填充cdate中参数的typeHandler，目前仅只有加解密处理器
     *
     * @param cdata
     * @param cfOperation
     * @param encryptKv4Cdata
     * @return
     */
    private String fillCdataTypeHandler(String cdata, CfOperation cfOperation, Map<String, String> encryptKv4Cdata) {
        Matcher m;
        switch (cfOperation.getParamType()) {
            case object:
                // 匹配格式 "#{bbb,jdbcType=VARCHAR}"
                m = PARAM_PATTERN.matcher(cdata);
                while (m.find()) {
                    String originalValue = m.group(1);
                    // 统一处理为（无空格，无换行）
                    String value = originalValue.replaceAll(" ", "").replaceAll("\n", "");
                    cdata = cdata.replace(originalValue, value);

                    String columnName = StringUtils.split(value, ",")[0];
                    if (encryptKv4Cdata.containsKey(columnName) && !StringUtils.containsIgnoreCase(value, "typeHandler=")
                            && StringUtils.isNotBlank(encryptKv4Cdata.get(columnName))) {
                        String typeHandlerText = ",typeHandler=" + encryptKv4Cdata.get(columnName);
                        cdata = cdata.replace(value, value + typeHandlerText);
                    }
                }
                break;

            case objectList:
                // 匹配格式 "#{item.settingId,jdbcType=VARCHAR}"
                m = PARAM_PATTERN.matcher(cdata);
                while (m.find()) {
                    String originalValue = m.group(1);
                    // 统一处理为（无空格，无换行）
                    String value = originalValue.replaceAll(" ", "").replaceAll("\n", "");
                    cdata = cdata.replace(originalValue, value);

                    String attr = StringUtils.split(value, ",")[0];
                    if (StringUtils.contains(attr, ".")) {
                        // 兼容insertBatch的场景
                        attr = StringUtils.split(attr, ".")[1];
                    }
                    String columnName = attr;
                    if (encryptKv4Cdata.containsKey(columnName) && !StringUtils.containsIgnoreCase(value, "typeHandler=")
                            && StringUtils.isNotBlank(encryptKv4Cdata.get(columnName))) {
                        String typeHandlerText = ",typeHandler=" + encryptKv4Cdata.get(columnName);
                        cdata = cdata.replace(value, value + typeHandlerText);
                    }
                }
                break;

            case primitive:

                /* 匹配格式：
                  aaa = #{bbb,jdbcType=VARCHAR}
                  item.aaa = #{bbb,jdbcType=VARCHAR}
                  aaa != #{bbb,jdbcType=VARCHAR}
                  aaa <> #{bbb,jdbcType=VARCHAR}
                */
                m = PARAM_PATTERN2.matcher(cdata);
                while (m.find()) {
                    String originalValue = m.group(0);
                    // 第一步。获取表字段
                    // 统一处理为（无空格，无换行，"!=、<>"替换为"="）
                    String value = originalValue.replaceAll("!=", "=").replaceAll("<>", "=")
                            .replaceAll(" ", "").replaceAll("\n", "");
                    String attr = StringUtils.split(value, "=")[0];
                    String columnName = CamelCaseUtils.toCamelCase(attr);

                    // 第二步。如果该表字段为加密字段且未指定typeHandler，则替换文本自动填充加密typeHandler
                    if (encryptKv4Cdata.containsKey(columnName) && !StringUtils.containsIgnoreCase(value, "typeHandler=")
                            && StringUtils.isNotBlank(encryptKv4Cdata.get(columnName))) {
                        String partOne = originalValue.split("#")[0];
                        String needReplaceText = StringUtils.removeStart(value, attr + "=");
                        String typeHandlerText = ",typeHandler=" + encryptKv4Cdata.get(columnName);
                        cdata = cdata.replace(originalValue, partOne + needReplaceText.replaceAll("}", "") + typeHandlerText + "}");
                    }
                }

                /* 匹配格式:
                  aaa LIKE CONCAT('%', #{bbb,jdbcType=VARCHAR},'%')
                  aaa LIKE CONCAT(#{bbb,jdbcType=VARCHAR},'%')
                  aaa LIKE CONCAT('%', #{bbb,jdbcType=VARCHAR})

                  aaa NOT LIKE CONCAT('%', #{bbb,jdbcType=VARCHAR})
                  aaa NOT LIKE('%', #{bbb,jdbcType=VARCHAR})

                  aaa LIKE "%"#{bbb,jdbcType=VARCHAR}"%"
                  aaa LIKE("%"#{bbb,jdbcType=VARCHAR}"%")
                */
                m = PARAM_PATTERN3.matcher(cdata);
                while (m.find()) {
                    String originalValue = m.group(0);

                    // 第一步。获取表字段
                    // 统一处理为（like相关命令变更为"="处理）
                    String value = originalValue.replaceAll(LIKE_CONCAT, "=#");
                    String attr = StringUtils.split(value, "=")[0];
                    String columnName = CamelCaseUtils.toCamelCase(attr.trim());

                    // 第二步。如果该表字段为加密字段且未指定typeHandler，则替换文本自动填充加密typeHandler
                    if (encryptKv4Cdata.containsKey(columnName) && !StringUtils.containsIgnoreCase(originalValue, "typeHandler=")
                            && StringUtils.isNotBlank(encryptKv4Cdata.get(columnName))) {
                        String partOne = originalValue.split("#")[0];
                        String needReplaceText = StringUtils.removeStart(value, attr + "=");
                        String typeHandlerText = ",typeHandler=" + encryptKv4Cdata.get(columnName);
                        cdata = cdata.replace(originalValue, partOne + needReplaceText.replaceAll("}", "") + typeHandlerText + "}");
                    }
                }

                /* 匹配格式:
                  aaa in
                    <foreach collection="list" item="item" separator="," open="(" close=")">
                     #{item, jdbcType=VARCHAR}
                    </foreach>
                */
                m = PARAM_PATTERN4.matcher(cdata);
                while (m.find()) {
                    String originalValue = m.group(0);

                    // 第一步。获取表字段
                    // 统一处理为（in foreach相关命令变更为"="处理）
                    String value = originalValue.replaceAll(IN_FOREACH_CONCAT, "=#");
                    String attr = StringUtils.split(value, "=")[0];
                    String columnName = CamelCaseUtils.toCamelCase(attr.trim());

                    // 第二步。如果该表字段为加密字段且未指定typeHandler，则替换文本自动填充加密typeHandler
                    if (encryptKv4Cdata.containsKey(columnName) && !StringUtils.containsIgnoreCase(originalValue, "typeHandler=")
                            && StringUtils.isNotBlank(encryptKv4Cdata.get(columnName))) {
                        String partOne = originalValue.split("#")[0];
                        String needReplaceText = StringUtils.removeStart(value, attr + "=");
                        String typeHandlerText = ",typeHandler=" + encryptKv4Cdata.get(columnName);
                        cdata = cdata.replace(originalValue, partOne + needReplaceText.replaceAll("}", "") + typeHandlerText + "}");
                    }
                }
                break;

            case autoParamClass:
            default:
                break;
        }
        return cdata;
    }

    private static final String REPLACE_TMP = " ( ⊙ o ⊙ ) ";

    /**
     * Sets cf operation page cdata.
     *
     * @param cdata       the cdata
     * @param cfOperation the cf operation
     */
    private void setCfOperationPageCdata(String cdata, CfOperation cfOperation, List<Element> optimizePagings) {
        String forCount = cdata;
        //分页配置
        if (cfOperation.getMultiplicity() == MultiplicityEnum.paging) {

            if (cfOperation.getPagingCntType() != PagingCntTypeEnum.pagingCustom) {
                String cdataCount = null;
                if (cfOperation.getPagingCntType() == PagingCntTypeEnum.paging) {
                    Matcher selectFromMather = SELECT_FROM_PATTERN.matcher(forCount);
                    if (selectFromMather.find()) {
                        forCount = selectFromMather
                            .replaceFirst("SELECT\n          COUNT(*) AS total \n        FROM\n");
                    }

                    cdataCount = forCount.replaceAll(ORDER_BY_PATTERN, REPLACE_TMP);
                    int indexof = cdataCount.indexOf(REPLACE_TMP);
                    if (indexof > 0) {
                        cdataCount = cdataCount.substring(0, indexof).replaceAll(
                            "(?m)^\\s*$" + System.lineSeparator(), "");
                    }
                    Validate.notEmpty(cdataCount, "分页cdataCount处理异常");
                } else if (cfOperation.getPagingCntType() == PagingCntTypeEnum.optimize) {
                    //提速模式
                    Validate.notEmpty(optimizePagings, "optimize 模式必须配置 <optimizePagings>");
                    forCount = optimizePagings.get(0).asXML();
                    Matcher selectFromMather = SELECT_FROM_PATTERN.matcher(forCount);
                    if (selectFromMather.find()) {
                        forCount = selectFromMather
                            .replaceFirst("SELECT\n          COUNT(*) AS total \n        FROM\n");
                    }

                    cdataCount = forCount.replaceAll(ORDER_BY_PATTERN, REPLACE_TMP);
                    int indexof = cdataCount.indexOf(REPLACE_TMP);
                    if (indexof > 0) {
                        cdataCount = cdataCount.substring(0, indexof).replaceAll(
                            "(?m)^\\s*$" + System.lineSeparator(), "");
                    }
                    cdataCount = cdataCount.replaceAll("<optimizePaging>", "");
                    cdataCount = cdataCount.replaceAll("</optimizePaging>", "");
                    Validate.notEmpty(cdataCount, "分页cdataCount处理异常");

                } else if (cfOperation.getPagingCntType() == PagingCntTypeEnum.pagingExtCnt) {
                    cdataCount = forCount.replaceAll(ORDER_BY_PATTERN, REPLACE_TMP);
                    int indexof = cdataCount.indexOf(REPLACE_TMP);
                    if (indexof > 0) {
                        cdataCount = cdataCount.substring(0, indexof).replaceAll(
                            "(?m)^\\s*$" + System.lineSeparator(), "");
                    }
                    Validate.notEmpty(cdataCount, "分页cdataCount处理异常");
                    cdataCount = "        SELECT\n          COUNT(*) AS total \n        FROM(" +
                        cdataCount + ") as tmp";

                }
                cfOperation.setCdataPageCount(cdataCount);
            }
        }
    }

    /**
     * 处理扩展参数
     *
     * @param extraparams
     * @param cfOperation
     */
    private void dalExtraparams(Element extraparams, CfOperation cfOperation) {
        if (extraparams != null) {
            final List<Element> params = extraparams.elements();

            if (CollectionUtils.isNotEmpty(params)) {
                for (Element param : params) {
                    final String name = attr(param, "name");
                    CfColumn primitiveParamsCfColumn = new CfColumn();
                    primitiveParamsCfColumn.setName(name);
                    primitiveParamsCfColumn.setJavatype(attr(param, "javatype"));
                    cfOperation.addPrimitiveParam(name, primitiveParamsCfColumn);
                    String testVal = attr(param, "testVal");
                    if (StringUtils.isNotBlank(testVal)) {
                        cfOperation.addPrimitiveParamTestVal(name, testVal);
                    }
                }
            }
        }
    }

    /**
     * ? 参数替换
     *
     * @param cdata
     * @param cfOperation
     * @param cfTable
     * @return
     */
    private String delQuestionMarkParam(String cdata, CfOperation cfOperation, CfTable cfTable) {
        //
        if (!StringUtils.contains(cdata, '?')) {
            return cdata;
        }
        cfTable.getColumns();
        if (StringUtils.startsWithIgnoreCase(cfOperation.getName(), "insert")) {
            String sql = cdata;
            //sql 特殊处理一下
            sql = sql.replaceAll("\\s{1,}", "");
            sql = sql.replaceAll("\\(\\)", "");
            sql = sql.replaceAll("\\(", "\n(\n");
            sql = sql.replaceAll("\\)", "\n)\n");

            String[] sqlLines = StringUtils.split(sql, "\n");

            int i = 0;
            for (String sqlLine : sqlLines) {
                if (StringUtils.startsWith(sqlLine, "(")) {
                    break;
                }
                i++;
            }
            String insertLine = sqlLines[i + 1];
            String valueLine = sqlLines[i + 5];
            valueLine = valueLine.replaceAll("\\w{1},\\w{1}", "");
            String[] columns = StringUtils.split(insertLine, ',');
            String[] params = StringUtils.split(valueLine, ',');

            for (int j = 0; j < params.length; j++) {
                if (StringUtils.equals(params[j], "?")) {
                    try {
                        String columnParam = CamelCaseUtils.toCamelCase(columns[j]);
                        cdata = StringUtils.replace(cdata, "?", "#{" + columnParam + "}", 1);
                    } catch (ArrayIndexOutOfBoundsException e) {
                        throw new DalgenException("参数设置错误#{}中,未正确使用 table=" + cfTable.getSqlname());
                    }

                }
            }

        } else {
            Matcher questionMarkPatternResult = QUESTION_MARK_PATTERN.matcher(cdata);
            while (questionMarkPatternResult.find()) {
                Matcher columnMatcher = QUESTION_COLUMN_PATTERN
                    .matcher(questionMarkPatternResult.group());
                while (columnMatcher.find()) {
                    String columnParam = CamelCaseUtils.toCamelCase(columnMatcher.group());
                    cdata = StringUtils.replace(cdata, "?", "#{" + columnParam + "}", 1);

                    CfColumn primitiveParamsCfColumn = new CfColumn();
                    primitiveParamsCfColumn.setName(columnParam);
                    primitiveParamsCfColumn.setJavatype("");
                    cfOperation.addPrimitiveParam(columnParam, primitiveParamsCfColumn);
                }
            }
        }
        return cdata;
    }

    /**
     * Add sql annotation string.
     *
     * @param cdata  the cdata
     * @param oName  the o name
     * @param tbName the tb name
     * @return the string
     */
    private String addSqlAnnotation(String cdata, String oName, String tbName) {

        String sqlAnnotation = StringUtils.upperCase(CamelCaseUtils
            .toInlineName(CamelCaseUtils.toCamelCase("ms_" + tbName + "_" + oName)));
        if (StringUtils.startsWithIgnoreCase(oName, "insert ")
            || StringUtils.startsWithIgnoreCase(oName, "update")
            || StringUtils.startsWithIgnoreCase(oName, "delete")) {
            if (StringUtils.contains(cdata, "update ")) {
                return StringUtils.replace(cdata, "update ", "update /*" + sqlAnnotation + "*/ ");
            }
            if (StringUtils.contains(cdata, "UPDATE ")) {
                return StringUtils.replace(cdata, "UPDATE ", "UPDATE /*" + sqlAnnotation + "*/ ");
            }
            if (StringUtils.contains(cdata, "insert ")) {
                return StringUtils.replace(cdata, "insert ", "insert /*" + sqlAnnotation + "*/ ");
            }
            if (StringUtils.contains(cdata, "INSERT ")) {
                return StringUtils.replace(cdata, "INSERT ", "INSERT /*" + sqlAnnotation + "*/ ");
            }
            if (StringUtils.contains(cdata, "delete ")) {
                return StringUtils.replace(cdata, "delete ", "delete /*" + sqlAnnotation + "*/ ");
            }
            if (StringUtils.contains(cdata, "DELETE ")) {
                return StringUtils.replace(cdata, "DELETE ", "DELETE /*" + sqlAnnotation + "*/ ");
            }
        } else {
            if (StringUtils.contains(cdata, "select ")) {
                return StringUtils.replace(cdata, "select ", "select /*" + sqlAnnotation + "*/ ");
            }
            if (StringUtils.contains(cdata, "SELECT ")) {
                return StringUtils.replace(cdata, "SELECT", "SELECT /*" + sqlAnnotation + "*/ ");
            }
        }

        return cdata;
    }

    /**
     * Fill operation params. 原生态参数获取 添加List参数支持
     *
     * @param e           the e
     * @param cfOperation the cf operation
     */
    private void fillOperationParams(Element e, DefaultElement table, CfOperation cfOperation,
                                     Map<String, Element> cfSqlMap) {

        if (cfOperation.getParamType() != ParamTypeEnum.primitive) {
            return;
        }

        //取出foreach 用来判断是否有List参数
        List<DefaultElement> items = e.elements();

        if (CollectionUtils.isNotEmpty(items)) {
            for (DefaultElement item : items) {
                List<Element> ies = item.elements();
                if (StringUtils.endsWithIgnoreCase("include", item.getName())) {
                    Element includeElement = elementById(table, item.attributeValue("refid"));
                    if (includeElement == null) {
                        includeElement = cfSqlMap.get(item.attributeValue("refid"));
                    }
                    Validate.notNull(includeElement, "include refid=" + item.attributeValue("refid") + " 对应的SQL 未配置");
                    fillOperationParams(includeElement, table, cfOperation, cfSqlMap);
                } else if (CollectionUtils.isNotEmpty(ies)) {
                    recursiveForeachElement(cfOperation, table, ies, cfSqlMap);
                } //找到List参数
                else if (StringUtils.equalsIgnoreCase(item.getName(), "foreach")) {
                    String collName = item.attributeValue("collection");
                    String itemName = item.attributeValue("item");
                    Validate.notEmpty(collName, "foreach 元素设置错误 table=" + cfOperation.getName());
                    Validate.notEmpty(itemName, "foreach 元素设置错误 table=" + cfOperation.getName());
                    String javatype = item.attributeValue("javatype");
                    if (StringUtils.isNotBlank(javatype)) {
                        cfOperation.addPrimitiveForeachOtherParam("list_" + itemName, collName,
                            javatype);
                    } else {
                        cfOperation.addPrimitiveForeachParam(itemName, collName);
                    }
                }
            }
        }

        Matcher m = PARAM_PATTERN.matcher(e.asXML());
        List<String> params = Lists.newArrayList();
        while (m.find()) {
            params.add(m.group(1));
        }
        for (String _p : params) {
            String p = _p.replaceAll(" ", "");
            String attr = null;
            String type = null;
            String encryptType = null;
            for (String s : StringUtils.split(p, ",")) {
                if (s.contains("=")) {
                    if (StringUtils.startsWithIgnoreCase(s, "javaType")
                        || StringUtils.startsWithIgnoreCase(s, "jdbcType")) {
                        type = StringUtils.split(s, "=")[1].trim();
                    }
                    if (StringUtils.startsWithIgnoreCase(s, "encryptType")) {
                        encryptType = StringUtils.split(s, "=")[1].trim();
                    }
                } else {
                    attr = StringUtils.trim(s);
                }
            }
            CfColumn primitiveParamsCfColumn = new CfColumn();
            primitiveParamsCfColumn.setName(attr);
            primitiveParamsCfColumn.setJavatype(type);
            primitiveParamsCfColumn.setEncryptType(encryptType);
            cfOperation.addPrimitiveParam(attr, primitiveParamsCfColumn);
        }
    }

    private void recursiveForeachElement(CfOperation cfOperation, DefaultElement table, List<Element> ies,
                                         Map<String, Element> cfSqlMap) {
        for (Element ie : ies) {
            if (StringUtils.endsWithIgnoreCase("include", ie.getName())) {
                Element includeElement = elementById(table, ie.attributeValue("refid"));
                if (includeElement == null) {
                    includeElement = cfSqlMap.get(ie.attributeValue("refid"));
                }
                Validate.notNull(includeElement, "include refid=" + ie.attributeValue("refid") + " 对应的SQL 未配置");
                fillOperationParams(includeElement, table, cfOperation, cfSqlMap);
            } else if (StringUtils.equalsIgnoreCase(ie.getName(), "foreach")) {
                String collName = ie.attributeValue("collection");
                String itemName = ie.attributeValue("item");
                Validate.notEmpty(collName, "foreach 元素设置错误 table=" + cfOperation.getName());
                Validate.notEmpty(itemName, "foreach 元素设置错误 table=" + cfOperation.getName());
                String javatype = ie.attributeValue("javatype");
                if (StringUtils.isNotBlank(javatype)) {
                    cfOperation.addPrimitiveForeachOtherParam("list_" + itemName, collName,
                        javatype);
                } else {
                    cfOperation.addPrimitiveForeachParam(itemName, collName);
                }
                List<Element> _ies = ie.elements();
                if(CollectionUtils.isNotEmpty(_ies)){
                    recursiveForeachElement(cfOperation, table, _ies, cfSqlMap);
                }
            } else {
                if ((CollectionUtils.isNotEmpty(ie.elements()))) {
                    recursiveForeachElement(cfOperation, table, ie.elements(), cfSqlMap);
                }
            }
        }
    }

    /**
     * Fill columns.
     *
     * @param cfTable the cf table
     * @param table   the table
     */
    private void fillColumns(CfTable cfTable, Element table) {

        List<Element> elements = table.elements("column");
        for (Element e : elements) {
            CfColumn cfColumn = new CfColumn();
            cfColumn.setName(attr(e, "name"));
            cfColumn.setJavatype(attr(e, "javatype"));
            cfColumn.setTestVal(attr(e, "testVal"));
            cfColumn.setEncryptType(attr(e, "encryptType"));
//            cfColumn.setTypeHandler(attr(e, "typeHandler"));
            cfColumn.setRelatedColumn(attr(e, "relatedColumn"));
            cfTable.addColumn(cfColumn);
        }

    }

    private void fillEncryptKv(CfTable cfTable, Element table, Config config) {

        List<Element> elements = table.elements("encrypt");
        if (CollectionUtils.isNotEmpty(elements)) {
            for (Element e : elements) {
                CfEncrypt cfEncrypt = new CfEncrypt();
                String type = attr(e, "type");
                if (StringUtils.isNotBlank(type)) {
                    cfEncrypt.setType(type);
                }
                cfEncrypt.setRemark(attr(e, "remark"));

                List<Element> ers = e.elements();
                if (CollectionUtils.isNotEmpty(ers)) {
                    for (Element er : ers) {
                        if (StringUtils.equals(er.getName(), "encryptcolumn")) {
                            CfEncryptColumn cfEncryptColumn = new CfEncryptColumn();
                            cfEncryptColumn.setName(attr(er, "name"));
                            cfEncryptColumn.setRemark(attr(er, "remark"));
                            cfEncrypt.addCfEncryptColumn(cfEncryptColumn);
                        }
                    }
                }
                if (CollectionUtils.isNotEmpty(cfEncrypt.getCfEncryptColumn())) {
                    for (CfEncryptColumn encryptColumn : cfEncrypt.getCfEncryptColumn()) {
                        //需要加密的字段. key-加密字段 value-加解密执行器
                        cfTable.addEncryptKv(CamelCaseUtils.toCamelCase(encryptColumn.getName()), cfEncrypt.getType());
                    }
                }
            }
        }
    }

    /**
     * Attr string.
     *
     * @param e    the e
     * @param attr the attr
     * @return the string
     */
    private String attr(Element e, String attr) {
        if (e == null || attr == null) {
            return null;
        }
        Attribute attribute = e.attribute(attr);
        if (attribute == null) {
            return null;
        } else {
            return attribute.getText();
        }
    }

    /**
     * Attr string.
     *
     * @param e    the e
     * @param attr the attr
     * @return the string
     */
    private Long attrLong(Element e, String attr) {
        if (e == null || attr == null) {
            return null;
        }
        Attribute attribute = e.attribute(attr);
        if (attribute == null) {
            return null;
        } else {
            return Long.valueOf(attribute.getText());
        }
    }

    /**
     * Attr string.
     *
     * @param e    the e
     * @param attr the attr
     * @return the string
     */
    private boolean attrBoolean(Element e, String attr) {
        if (e == null || attr == null) {
            return false;
        }
        Attribute attribute = e.attribute(attr);
        if (attribute == null) {
            return false;
        } else {
            return Boolean.valueOf(attribute.getText());
        }
    }

    private DefaultElement elementById(Element element, String id) {
        List<DefaultElement> elements = element.elements();
        if (CollectionUtils.isNotEmpty(elements)) {
            for (DefaultElement element1 : elements) {
                if (StringUtils.equals(element1.attributeValue("id"), id)) {
                    return element1;
                } else if (CollectionUtils.isNotEmpty(element1.elements())) {
                    DefaultElement defaultElement = elementById(element1, id);
                    if (defaultElement != null) {
                        return defaultElement;
                    }
                }
            }

        }
        return null;
    }
}