//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//

package com.microsoft.cognitiveservices.speech.util;

import com.microsoft.cognitiveservices.speech.util.JsonBuilder;
import com.microsoft.cognitiveservices.speech.util.JsonValueJNI;

/**
 * The type Json builder jni.
 */
public final class JsonBuilderJNI {

    /**
     * Create builder json value.
     *
     * @return the JsonBuilder object
     */
    public static JsonBuilder createBuilder() {
        IntRef handle = new IntRef(0);
        int root = createJsonBuilderHandle(handle);

        SafeHandle builderHandle = new SafeHandle(handle.getValue(), SafeHandleType.JSONBUILDER);
        return new JsonBuilder(builderHandle, root);
    }

    /**
     * Adds an item to the specified ajv item
     *
     * @param builderHandle the parser handle
     * @param item the specified ajv item to add to
     * @param index the index to add the new item at
     * @param find the json key to associate with the new item
     * @return the ajv item added
     */
    public static int addItem(SafeHandle builderHandle, int item, int index, String find) {
        return builderAdd(builderHandle, item, index, find);
    }

    /**
     * Adds an item to the specified ajv item
     *
     * @param builderHandle the parser handle
     * @param item the specified ajv item to add to
     * @param json the json string to be used if kind = JSON
     * @param kind the kind of value being added
     * @param str the string value to be used if kind = STRING
     * @param boolValue the boolean value to be used if kind = BOOLEAN
     * @param integer the integer value to be used if kind = NUMBER
     * @param number the doubl value to be used if kind = NUMBER
     * @return An HR result indicating error or success
     */
    public static int setItem(
        SafeHandle builderHandle, int item, String json, JsonValueJNI.ValueKind kind, String str, boolean boolValue, int integer, double number) {
        return builderSet(builderHandle, item, json, kind.getValue(), str, boolValue, integer, number);
    }

    /**
     * As json string.
     *
     * @param builder the builder
     * @param item   the item
     * @return the string
     */
    public static String asJsonCopy(SafeHandle builder, int item){
        return builderAsJsonCopy(builder, item);
    }

    private final native static String builderAsJsonCopy(SafeHandle builder, int item);
    private final native static int createJsonBuilderHandle(IntRef builder);
    private final native static int builderAdd(SafeHandle builder, int item, int index, String find);
    private final native static int builderSet(SafeHandle builder, int item, String json, int kind, String str, boolean bool, int integer, double number);
    private static long INVALID_ARG = 0x005;
}
