//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.util;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.Callable;

import com.microsoft.cognitiveservices.speech.util.Contracts;

/*! \cond INTERNAL */

/**
 * Internal AsyncThreadService class, which wraps Java ExecutorService as static and shared between all 
 * SDK instances who needs to do async operations.
 * Shutdown needs to be called to ensure resource release after service is not needed.
 * According to Java documentation, by using newCachedThreadPool, ExecuteService will release the thread 
 * pool after 60 seconds if there is no active threads.
 */

public class AsyncThreadService {

    private static ExecutorService s_executorService;
    private static int s_activeExecutorServiceClientCounter = 0;
    private static Object s_asyncThreadServiceLock = new Object();
    /**
     * See https://docs.oracle.com/javase/8/docs/api/java/util/concurrent/Executors.html#newCachedThreadPool--
    */
    public static void initialize() {
        synchronized (s_asyncThreadServiceLock) {
            if (s_activeExecutorServiceClientCounter == 0) {
                // See https://docs.oracle.com/javase/8/docs/api/java/util/concurrent/Executors.html#newCachedThreadPool-- 
                // for newCachedThreadPool documentation
                s_executorService = Executors.newCachedThreadPool();
            }
            s_activeExecutorServiceClientCounter++;
            //System.out.println("AsyncThreadService active clients " + String.valueOf(s_activeExecutorServiceClientCounter));
        }
    }

    /**
    * shutdown the internal thread service.
    */
    public static void shutdown() {
        synchronized (s_asyncThreadServiceLock) {
            s_activeExecutorServiceClientCounter--;
            if (s_activeExecutorServiceClientCounter == 0) {
                s_executorService.shutdownNow();
            }
            //System.out.println("AsyncThreadService active clients " + String.valueOf(s_activeExecutorServiceClientCounter));
        }
    }
    
    /**
     * internal submit function.
     * @param <T> Type of future task
     * @param task The task to schedule
     * @return The future task to wait on
    */
    public static <T> Future<T> submit(Callable<T> task) {
        synchronized (s_asyncThreadServiceLock) {
            if (s_executorService.isShutdown() || s_activeExecutorServiceClientCounter == 0) {
                throw new IllegalStateException("ExecutorService has been shutdown", new IllegalStateException());
            }
            return s_executorService.submit(task);
        }
    }

}

/*! \endcond */