//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.speaker;

import java.lang.AutoCloseable;
import java.math.BigInteger;
import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.PropertyId;
import com.microsoft.cognitiveservices.speech.RecognitionResult;
import com.microsoft.cognitiveservices.speech.ResultReason;
import com.microsoft.cognitiveservices.speech.PropertyCollection;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.StringRef;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;

/**
 * Defines result of voice profile enrollment.
 */
public final class VoiceProfileEnrollmentResult implements AutoCloseable
{

    VoiceProfileEnrollmentResult(long result) {
        this.resultHandle = new SafeHandle(result, SafeHandleType.VoiceProfileEnrollmentResult);

        StringRef stringVal = new StringRef("");
        Contracts.throwIfFail(getResultId(resultHandle, stringVal));
        this.resultId = stringVal.getValue();
                
        IntRef intVal = new IntRef(0);
        Contracts.throwIfFail(getResultReason(resultHandle, intVal));
        this.reason = ResultReason.values()[(int)intVal.getValue()];

        IntRef propertyHandle = new IntRef(0);
        Contracts.throwIfFail(getPropertyBagFromResult(resultHandle, propertyHandle));
        this.properties = new PropertyCollection(propertyHandle);

        this.profileId = properties.getProperty("enrollment.profileId");
        String value = properties.getProperty("enrollment.enrollmentsCount");
        this.enrollmentsCount = value.isEmpty() ? 0 : Integer.parseInt(value);
        value = properties.getProperty("enrollment.remainingEnrollmentsCount");
        this.remainingEnrollmentsCount = value.isEmpty() ? 0 : Integer.parseInt(value);
        value = properties.getProperty("enrollment.enrollmentsLengthInSec");
        this.enrollmentsLength = value.isEmpty() ? BigInteger.ZERO : new BigInteger(value);
        value = properties.getProperty("enrollment.remainingEnrollmentsSpeechLengthInSec");
        this.remainingEnrollmentsSpeechLength = value.isEmpty() ? BigInteger.ZERO : new BigInteger(value);
        value = properties.getProperty("enrollment.audioLengthInSec");
        this.audioLength = value.isEmpty() ? BigInteger.ZERO : new BigInteger(value);
        value = properties.getProperty("enrollment.audioSpeechLengthInSec");
        this.audioSpeechLength = value.isEmpty() ? BigInteger.ZERO : new BigInteger(value);
        value = properties.getProperty("enrollment.enrollmentsSpeechLengthInSec");
        this.enrollmentsSpeechLength = value.isEmpty() ? BigInteger.ZERO : new BigInteger(value);
        this.createdTime = properties.getProperty("enrollment.createdDateTime");
        this.lastUpdatedDateTime = properties.getProperty("enrollment.lastUpdatedDateTime");
    }

    /**
     * Presents the result identifier.
     * @return The the result identifier.
     */
    public String getResultId() {  
        return this.resultId;
    }

    /**
     * Specifies status of enrollment result.
     * @return The reason of the result.
     */
    public ResultReason getReason() {
        return this.reason;
    }

    /**
     * Presents the profile id in the enrollment result.
     * @return The profile identifier.
     */
    public String getProfileId() {  
        return this.profileId;
    }

    /**
     * The number of enrollment audios accepted for this profile.
     * @return The number of enrollment audios accepted for this profile.
     */
    public int getEnrollmentsCount() {  
        return this.enrollmentsCount;
    }

    /**
     * The total length of enrollment audios accepted for this profile in hundred nanoseconds.
     * @return The total length of enrollment audios accepted for this profile in hundred nanoseconds.
     */
    public BigInteger getEnrollmentsLength() {  
        return this.enrollmentsLength;
    }

    /**
     * The number of enrollments needed to complete profile enrollment.
     * @return The number of enrollments needed to complete profile enrollment.
     */
    public int getRemainingEnrollmentsCount() {  
        return this.remainingEnrollmentsCount;
    }

    /**
     * The amount of pure speech (which is the amount of audio after removing silence and non-speech segments) needed to complete profile enrollment in hundred nanoseconds.
     * @return The remaining length of speech required to complete enrollment.
     */
    public BigInteger getRemainingEnrollmentsSpeechLength() {  
        return this.remainingEnrollmentsSpeechLength;
    }

    /**
     * This enrollment audio length in hundred nanoseconds.
     * @return The audio length of this enrollment in hundred nanoseconds.
     */
    public BigInteger getAudioLength() {  
        return this.audioLength;
    }

    /**
     * This enrollment audio pure speech(which is the amount of audio after removing silence and non - speech segments) length in hundred nanoseconds.
     * @return The audio length of this enrollment in hundred nanoseconds.
     */
    public BigInteger getAudioSpeechLength() {  
        return this.audioSpeechLength;
    }

    /**
     * The summation of pure speech(which is the amount of audio after removing silence and non - speech segments) across all profile enrollments in hundred nanoseconds.
     * @return The sum of speech across all enrollments.
     */
    public BigInteger getEnrollmentsSpeechLength() {  
        return this.enrollmentsSpeechLength;
    }

    /**
     * The set of properties exposed in the result.
     * @return The set of properties exposed in the result.
     */
    public PropertyCollection getProperties() {
        return this.properties;
    }

    /**
     * A textual representation of the created time of the voice profile.
     * @return The created time of the voice profile.
     */
    public String getCreatedTime() {  
        return this.createdTime;
    }

    /**
     * A textual representation of the last updated time of the voice profile.
     * @return The last updated time of the voice profile.
     */
    public String getLastUpdatedDateTime() {  
        return this.lastUpdatedDateTime;
    }

    /**
     * Explicitly frees any external resource attached to the object
     */
    public void close() {
        if (properties != null) {
            properties.close();
            properties = null;
        }
        if (resultHandle != null) {
            resultHandle.close();
            resultHandle = null;
        }        
    }

    /**
     * Returns a String that represents the voice enrollment result.
     * @return A String that represents the voice enrollment result.
     */
    @Override
    public String toString() {
        return "ResultId:" + getResultId() +
                " Reason:" + getReason() +
                " Json:" + properties.getProperty(PropertyId.SpeechServiceResponse_JsonResult);
    }

    /*! \cond INTERNAL */
    /**
     * Returns the voice enrollment result implementation.
     * @return The implementation of the result.
     */
    public SafeHandle getImpl() {
        Contracts.throwIfNull(this.resultHandle, "result");
        return this.resultHandle;
    }
    /*! \endcond */

    private SafeHandle resultHandle = null;
    private PropertyCollection properties = null;
    private String resultId = "";
    private ResultReason reason;
    private String profileId = "";
    private int enrollmentsCount = 0;
    private BigInteger enrollmentsLength;
    private int remainingEnrollmentsCount = 0;
    private BigInteger remainingEnrollmentsSpeechLength;
    private BigInteger audioLength;
    private BigInteger audioSpeechLength;
    private BigInteger enrollmentsSpeechLength;
    private String createdTime = "";
    private String lastUpdatedDateTime = "";

    private final native long getResultId(SafeHandle resultHandle, StringRef resultId);
    private final native long getResultReason(SafeHandle resultHandle, IntRef reasonVal);
    private final native long getPropertyBagFromResult(SafeHandle resultHandle, IntRef propertyHandle);

}
