//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.speaker;

import java.util.concurrent.Future;
import java.lang.AutoCloseable;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.List;

import com.microsoft.cognitiveservices.speech.speaker.VoiceProfile;
import com.microsoft.cognitiveservices.speech.speaker.VoiceProfileEnrollmentResult;
import com.microsoft.cognitiveservices.speech.speaker.VoiceProfilePhraseResult;
import com.microsoft.cognitiveservices.speech.speaker.VoiceProfileResult;
import com.microsoft.cognitiveservices.speech.SpeechConfig;
import com.microsoft.cognitiveservices.speech.audio.AudioConfig;
import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.StringRef;
import com.microsoft.cognitiveservices.speech.util.AsyncThreadService;
import com.microsoft.cognitiveservices.speech.PropertyCollection;

/**
 * A voice profile client performs voice profile management for speaker recognition.
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class VoiceProfileClient implements AutoCloseable {

    /**
     * Creates a new instance of a voice profile client.
     * @param speechConfig speech configuration.
     */
    public VoiceProfileClient(SpeechConfig speechConfig) {

        Contracts.throwIfNull(speechConfig, "speechConfig");
        IntRef handle = new IntRef(0);
        Contracts.throwIfFail(createFromConfig(handle, speechConfig.getImpl()));
        voiceProfileClientHandle = new SafeHandle(handle.getValue(), SafeHandleType.VoiceProfileClient);
        initialize();
    }

    /**
     * The collection of properties and their values defined for VoiceProfileClient.
     * @return The collection of properties and their values defined for VoiceProfileClient.
     */
    public PropertyCollection getProperties() {
        return propertyHandle;
    }

    /**
     * Creates a voice profile as an asynchronous operation.
     * @param voiceProfileType A voice profile type.
     * @param locale a locale, e.g "en-us".
     * @return An asynchronous operation representing the result of creating a voice profile.
     */
    public Future<VoiceProfile> createProfileAsync(final VoiceProfileType voiceProfileType, final String locale) {

        return AsyncThreadService.submit(new java.util.concurrent.Callable<VoiceProfile>() {
            public VoiceProfile call() {
                IntRef result = new IntRef(0);
                Contracts.throwIfFail(createVoiceProfile(voiceProfileClientHandle, voiceProfileType.getValue(), locale, result));
                return new VoiceProfile(result.getValue());
            }
        });
    }

    /**
     * Enroll a voice profile asynchronously.
     * @param voiceProfile A voice profile.
     * @param audioConfig An audio config.
     * @return An asynchronous operation representing the result of enrollment of a voice profile.
     */
    public Future<VoiceProfileEnrollmentResult> enrollProfileAsync(final VoiceProfile voiceProfile, final AudioConfig audioConfig) {

        return AsyncThreadService.submit(new java.util.concurrent.Callable<VoiceProfileEnrollmentResult>() {
            public VoiceProfileEnrollmentResult call() {
                IntRef result = new IntRef(0);
                Contracts.throwIfFail(enrollVoiceProfile(voiceProfileClientHandle, voiceProfile.getImpl(), audioConfig.getImpl(), result));
                return new VoiceProfileEnrollmentResult(result.getValue());
            }
        });
    }

    /**
     * Get valid activation phrases for enrollment.
     * @param voiceProfileType A voice profile type.
     * @param locale a locale, e.g "en-us".
     * @return An asynchronous operation representing the result of getting activation phrases.
     */
    public Future<VoiceProfilePhraseResult> getActivationPhrasesAsync(final VoiceProfileType voiceProfileType, final String locale) {

        return AsyncThreadService.submit(new java.util.concurrent.Callable<VoiceProfilePhraseResult>() {
            public VoiceProfilePhraseResult call() {
                IntRef result = new IntRef(0);
                Contracts.throwIfFail(getActivationPhrases(voiceProfileClientHandle, voiceProfileType.getValue(), locale, result));
                return new VoiceProfilePhraseResult(result.getValue());
            }
        });
    }

    /**
     * Delete a voice profile asynchronously.
     * @param voiceProfile A voice profile.
     * @return An asynchronous operation representing the result of deleting a voice profile.
     */
    public Future<VoiceProfileResult> deleteProfileAsync(final VoiceProfile voiceProfile) {

        return AsyncThreadService.submit(new java.util.concurrent.Callable<VoiceProfileResult>() {
            public VoiceProfileResult call() {
                IntRef result = new IntRef(0);
                Contracts.throwIfFail(deleteVoiceProfile(voiceProfileClientHandle, voiceProfile.getImpl(), result));
                return new VoiceProfileResult(result.getValue());
            }
        });
    }

    /**
     * Reset a voice profile asynchronously.
     * @param voiceProfile A voice profile.
     * @return An asynchronous operation representing the result of reset a voice profile.
     */
    public Future<VoiceProfileResult> resetProfileAsync(final VoiceProfile voiceProfile) {

        return AsyncThreadService.submit(new java.util.concurrent.Callable<VoiceProfileResult>() {
            public VoiceProfileResult call() {
                IntRef result = new IntRef(0);
                Contracts.throwIfFail(resetVoiceProfile(voiceProfileClientHandle, voiceProfile.getImpl(), result));
                return new VoiceProfileResult(result.getValue());
            }
        });
    }

    /**
     * Retrieve an enrollment result given the Voice Profile.
     * @param voiceProfile A voice profile.
     * @return An asynchronous operation representing the result of enrollment of a voice profile.
     */
    public Future<VoiceProfileEnrollmentResult> retrieveEnrollmentResultAsync(final VoiceProfile voiceProfile) {

        return AsyncThreadService.submit(new java.util.concurrent.Callable<VoiceProfileEnrollmentResult>() {
            public VoiceProfileEnrollmentResult call() {
                IntRef result = new IntRef(0);
                Contracts.throwIfFail(retrieveEnrollmentResult(voiceProfileClientHandle, voiceProfile.getId(), voiceProfile.getType().getValue(), result));
                return new VoiceProfileEnrollmentResult(result.getValue());
            }
        });
    }    

    /**
     * Get all voice profiles as an asynchronous operation.
     * @param voiceProfileType A voice profile type.
     * @return An asynchronous operation representing the result of getting all voice profile.
     */
    public Future<List<VoiceProfile>> getAllProfilesAsync(final VoiceProfileType voiceProfileType) {

        return AsyncThreadService.submit(new java.util.concurrent.Callable<List<VoiceProfile>>() {
            public List<VoiceProfile> call() {
                List<VoiceProfile> profiles = new ArrayList<VoiceProfile>();
                
                StringRef profileListRef = new StringRef("");
                IntRef sizeRef = new IntRef(0);
                Contracts.throwIfFail(getProfilesJson(voiceProfileClientHandle, voiceProfileType.getValue(), profileListRef, sizeRef));
                
                if (!profileListRef.getValue().isEmpty()) {
                    List<String> profileList = Arrays.asList(profileListRef.getValue().split("\\|"));
                    for (String prof: profileList) {
                        profiles.add(new VoiceProfile(prof, voiceProfileType));
                    }
                }

                return profiles;
            }
        });
    }

    /**
     * Dispose of associated resources.
     * Note: close() must be called in order to release underlying resources held by the object.
     */
    @Override
    public void close() {
        if (disposed) {
            return;
        }

        if (propertyHandle != null) {
            propertyHandle.close();
            propertyHandle = null;
        }
        if (voiceProfileClientHandle != null) {
            voiceProfileClientHandle.close();
            voiceProfileClientHandle = null;
        }

        AsyncThreadService.shutdown();
        disposed = true;
    }

    /*! \cond INTERNAL */
    /**
     * INTERNAL
     * @return SafeHandle to {@link VoiceProfileClient}
     */
    public SafeHandle getImpl() {
        return voiceProfileClientHandle;
    }

    /*! \endcond */

    private void initialize() {
        final VoiceProfileClient _this = this;
        AsyncThreadService.initialize();
        IntRef propHandle = new IntRef(0);
        Contracts.throwIfFail(getPropertyBagFromHandle(_this.voiceProfileClientHandle, propHandle));
        propertyHandle = new PropertyCollection(propHandle);
    }

    private SafeHandle voiceProfileClientHandle = null;
    private PropertyCollection propertyHandle = null;
    private boolean disposed = false;

    private final native long createFromConfig(IntRef handle, SafeHandle speechConfigHandle);
    private final native long createVoiceProfile(SafeHandle handle, int type, String locale, IntRef resultHandle);
    private final native long getProfilesJson(SafeHandle handle, int type, StringRef profilesJson, IntRef size);
    private final native long enrollVoiceProfile(SafeHandle handle, SafeHandle profile, SafeHandle audioConfig, IntRef resultHandle);
    private final native long deleteVoiceProfile(SafeHandle handle, SafeHandle profile, IntRef resultHandle);
    private final native long resetVoiceProfile(SafeHandle handle, SafeHandle profile, IntRef resultHandle);
    private final native long retrieveEnrollmentResult(SafeHandle handle, String id, int type, IntRef resultHandle);
    private final native long getActivationPhrases(SafeHandle handle, int type, String locale, IntRef resultHandle);
    private final native long getPropertyBagFromHandle(SafeHandle handle, IntRef propertyHandle);
}
