//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.intent;

import java.util.Collection;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.Properties;


import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.PropertyId;
import com.microsoft.cognitiveservices.speech.ResultReason;
import com.microsoft.cognitiveservices.speech.SpeechRecognitionResult;
import com.microsoft.cognitiveservices.speech.util.JsonValue;
import com.microsoft.cognitiveservices.speech.util.StringRef;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;

/**
 * Defines result of intent recognition.
 */
public final class IntentRecognitionResult extends com.microsoft.cognitiveservices.speech.SpeechRecognitionResult {

    IntentRecognitionResult(long result) {
        super(result);
        Contracts.throwIfNull(resultHandle, "resultHandle");
        StringRef intentIdStr = new StringRef("");
        Contracts.throwIfFail(getIntentId(resultHandle, intentIdStr));
        this.intentId = intentIdStr.getValue();
        Contracts.throwIfNull(this.intentId, "IntentId");
        if( this.getReason() == ResultReason.RecognizedIntent) {
            String jsonSLE = properties.getProperty("LanguageUnderstandingSLE_JsonResult");
            entities = new Hashtable();
            if (jsonSLE != null) {
                JsonValue parsed = JsonValue.Parse(jsonSLE);
                int count = parsed.count();
                for (int i = 0; i < count; i++) {
                    String name = parsed.getName(i);
                    String value = parsed.get(i).asString();
                    entities.put(name, value);
                }
            }
        }
    }

    /**
     * Explicitly frees any external resource attached to the object
     */
    @Override
    public void close() {
        super.close();
    }

    private Dictionary<String, String> entities;

    /**
     * Gets the entities found in the utterance.
     * This does not currently support LUIS entities.
     * @return A {@code Dictionary} containing the entities extracted. The key 
     * will be the entity name.
    */
    public Dictionary<String, String> getEntities() {
        return entities;
    }

    /**
     * A String that represents the intent identifier being recognized.
     * @return A String that represents the intent identifier being recognized.
     */
    public String getIntentId() {
        return intentId;
    }

    private String intentId;

    /**
     * Returns a String that represents the intent recognition result.
     * @return A String that represents the intent recognition result.
     */
    @Override
    public String toString() {
        return "ResultId:" + getResultId() +
                " Reason:" + getReason() +
                " IntentId:<" + intentId +
                "> Recognized text:<" + getText() +
                "> Recognized json:<" + getProperties().getProperty(PropertyId.SpeechServiceResponse_JsonResult) +
                "> LanguageUnderstandingJson <" + getProperties().getProperty(PropertyId.LanguageUnderstandingServiceResponse_JsonResult) +
                ">.";
    }

    private final native long getIntentId(SafeHandle resultHandle, StringRef intentId);
}
