//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech;

import java.lang.AutoCloseable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.StringRef;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.PropertyCollection;

/**
 * Contains detailed information about the synthesis voice information.
 * Note: close() must be called in order to release underlying resources held by the object.
 * Updated in version 1.17.0
 */
public class VoiceInfo implements AutoCloseable {

    /*! \cond PROTECTED */
    /**
     * PROTECTED
     * @param result PROTECTED {@link VoiceInfo}
     */
    protected VoiceInfo(IntRef result) {
        Contracts.throwIfNull(result, "result");

        this.voiceInfoHandle = new SafeHandle(result.getValue(), SafeHandleType.VoiceInfo);

        this.name = getName(voiceInfoHandle);
        this.locale = getLocale(voiceInfoHandle);
        this.shortName = getShortName(voiceInfoHandle);
        this.localName = getLocalName(voiceInfoHandle);

        IntRef intRef = new IntRef(0);
        Contracts.throwIfFail(getVoiceType(voiceInfoHandle, intRef));
        this.voiceType = SynthesisVoiceType.values()[(int)intRef.getValue() - 1]; // native SynthesisVoiceType starts from 1.

        String styleListString = getStyleListString(voiceInfoHandle);
        if (styleListString.isEmpty()) {
            this.styleList = new ArrayList<String>();
        } else {
            this.styleList = Arrays.asList(styleListString.split("\\|"));
        }

        this.voicePath = getVoicePath(voiceInfoHandle);

        IntRef propertyRef = new IntRef(0);
        Contracts.throwIfFail(getPropertyBagFromResult(voiceInfoHandle, propertyRef));
        this.properties = new PropertyCollection(propertyRef);

        String genderStr = properties.getProperty("Gender");
        this.gender = genderStr.equals("Female") ? SynthesisVoiceGender.Female : genderStr.equals("Male") ? SynthesisVoiceGender.Male : SynthesisVoiceGender.Unknown;
    }

    /*! \endcond */

    /**
     * Gets the voice name.
     * @return The voice name.
     */
    public String getName() {
        return this.name;
    }

    /**
     * Gets the locale of the voice.
     * @return The locale.
     */
    public String getLocale() {
        return this.locale;
    }

    /**
     * Gets the short name of the voice.
     * @return The short name.
     */
    public String getShortName() {
        return this.shortName;
    }

    /**
     * Gets the local name of the voice.
     * @return The local name.
     */
    public String getLocalName() {
        return this.localName;
    }

    /**
     * Gets the voice gender.
     * Added in version 1.17.0
     * @return The voice gender.
     */
    public SynthesisVoiceGender getGender() {
        return this.gender;
    }

    /**
     * Gets the voice type.
     * @return The voice type.
     */
    public SynthesisVoiceType getVoiceType() {
        return this.voiceType;
    }

    /**
     * Gets the style list.
     * @return The style list.
     */
    public List<String> getStyleList() {
        return this.styleList;
    }

    /**
     * Gets the voice path, only valid for offline voices.
     * @return The voice path.
     */
    public String getVoicePath() {
        return this.voicePath;
    }

    /**
     *  The set of properties exposed in the result.
     * @return The set of properties exposed in the result.
     */
    public PropertyCollection getProperties() {
        return this.properties;
    }

    /**
     * Explicitly frees any external resource attached to the object
     */
    public void close() {
        if (this.voiceInfoHandle != null) {
            this.voiceInfoHandle.close();
            this.voiceInfoHandle = null;
        }

        if (this.properties != null) {
            this.properties.close();
            this.properties = null;
        }
    }

    /*! \cond INTERNAL */

    /**
     * Returns the vice info implementation.
     * @return The implementation handle.
     */
    public SafeHandle getImpl() {
        return this.voiceInfoHandle;
    }

    /*! \endcond */

    private final native String getName(SafeHandle pronConfigRef);
    private final native String getLocale(SafeHandle pronConfigRef);
    private final native String getShortName(SafeHandle pronConfigRef);
    private final native String getLocalName(SafeHandle pronConfigRef);
    private final native String getStyleListString(SafeHandle pronConfigRef);
    private final native String getVoicePath(SafeHandle pronConfigRef);
    private final native long getVoiceType(SafeHandle voiceInfoHandle, IntRef voiceTypeRef);
    private final native long getPropertyBagFromResult(SafeHandle voiceInfoHandle, IntRef propertyRef);

    private String name;
    private String locale;
    private String shortName;
    private String localName;
    private SynthesisVoiceGender gender;
    private SynthesisVoiceType voiceType;
    private List<String> styleList;
    private String voicePath;
    private PropertyCollection properties;
    private SafeHandle voiceInfoHandle = null;
}
