//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech;

import java.lang.AutoCloseable;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.StringRef;

/**
 * Defines contents of speech synthesis viseme event.
 * Added in version 1.16.0
 */
public class SpeechSynthesisVisemeEventArgs {

    /*! \cond INTERNAL */

    SpeechSynthesisVisemeEventArgs(long eventArgs) {
        Contracts.throwIfNull(eventArgs, "eventArgs");

        SafeHandle eventHandle = new SafeHandle(eventArgs, SafeHandleType.SynthesisEvent);
        IntRef audioOffsetRef = new IntRef(0);
        IntRef visemeIdRef = new IntRef(0);
        Contracts.throwIfFail(getVisemeEventValues(eventHandle, audioOffsetRef, visemeIdRef));
        this.audioOffset = audioOffsetRef.getValue();
        this.visemeId = visemeIdRef.getValue();
        StringRef stringVal = new StringRef("");
        Contracts.throwIfFail(getAnimationFromHandle(eventHandle, stringVal));
        this.animation = stringVal.getValue();
        eventHandle.close();
    }

    /*! \endcond */

    /**
     * Specifies current viseme's time offset in output audio, by ticks (100ns).
     * @return Current viseme's time offset in output audio, by ticks (100ns).
     */
    public long getAudioOffset() {
        return this.audioOffset;
    }

    /**
     * Specifies current viseme ID.
     * @return The current viseme ID.
     */
    public long getVisemeId() {
        return this.visemeId;
    }

    /**
     * Specifies the animation, could be in svg or other format.
     * @return The animation string.
     */
    public String getAnimation() {
        return this.animation;
    }

    private final native long getVisemeEventValues(SafeHandle eventHandle, IntRef audioOffset, IntRef visemeId);
    private final native long getAnimationFromHandle(SafeHandle eventHandle, StringRef animationText);

    private long audioOffset;
    private long visemeId;
    private String animation;

}
