//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech;

import java.lang.AutoCloseable;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.StringRef;

/**
 * Defines contents of speech synthesis bookmark event.
 * Added in version 1.16.0
 */
public class SpeechSynthesisBookmarkEventArgs {

    /*! \cond INTERNAL */

    SpeechSynthesisBookmarkEventArgs(long eventArgs) {
        Contracts.throwIfNull(eventArgs, "eventArgs");

        SafeHandle eventHandle = new SafeHandle(eventArgs, SafeHandleType.SynthesisEvent);
        IntRef audioOffsetRef = new IntRef(0);
        Contracts.throwIfFail(getBookmarkEventValues(eventHandle, audioOffsetRef));
        this.audioOffset = audioOffsetRef.getValue();
        StringRef stringVal = new StringRef("");
        Contracts.throwIfFail(getTextFromHandle(eventHandle, stringVal));
        this.text = stringVal.getValue();
        eventHandle.close();
    }

    /*! \endcond */

    /**
     * Specifies current bookmark's time offset in output audio, by ticks (100ns).
     * @return Current bookmark's time offset in output audio, by ticks (100ns).
     */
    public long getAudioOffset() {
        return this.audioOffset;
    }

    /**
     * Specifies the bookmark text.
     * @return The bookmark text.
     */
    public String getText() {
        return this.text;
    }

    private final native long getBookmarkEventValues(SafeHandle eventHandle, IntRef audioOffset);
    private final native long getTextFromHandle(SafeHandle eventHandle, StringRef text);

    private long audioOffset;
    private String text;
}
