//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech;

import java.lang.AutoCloseable;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.IntRef;

/**
* Defines payload for Connection's MessageReceived events.
* Added in version 1.15.0.
*/
public final class ConnectionMessageEventArgs {

    /*! \cond PROTECTED */
    /**
     * PROTECTED
     * @param eventArgs PROTECTED
     */
    protected ConnectionMessageEventArgs(long eventArgs) {
        Contracts.throwIfNull(eventArgs, "eventArgs is null");

        SafeHandle eventHandle = null;
        eventHandle = new SafeHandle(eventArgs, SafeHandleType.ConnectionMessageEvent);
        Contracts.throwIfNull(eventHandle, "eventHandle");

        IntRef messageHandle = new IntRef(0);
        Contracts.throwIfFail(getConnectionMessage(eventHandle, messageHandle));
        message = new ConnectionMessage(messageHandle.getValue());

        eventHandle.close();
        eventHandle = null;
    }
    /*! \endcond */

    /**
     * Gets the ConnectionMessage associated with this ConnectionMessageEventArgs.
     * @return A ConnectionMessage containing the message.
     */
    public ConnectionMessage getMessage() {
        return this.message;
    }

    /**
     * Returns a string that represents the connection message.
     * @return A string that represents the connection message.
     */
    public String toString() {
        return "Message: " + this.getMessage().toString();
    }

    private final native long getConnectionMessage(SafeHandle eventHandle, IntRef messageHandle);
    private ConnectionMessage message;
}
