//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech;

import java.lang.AutoCloseable;

import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.Contracts;

/**
 * ConnectionMessage represents implementation specific messages sent to and received 
 * from the speech service. These messages are provided for debugging purposes and should
 * not be used for production use cases with the Azure Cognitive Services Speech Service.
 * Messages sent to and received from the Speech Service are subject to change without notice.
 * This includes message contents, headers, payloads, ordering, etc.
 * Note: close() must be called in order to release underlying resources held by the object.
 * Added in version 1.15.0.
 */
public final class ConnectionMessage implements AutoCloseable {

    /*! \cond PROTECTED */
    /**
     * PROTECTED
     * @param message PROTECTED
     */
    protected ConnectionMessage(long message) {
        Contracts.throwIfNull(message, "message is null");

        this.messageHandle = new SafeHandle(message, SafeHandleType.ConnectionMessage);

        IntRef propertyHandle = new IntRef(0);
        Contracts.throwIfFail(getPropertyBag(messageHandle, propertyHandle));
        this.properties = new PropertyCollection(propertyHandle);
    }
    /*! \endcond */

    /**
     * The message Path
     * @return A string that specifies the message type.
     */
    public String getPath() {
        Contracts.throwIfNull(messageHandle, "messageHandle is null");
        return this.properties.getProperty("connection.message.path");
    }

    /**
     * Gets the text message payload. Typically the text message content-type is application/json.
     * To determine other content-types use Properties.GetProperty("Content-Type").
     * @return A string containing the text message.
     */
    public String getTextMessage() {
        Contracts.throwIfNull(messageHandle, "messageHandle is null");
        return this.properties.getProperty("connection.message.text.message");
    }

    /**
     * Gets the binary message payload.
     * @return An array of bytes containing the binary message.
     */
    public byte[] getBinaryMessage() {
        Contracts.throwIfNull(messageHandle, "messageHandle is null");
        if (binaryMessage == null) {
            IntRef hr = new IntRef(0);
            binaryMessage = getMessageData(messageHandle, hr);
            Contracts.throwIfFail(hr.getValue());
        }
        return binaryMessage;
    }

    /**
     * Checks to see if the ConnectionMessage is a binary message. See also GetBinaryMessage().
     * @return A boolean indicated if the message payload is binary.
     */
    public boolean isBinaryMessage() {
        Contracts.throwIfNull(messageHandle, "messageHandle is null");
        return this.properties.getProperty("connection.message.type").equals("binary");
    }

    /**
     * Checks to see if the ConnectionMessage is a text message. See also IsBinaryMessage().
     * @return A bool indicated if the message payload is text.
     */
    public boolean isTextMessage() {
        Contracts.throwIfNull(messageHandle, "messageHandle is null");
        return this.properties.getProperty("connection.message.type").equals("text");
    }

    /**
     * Returns a string that represents the connection message.
     * @return A string that represents the connection message.
     */
    public String toString() {
        String output = "";
        Contracts.throwIfNull(messageHandle, "messageHandle is null");
        if (this.isTextMessage()) {
            output = "Path: " + this.getPath() + ", Type: text, Message: " + this.getTextMessage();
        } else if (this.isBinaryMessage()) {
            output = "Path: " + this.getPath() + ", Type: binary, Size: " + (this.getBinaryMessage() == null ? 0 : this.getBinaryMessage().length) + " bytes";
        }
        return output;
    }

    /**
     * A collection of properties and their values defined for this ConnectionMessage. Message headers can be accessed via this collection (e.g. "Content-Type").
     * @return PropertyCollection
     */
    public PropertyCollection getProperties() {
        Contracts.throwIfNull(messageHandle, "messageHandle is null");
        return this.properties;
    }

    /**
     * Explicitly frees any external resource attached to the object.
     * Note: close() must be called in order to release underlying resources held by the object.
     */
    public void close() {
        if (this.messageHandle != null) {
            messageHandle.close();
            this.messageHandle = null;
        }

        if (this.properties != null) {
            this.properties.close();
            this.properties = null;
        }

        this.binaryMessage = null;
    }

    private SafeHandle messageHandle = null;
    private PropertyCollection properties = null;
    private byte[] binaryMessage = null;
    private final native long getPropertyBag(SafeHandle messageHandle, IntRef propertyHandle);
    private final native byte[] getMessageData(SafeHandle messageHandle, IntRef hr);
}




