/**
 * fshows.com
 * Copyright (C) 2013-2024 All Rights Reserved.
 */
package com.fshows.yeepay.sdk.util;

import com.yeepay.g3.sdk.yop.client.YopRequest;
import com.yeepay.g3.sdk.yop.encrypt.CertTypeEnum;
import com.yeepay.g3.sdk.yop.encrypt.DigestAlgEnum;
import com.yeepay.g3.sdk.yop.encrypt.DigitalSignatureDTO;
import com.yeepay.g3.sdk.yop.exception.YopClientException;
import com.yeepay.g3.sdk.yop.http.HttpMethodName;
import com.yeepay.g3.sdk.yop.http.HttpUtils;
import com.yeepay.g3.sdk.yop.utils.CheckUtils;
import com.yeepay.g3.sdk.yop.utils.DateUtils;
import com.yeepay.g3.sdk.yop.utils.DigitalEnvelopeUtils;
import com.yeepay.g3.sdk.yop.utils.Exceptions;
import com.yeepay.g3.sdk.yop.utils.RSAKeyUtils;
import com.yeepay.g3.sdk.yop.utils.UUIDUtils;
import com.yeepay.shade.com.google.common.base.Joiner;
import com.yeepay.shade.com.google.common.collect.Lists;
import com.yeepay.shade.com.google.common.collect.Maps;
import com.yeepay.shade.com.google.common.collect.Sets;
import com.yeepay.shade.org.apache.commons.lang3.StringUtils;
import lombok.Data;

import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.spec.InvalidKeySpecException;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;

/**
 * @author wangyi
 * @version YopSignUtil.java, v 0.1 2024-01-15 16:34 wangyi
 */
@Data
public class YopSignUtil {

    protected static String getUUID() {
        return UUIDUtils.compressV4UUID();
    }

    public static void getSignData(String apiUri, YopRequest request, HttpMethodName method) {
        CheckUtils.checkApiUri(apiUri);
        sign(apiUri, request, method);
    }

    private static void sign(String apiUri, YopRequest request, HttpMethodName httpMethod) {
        String appKey = request.getAppSdkConfig().getAppKey();
        String timestamp = DateUtils.formatAlternateIso8601Date(new Date());
        Map<String, String> headers = request.getHeaders();
        headers.put("x-yop-session-id", getUUID());
        headers.put("x-yop-request-id", getUUID());
        Set<String> headersToSignSet = new HashSet();
        headersToSignSet.add("x-yop-request-id");
        String authString = "yop-auth-v2/" + appKey + "/" + timestamp + "/" + "1800";
        String canonicalURI = HttpUtils.getCanonicalURIPath(apiUri);
        String canonicalQueryString = HttpUtils.getCanonicalQueryString(request.getParams(), true);
        SortedMap<String, String> headersToSign = getHeadersToSign(headers, headersToSignSet);
        String canonicalHeader = getCanonicalHeaders(headersToSign);
        String signedHeaders = Joiner.on(';').join(headersToSign.keySet());
        signedHeaders = signedHeaders.trim().toLowerCase();
        String canonicalRequest = authString + "\n" + httpMethod + "\n" + canonicalURI + "\n" + canonicalQueryString + "\n" + canonicalHeader;
        PrivateKey isvPrivateKey;
        if (StringUtils.length(request.getSecretKey()) > 128) {
            try {
                isvPrivateKey = RSAKeyUtils.string2PrivateKey(request.getSecretKey());
            } catch (NoSuchAlgorithmException var16) {
                throw Exceptions.unchecked(var16);
            } catch (InvalidKeySpecException var17) {
                throw Exceptions.unchecked(var17);
            }
        } else {
            isvPrivateKey = request.getAppSdkConfig().getDefaultIsvPrivateKey();
        }

        if (isvPrivateKey == null) {
            throw new YopClientException("Can't init ISV private key!");
        } else {
            DigitalSignatureDTO digitalSignatureDTO = new DigitalSignatureDTO();
            digitalSignatureDTO.setPlainText(canonicalRequest);
            digitalSignatureDTO.setCertType(CertTypeEnum.RSA2048);
            digitalSignatureDTO.setDigestAlg(DigestAlgEnum.SHA256);
            digitalSignatureDTO = DigitalEnvelopeUtils.sign(digitalSignatureDTO, isvPrivateKey);

            headers.put("Authorization", "YOP-RSA2048-SHA256 yop-auth-v2/" + appKey + "/" + timestamp + "/" + "1800" + "/" + signedHeaders + "/" + digitalSignatureDTO.getSignature());
        }
    }

    private static String getCanonicalHeaders(SortedMap<String, String> headers) {
        if (headers.isEmpty()) {
            return "";
        } else {
            List<String> headerStrings = Lists.newArrayList();
            Iterator var3 = headers.entrySet().iterator();

            while(var3.hasNext()) {
                Map.Entry<String, String> entry = (Map.Entry)var3.next();
                String key = (String)entry.getKey();
                if (key != null) {
                    String value = (String)entry.getValue();
                    if (value == null) {
                        value = "";
                    }

                    headerStrings.add(HttpUtils.normalize(key.trim().toLowerCase()) + ':' + HttpUtils.normalize(value.trim()));
                }
            }

            Collections.sort(headerStrings);
            return Joiner.on('\n').join(headerStrings);
        }
    }

    private static SortedMap<String, String> getHeadersToSign(Map<String, String> headers, Set<String> headersToSign) {
        SortedMap<String, String> ret = Maps.newTreeMap();
        Iterator var5;
        if (headersToSign != null) {
            Set<String> tempSet = Sets.newHashSet();
            var5 = ((Set)headersToSign).iterator();

            while(var5.hasNext()) {
                String header = (String)var5.next();
                tempSet.add(header.trim().toLowerCase());
            }

            headersToSign = tempSet;
        }

        var5 = headers.entrySet().iterator();

        while(true) {
            String key;
            Map.Entry entry;
            do {
                do {
                    do {
                        if (!var5.hasNext()) {
                            return ret;
                        }

                        entry = (Map.Entry)var5.next();
                        key = (String)entry.getKey();
                    } while(entry.getValue() == null);
                } while(((String)entry.getValue()).isEmpty());
            } while((headersToSign == null || !((Set)headersToSign).contains(key.toLowerCase()) || "Authorization".equalsIgnoreCase(key)) && (headersToSign != null || !isDefaultHeaderToSign(key)));

            ret.put(key, (String)entry.getValue());
        }
    }

    private static boolean isDefaultHeaderToSign(String header) {
        header = header.trim().toLowerCase();
        return header.startsWith("x-yop-") || Sets.newHashSet().contains(header);
    }
}