/**
 * fshows.com
 * Copyright (C) 2013-2023 All Rights Reserved.
 */
package
        com.fshows.umpay.bankchannel.util;

import com.fshows.umpay.sdk.util.SignUtil;
import com.fshows.umpay.sdk.util.StringPool;
import com.google.gson.FieldNamingPolicy;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.umpay.mer.ConfigContext;
import com.umpay.util.StringUtils;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicHeader;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

/**
 * @author liluqing
 * @version UmHttpClient.java, v 0.1 2023-04-06 14:21
 */
public class UmHttpClient {

    private static final Logger log = LoggerFactory.getLogger(UmHttpClient.class);
    private static final String CONTENT_TYPE_JSON = "application/json";
    private static final String CONTENT_TYPE_TEXT_JSON = "text/json";
    public static final String CONTENT_TYPE = "Content-Type";
    public static final String CHARSET = "UTF-8";
    public static final String HEADER = "header";
    public static final String DATA = "data";
    public static final String RULE = "spay";
    public static final String STATUSCODE = "statusCode";
    public static final int STATUS_200 = 200;
    public static final int STATUS_302 = 302;
    public static int CONNECTTIMEOUT = 6000;
    public static int SOCKETTIMEOUT = 120000;

    public UmHttpClient() {
    }

    public static Map<String, Object> get(ConfigContext apicontext, Map<String, String> reqMap, String fbPrivateKey) throws Exception {
        String plain = getPlain(reqMap);
        String plainEncode = getPlainEncode(reqMap);
        String url = apicontext.getUrl();
        String merId = apicontext.getMerId();
        String charset = apicontext.getCharset();
        if (charset == null || "".equals(charset)) {
            charset = "UTF-8";
        }

//        String sign = RSAUtil.createSign(plain, merId, "spay", charset);
        String sign = SignUtil.sign(plain, StringPool.DEFAULT_SIGN_TYPE, fbPrivateKey);
        log.info(String.format("签名：%s", sign));

        return get(url + "?" + plainEncode, sign, charset);
    }

    public static Map<String, Object> post(ConfigContext apicontext, String body, String fbPrivateKey) throws Exception {
        String url = apicontext.getUrl();
        String merId = apicontext.getMerId();
        String rsaCharset = apicontext.getCharset();
        if (rsaCharset == null || "".equals(rsaCharset)) {
            rsaCharset = "UTF-8";
        }

        GsonBuilder gsonBuilder = new GsonBuilder();
        gsonBuilder.setFieldNamingPolicy(FieldNamingPolicy.LOWER_CASE_WITH_UNDERSCORES);

        String sign = SignUtil.sign(body, StringPool.DEFAULT_SIGN_TYPE, fbPrivateKey);
//        String sign = RSAUtil.createSign(body, merId, "spay", rsaCharset);
        return post(url, body, sign, rsaCharset);
    }

    public static Map<String, Object> get(String url, String charset, Map<String, String> reqMap) throws Exception {
        String plain = getPlainEncode(reqMap);
        if (charset == null || "".equals(charset)) {
            charset = "UTF-8";
        }

        return get(url + "?" + plain, (String) null, (String) charset);
    }

    public static <T> Map<String, Object> post(String url, T t, String charset) throws Exception {
        Gson gson = new Gson();
        String body = gson.toJson(t);
        return post(url, body, (String) null, charset);
    }

    private static Map<String, Object> post(String url, String json, String sign, String charset) throws Exception {
        if (StringUtils.isEmpty(url)) {
            throw new RuntimeException("HttpClient,error url:" + url);
        } else {
            RequestConfig config = RequestConfig.custom().setConnectTimeout(CONNECTTIMEOUT).setSocketTimeout(SOCKETTIMEOUT).build();
            CloseableHttpClient httpClient = HttpClientBuilder.create().setDefaultRequestConfig(config).build();
            if (StringUtils.isEmpty(charset)) {
                charset = "UTF-8";
            }

            HttpPost httpPost = new HttpPost(url);
            httpPost.setHeader("Content-Type", "application/json");
            if (sign != null && !"".equals(sign.trim())) {
                httpPost.setHeader("Signature", sign);
            }

            StringEntity se = new StringEntity(json, charset);
            CloseableHttpResponse response = null;
            HashMap map = new HashMap();

            try {
                se.setContentType("text/json");
                se.setContentEncoding(new BasicHeader("Content-Type", "application/json"));
                httpPost.setEntity(se);
                response = httpClient.execute(httpPost);
                int statusCode = response.getStatusLine().getStatusCode();
                String result = null;
                if (statusCode != 200) {
                    if (statusCode != 302) {
                        httpPost.abort();
                        throw new RuntimeException("HttpClient,error status code :" + statusCode);
                    }

                    result = response.getHeaders("location")[0].getValue();
                } else {
                    HttpEntity entity = response.getEntity();
                    if (entity != null) {
                        result = EntityUtils.toString(entity, charset);
                    }

                    Header[] headers = response.getAllHeaders();
                    EntityUtils.consume(entity);
                    map.put("header", headers);
                }

                map.put("statusCode", statusCode);
                map.put("data", result);
            } catch (Exception var17) {
                var17.printStackTrace();
            } finally {
                if (response != null) {
                    response.close();
                }

            }

            return map;
        }
    }

    private static Map<String, Object> get(String url, String sign, String charset) throws Exception {
        if (StringUtils.isEmpty(url)) {
            throw new RuntimeException("HttpClient,error url:" + url);
        } else {
            RequestConfig config = RequestConfig.custom().setConnectTimeout(CONNECTTIMEOUT).setSocketTimeout(SOCKETTIMEOUT).build();
            CloseableHttpClient httpClient = HttpClientBuilder.create().setDefaultRequestConfig(config).build();
            if (StringUtils.isEmpty(charset)) {
                charset = "UTF-8";
            }

            HttpGet httpget = new HttpGet(url);
            if (sign != null && !"".equals(sign.trim())) {
                httpget.setHeader("Signature", sign);
            }

            CloseableHttpResponse response = null;
            HashMap map = new HashMap();

            try {
                response = httpClient.execute(httpget);
                String result = null;
                int statusCode = response.getStatusLine().getStatusCode();
                if (statusCode != 200) {
                    if (statusCode != 302) {
                        httpget.abort();
                        throw new RuntimeException("HttpClient,error status code :" + statusCode);
                    }

                    result = response.getHeaders("location")[0].getValue();
                } else {
                    HttpEntity entity = response.getEntity();
                    if (entity != null) {
                        result = EntityUtils.toString(entity, charset);
                    }

                    Header[] headers = response.getAllHeaders();
                    EntityUtils.consume(entity);
                    map.put("header", headers);
                }

                map.put("statusCode", statusCode);
                map.put("data", result);
            } catch (Exception var15) {
                var15.printStackTrace();
            } finally {
                if (response != null) {
                    response.close();
                }

            }

            return map;
        }
    }

    public static String getPlain(Map<String, String> reqMap) {
        Object[] obj = reqMap.keySet().toArray();
        String value = "";
        StringBuilder reqstr = new StringBuilder("");
        Arrays.sort(obj);
        Object[] var8 = obj;
        int var7 = obj.length;

        for (int var6 = 0; var6 < var7; ++var6) {
            Object object = var8[var6];
            String s = object.toString();
            if (!s.equalsIgnoreCase("Signature") && !"Signature".equals(s)) {
                value = StringUtils.trim((String) reqMap.get(s));
                if (StringUtils.isNotEmpty(value)) {
                    reqstr.append(s).append("=").append(value).append("&");
                }
            }
        }

        String plain = reqstr.substring(0, reqstr.length() - 1);
        return plain;
    }

    public static String getPlainEncode(Map<String, String> reqMap) {
        Object[] obj = reqMap.keySet().toArray();
        String value = "";
        StringBuilder reqstr = new StringBuilder("");
        Arrays.sort(obj);
        Object[] var8 = obj;
        int var7 = obj.length;

        for (int var6 = 0; var6 < var7; ++var6) {
            Object object = var8[var6];
            String s = object.toString();
            if (!s.equalsIgnoreCase("Signature") && !"Signature".equals(s)) {
                value = (String) reqMap.get(s);
                if (StringUtils.isNotEmpty(value)) {
                    try {
                        value = URLEncoder.encode(value.replaceAll("\\+", "%2B"), "UTF-8");
                    } catch (Exception var11) {
                        RuntimeException rex = new RuntimeException();
                        rex.setStackTrace(var11.getStackTrace());
                        throw rex;
                    }

                    reqstr.append(s).append("=").append(value).append("&");
                }
            }
        }

        String plain = reqstr.substring(0, reqstr.length() - 1);
        return plain;
    }

    public static InputStream downloadFile(ConfigContext apicontext, Map<String, String> reqMap, String fbPrivateKey) {
        String plain = getPlain(reqMap);
        String plainEncode = getPlainEncode(reqMap);
        String url = apicontext.getUrl() + "?" + plainEncode;
        String merId = apicontext.getMerId();
        String charset = apicontext.getCharset();
        if (charset == null || "".equals(charset)) {
            charset = "UTF-8";
        }

        String sign = SignUtil.sign(plain, StringPool.DEFAULT_SIGN_TYPE, fbPrivateKey);
        log.info(String.format("签名：%s", sign));

        if (StringUtils.isEmpty(url)) {
            throw new RuntimeException("HttpClient,error url:" + url);
        } else {
            try {
                URL finalUrl = new URL(url);
                HttpURLConnection httpURLConnection = (HttpURLConnection) finalUrl.openConnection();
                httpURLConnection.setRequestMethod("GET");
                httpURLConnection.setConnectTimeout(20 * 1000);
                httpURLConnection.setReadTimeout(10 * 1000);
                httpURLConnection.setRequestProperty("Signature",sign);
                httpURLConnection.connect();
                return httpURLConnection.getInputStream();
            } catch (Exception e) {

            }
        }
        return null;

    }
}