/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.umpay.bankchannel.client;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.PropertyNamingStrategy;
import com.alibaba.fastjson.serializer.SerializeConfig;
import com.fshows.umpay.annotation.EncryptField;
import com.fshows.umpay.bankchannel.request.UmBankBizItemRequest;
import com.fshows.umpay.bankchannel.request.UmBankBizRequest;
import com.fshows.umpay.bankchannel.response.UmBankBaseResponse;
import com.fshows.umpay.bankchannel.response.UmBankLinkResponse;
import com.fshows.umpay.bankchannel.response.UmBankMetaResponse;
import com.fshows.umpay.bankchannel.util.UmFileClinet;
import com.fshows.umpay.bankchannel.util.UmHttpClient;
import com.fshows.umpay.sdk.client.impl.UmpayApiDefinition;
import com.fshows.umpay.sdk.exception.UmPayException;
import com.fshows.umpay.sdk.request.UmBizRequest;
import com.fshows.umpay.sdk.util.SignUtil;
import com.fshows.umpay.sdk.util.ValidateUtil;
import com.umpay.mer.ConfigContext;
import com.umpay.util.Base64;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;

import javax.crypto.Cipher;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author youmingming
 * @version UmBaseClientImpl.java, v 0.1 2022-02-09 下午11:06 youmingming
 */
@Slf4j
public class UmBankClientImpl implements UmBankClient {
    /**
     * 版本号
     */
    private static final String VERSION = "1.0";
    /**
     * 应用appId;
     */
    private String appId;
    /**
     * 私钥
     */
    private String fbPrivateKey;
    /**
     * 公钥
     */
    private String umPublicKey;
    /**
     * 待加密字段缓存
     */
    private static Map<Class<?>, List<Field>> encryptionMap = new ConcurrentHashMap<>();

    private static SerializeConfig SNAKE_CASE_CONFIG = new SerializeConfig();

    static {
        SNAKE_CASE_CONFIG.propertyNamingStrategy = PropertyNamingStrategy.SnakeCase;
    }

    public UmBankClientImpl(String appId, String fbPrivateKey, String umPublicKey) {
        this.appId = appId;
        this.fbPrivateKey = fbPrivateKey;
        this.umPublicKey = umPublicKey;
    }

    /**
     * 请求联动接口
     *
     * @param request            请求参数
     * @param umpayApiDefinition 方法枚举
     * @param postUrl            请求地址
     * @param <R>                返参
     * @return UmBaseResponse
     */
    @Override
    public <R> UmBankBaseResponse<R> excute(UmBankBizRequest<R> request, UmpayApiDefinition umpayApiDefinition, String postUrl) throws UmPayException {
        return excute(request, umpayApiDefinition, postUrl, this.appId);
    }

    /**
     * 请求联动接口
     *
     * @param request            请求参数
     * @param umpayApiDefinition 方法枚举
     * @param postUrl            请求地址
     * @param <R>                返参
     * @return UmBaseResponse
     */
    @Override
    public <R> UmBankBaseResponse<R> excute(UmBankBizRequest<R> request, UmpayApiDefinition umpayApiDefinition, String postUrl, String subAppId) throws UmPayException {
        String method = umpayApiDefinition.getMethod();
        //验证参数
        validateParam(request, method);
        if (StringUtils.isBlank(request.getMerId())) {
            request.setMerId(this.appId);
        }
        if (StringUtils.isBlank(request.getVersion())) {
            request.setVersion(VERSION);
        }
        String apiURL = postUrl + umpayApiDefinition.getMethod();
        //获取开始时间
        final long startTime = System.currentTimeMillis();

        //获取请求参数
        log.info("【umpay-sdk】接口调用开始 >> url={}, request={}", apiURL, JSON.toJSONString(request, SNAKE_CASE_CONFIG));
        try {
            // 字段加密
            encryptFiled(request);
            // 获取请求参数
            String body = getRequestData(request, method, this.appId);
            ConfigContext apicontext = new ConfigContext(apiURL, this.appId);
            // 执行post请求
            final Map<String, Object> result = UmHttpClient.post(apicontext, body, this.fbPrivateKey);
            log.info("【umpay-sdk】接口调用结束 >> url={},request={},response={},cost={}", apiURL, body, result.get("data"), System.currentTimeMillis() - startTime);
            // 校验请求是否成功
            checkHttpCodeSuccess(result);
            // 获取联动的响应结果
            String data = (String) result.get("data");
            return parseResponse(data, request);
        } catch (Exception ex) {
            log.error("【umpay-sdk】接口调用失败 >> url={},method={},request={},ex={},cost={}", postUrl, method, JSON.toJSONString(request, SNAKE_CASE_CONFIG), ExceptionUtils.getStackTrace(ex), System.currentTimeMillis() - startTime);
            throw new UmPayException(ex.getMessage());
        }
    }

    @Override
    public <R> UmBankBaseResponse<R> uploadFile(UmBankBizRequest<R> request, UmpayApiDefinition umpayApiDefinition, File file, String postUrl, String subAppId) throws UmPayException {
        String method = umpayApiDefinition.getMethod();
        //验证参数
        validateParam(request, method);
        if (StringUtils.isBlank(request.getMerId())) {
            request.setMerId(this.appId);
        }

        String apiURL = postUrl + umpayApiDefinition.getMethod();
        //获取开始时间
        final long startTime = System.currentTimeMillis();

        //获取请求参数
        log.info("【umpay-sdk】接口调用开始 >> url={}, request={}, file={}", apiURL, JSON.toJSONString(request, SNAKE_CASE_CONFIG), file.getName());
        try {
            // 字段加密
            encryptFiled(request);
            // 获取请求参数
            String body = getRequestData(request, method, this.appId);
            ConfigContext apicontext = new ConfigContext(apiURL, this.appId);
            // 执行post请求
            final Map<String, Object> result = UmFileClinet.post(apicontext, body, file, this.fbPrivateKey);
            log.info("【umpay-sdk】接口调用结束 >> url={},request={},file={}, response={},cost={}", apiURL, body, file.getName(), result.get("data"), System.currentTimeMillis() - startTime);
            // 校验请求是否成功
            checkHttpCodeSuccess(result);
            // 获取联动的响应结果
            String data = (String) result.get("data");
            return parseResponse(data, request);
        } catch (Exception ex) {
            log.error("【umpay-sdk】接口调用失败 >> url={},method={},request={},ex={},cost={}", postUrl, method, JSON.toJSONString(request, SNAKE_CASE_CONFIG), ExceptionUtils.getStackTrace(ex), System.currentTimeMillis() - startTime);
            throw new UmPayException(ex.getMessage());
        }
    }

    /**
     * 文件上传
     *
     * @param request            请求参数
     * @param umpayApiDefinition 方法枚举
     * @param getUrl             获取url
     * @return {@link UmBankBaseResponse }<{@link R }>
     */
    @Override
    public <R> InputStream downloadFile(UmBankBizRequest<R> request, UmpayApiDefinition umpayApiDefinition, String getUrl) throws UmPayException {
        String method = umpayApiDefinition.getMethod();
        //验证参数
        validateParam(request, method);
        if (StringUtils.isBlank(request.getMerId())) {
            request.setMerId(this.appId);
        }
        if (StringUtils.isBlank(request.getVersion())) {
            request.setVersion(VERSION);
        }
        String apiURL = getUrl + umpayApiDefinition.getMethod();
        //获取开始时间
        final long startTime = System.currentTimeMillis();

        //获取请求参数
        log.info("【umpay-sdk】接口调用开始 >> url={}, request={}", apiURL, JSON.toJSONString(request, SNAKE_CASE_CONFIG));
        try {
            // 字段加密
            encryptFiled(request);
            // 获取请求参数
            ConfigContext apicontext = new ConfigContext(apiURL, this.appId);
            Map<String, String> reqMap = new HashMap<>(16);
            for (Field field : request.getClass().getDeclaredFields()) {
                field.setAccessible(true);
                String fieldName = field.getName();
                String finalFieldName = fieldName.replaceAll("([A-Z])", "_$1").toLowerCase();
                String value = field.get(request).toString();
                reqMap.put(finalFieldName, value);
            }
            // 执行get请求
            InputStream inputStream = UmHttpClient.downloadFile(apicontext, reqMap, this.fbPrivateKey);
            log.info("【umpay-sdk】接口调用结束 >> url={},request={},response={},cost={}", apiURL, reqMap, inputStream, System.currentTimeMillis() - startTime);

            return inputStream;
        } catch (Exception ex) {
            log.error("【umpay-sdk】接口调用失败 >> url={},method={},request={},ex={},cost={}", getUrl, method, JSON.toJSONString(request, SNAKE_CASE_CONFIG), ExceptionUtils.getStackTrace(ex), System.currentTimeMillis() - startTime);
            throw new UmPayException(ex.getMessage());
        }
    }

    /**
     * 校验http响应码是否请求成功
     *
     * @param resMap
     */
    private static void checkHttpCodeSuccess(Map<String, Object> resMap) {
        if (resMap != null && resMap.size() != 0) {
            int statusCode = (Integer) resMap.get("statusCode");
            if (statusCode != 200) {
                throw new RuntimeException("HttpClient,error status code :" + statusCode);
            }
        } else {
            throw new RuntimeException("Connection refused: connect");
        }
    }

    /**
     * 参数校验
     *
     * @param request 请求参数
     * @param method  请求方法
     * @param <R>
     */
    private <R> void validateParam(UmBizRequest<R> request, String method) {
        if (request == null) {
            throw new IllegalArgumentException("接口请求参数不能为空");
        }
        if (StringUtils.isBlank(fbPrivateKey)) {
            throw new IllegalArgumentException("私钥不能为空");
        }
        if (StringUtils.isBlank(umPublicKey)) {
            throw new IllegalArgumentException("公钥不能为空");
        }
        if (StringUtils.isBlank(method)) {
            throw new IllegalArgumentException("请求方法不能为空");
        }
        //注解验证
        ValidateUtil.validateWithThrow(request);
    }

    /**
     * 获取签名参数
     *
     * @param request 请求参数
     * @param method  方法名
     * @param <R>     返参
     * @return Map<String, String>
     */
    private <R> String getRequestData(UmBizRequest<R> request, String method, String appId) {
        return JSON.toJSONString(request, SNAKE_CASE_CONFIG);
    }

    /**
     * 字段加密
     */
    private void encryptFiled(Object request) throws Exception {
        // 初始化字段加密元信息
        List<Field> fieldList = initEncryptFiledMap(request.getClass());
        for (Field field : fieldList) {
            Object value = field.get(request);
            if (value instanceof String) {
                String encryptValue = encrypt((String) value);
                field.set(request, encryptValue);
            } else if (value instanceof UmBankBizItemRequest) {
                // 如果是子对象，则递归处理
                encryptFiled(value);
            }
        }
    }

    /**
     * 初始化字段加密map
     *
     * @param clazz
     */
    private List<Field> initEncryptFiledMap(Class<?> clazz) {
        List<Field> fieldList = encryptionMap.get(clazz);
        if (fieldList == null) {
            synchronized (UmBankClientImpl.class) {
                fieldList = encryptionMap.get(clazz);
                if (fieldList == null) {
                    fieldList = new ArrayList<>();
                    for (Field field : clazz.getDeclaredFields()) {
                        if (field.isAnnotationPresent(EncryptField.class)) {
                            field.setAccessible(true);
                            fieldList.add(field);
                            encryptionMap.put(clazz, fieldList);
                        }
                    }
                }
            }
        }
        return fieldList;
    }

    /**
     * UTF-8编码后使用联动公钥进行RSA加密，最后使用BASE64编码
     *
     * @param value
     * @return
     * @throws Exception
     */
    private String encrypt(String value) throws Exception {
        PublicKey pubKey = SignUtil.getPublicKeyFromX509("RSA",
                new ByteArrayInputStream(this.umPublicKey.getBytes()));
        Cipher cipher = Cipher.getInstance(KeyFactory.getInstance("RSA").getAlgorithm());
        cipher.init(1, pubKey);
        String str = (new String(Base64.encode(cipher.doFinal(value.getBytes("UTF-8"))))).replace("\n", "");
        return str;
    }

    /**
     * 结果解析
     *
     * @param result  返回结果
     * @param request 请求参数
     * @param <R>     结果
     * @return UmBaseResponse
     */
    @SuppressWarnings("unchecked")
    private <R> UmBankBaseResponse<R> parseResponse(String result, UmBizRequest<R> request) {
        JSONObject resultJson = JSON.parseObject(result);
        UmBankMetaResponse metaResponse = resultJson.getObject("meta", UmBankMetaResponse.class);

        // 结算转换
        UmBankBaseResponse<R> response = new UmBankBaseResponse<>();
        if (metaResponse != null) {
            response.setRetCode(metaResponse.getRetCode());
            response.setRetMsg(metaResponse.getRetMsg());
        } else {
            response.setRetCode("9999");
            response.setRetMsg("联动返回空响应结果");
        }
        if (resultJson.containsKey("data")) {
            response.setData(JSON.parseObject(resultJson.getString("data"), request.getResponseClass()));
        }
        JSONArray jsonArray = resultJson.getJSONArray("links");
        if (jsonArray != null && jsonArray.size() > 0) {
            response.setLinks(jsonArray.toJavaList(UmBankLinkResponse.class));
        }
        return response;
    }
}