/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.umpay.sdk.client.impl;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.http.HttpException;
import cn.hutool.http.HttpResponse;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.fshows.umpay.sdk.client.UploadPictureClient;
import com.fshows.umpay.sdk.enums.UmPictureApiEnum;
import com.fshows.umpay.sdk.exception.UmPayException;
import com.fshows.umpay.sdk.request.picture.AggregationFilesUploadDirectBase64Request;
import com.fshows.umpay.sdk.request.picture.FilesUploadDirectRequest;
import com.fshows.umpay.sdk.request.picture.item.FilesUploadDirectBase64ItemRequest;
import com.fshows.umpay.sdk.response.picture.item.FileInfoItemResponse;
import com.fshows.umpay.sdk.util.FsHttpUtil;
import com.fshows.umpay.sdk.util.SignUtil;
import com.fshows.umpay.sdk.util.StringPool;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.net.URI;
import java.util.Base64;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author youmingming
 * @version UploadPictureClientImpl.java, v 0.1 2022-03-09 2:43 PM youmingming
 */
@Slf4j
public class UploadPictureClientImpl implements UploadPictureClient {

    /**
     * 版本号
     */
    private static final String VERSION = "1.0";
    private static final Integer TIMEOUT = 30000;

    public UploadPictureClientImpl() {
    }

    /**
     * 图片上传
     *
     * @param request 图片地址
     * @return AggregationFilesUploadDirectResponse
     */
    @Override
    public FileInfoItemResponse upload(FilesUploadDirectRequest request) throws UmPayException {
        long beginTime = System.currentTimeMillis();
        log.info("【umpay-sdk】>> 图片上传开始 request={},begin={}", request, beginTime);
        String uploadUrl = request.getUploadUrl();
        String pictureUrl = request.getPictureUrl();
        if (StringUtils.isBlank(uploadUrl) || StringUtils.isBlank(pictureUrl)) {
            throw new IllegalArgumentException("上传和下载地址都不能为空");
        }
        try {
            final String path = new URI(pictureUrl).getPath();
            final int index = path.lastIndexOf(".");
            String suffix = "png";
            if (index > 0) {
                suffix = StringUtils.substring(path, index + 1);
            }
            final HttpResponse httpResponse = HttpUtil.createGet(pictureUrl).executeAsync();
            if (!httpResponse.isOk()) {
                throw new HttpException("Server response error with status code: [{}]", httpResponse.getStatus());
            }
            log.info("【umpay-sdk】 >> 图片连接下载成功 >> request={},result={},cost={}", request, System.currentTimeMillis() - beginTime);
            beginTime = System.currentTimeMillis();
            final String pictureBase64 = Base64.getEncoder().encode(httpResponse.bodyBytes()).toString();
            final FilesUploadDirectBase64ItemRequest itemRequest = new FilesUploadDirectBase64ItemRequest();
            itemRequest.setFile(pictureBase64);
            itemRequest.setMineType(suffix);
            itemRequest.setRemarks(suffix);
            AggregationFilesUploadDirectBase64Request base64Request = new AggregationFilesUploadDirectBase64Request();
            base64Request.setFileList(Collections.singletonList(itemRequest));

            final Map<String, String> requestData = this.getRequestData(request, UmPictureApiEnum.AGGREGATION_FILES_UPLOAD_DIRECT_BASE64.getValue(), request.getAppId(), request.getFbPrivateKey());
            final Integer timeout = ObjectUtil.isNotNull(request.getTimeout()) && request.getTimeout() > 0 ? request.getTimeout() : TIMEOUT;
            log.info("【umpay-sdk】 >> 图片上传接口调用开始 >> request={},requestData={},result={}", request, requestData);
            final String result = FsHttpUtil.post(uploadUrl, requestData, timeout);
            log.info("【umpay-sdk】 >> 图片上传结束 >> request={},requestData={},result={},cost={}", request, requestData, result, System.currentTimeMillis() - beginTime);
            if (StringUtils.isBlank(result)) {
                log.warn("【umpay-sdk】 >> 图片上传失败 >> request={}", request);
                throw new UmPayException("图片上传失败");
            }
            final JSONObject resultObject = JSONObject.parseObject(result);
            final Boolean success = resultObject.getBoolean("success");
            if (ObjectUtil.isNull(success) || !success) {
                log.warn("【umpay-sdk】 >> 图片上传失败 >> request={},result={}", request, result);
                final String errorCode = resultObject.getString("error_code");
                final String errorMsg = resultObject.getString("error_message");
                throw new UmPayException(StrUtil.format("[{}]{}", errorCode, errorMsg));
            }
            final List<FileInfoItemResponse> fileInfoItemResponses = JSON.parseArray(resultObject.getString("return_value"), FileInfoItemResponse.class);
            return fileInfoItemResponses.get(0);
        } catch (Exception e) {
            log.error("【umpay-sdk】 >> 图片上传异常 >> request={},e={},cost={}", request, e, System.currentTimeMillis() - beginTime);
            throw new UmPayException(e.getMessage(), e);
        }
    }

    /**
     * 获取签名参数
     *
     * @param request 请求参数
     * @param method  方法名
     * @return Map<String, String>
     */
    private Map<String, String> getRequestData(FilesUploadDirectRequest request, String method, String appId, String fbPrivateKey) {
        Map<String, String> signMap = new HashMap<>(16);
        signMap.put("app_id", appId);
        signMap.put("method", method);
        signMap.put("version", VERSION);
        signMap.put("content", JSON.toJSONString(request));
        String sign = SignUtil.sign(signMap, StringPool.DEFAULT_SIGN_TYPE, fbPrivateKey);
        signMap.put("sign", sign);
        return signMap;
    }
}