/**
 * fshows.com
 * Copyright (C) 2013-2025 All Rights Reserved.
 */
package com.fshows.postar;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.fshows.postar.apienum.PostarShareApiEnum;
import com.fshows.postar.request.PostarBaseReq;
import com.fshows.postar.response.PostarBizRes;
import com.fshows.postar.util.SignUtil;
import com.fshows.sdk.core.client.base.AbstractApiClient;
import com.fshows.sdk.core.client.base.handler.IHttpRequestHandler;
import com.fshows.sdk.core.client.base.handler.ISerializableHandler;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.ClientInfoModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.client.component.http.PostHttpRequestHandler;
import com.fshows.sdk.core.client.component.serializable.JsonSerializableHandler;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 分账相关API客户端
 *
 * @author wangyi
 * @version PostarShareApiClient.java, v 0.1 2025-12-10 09:19 wangyi
 */
@Slf4j
public class PostarShareApiClient extends AbstractApiClient<PostarBaseReq, PostarBizRes, PostarShareApiEnum> {

    /**
     * 序列化处理器
     */
    protected ISerializableHandler paramSerializable = new JsonSerializableHandler();

    /**
     * 请求执行器
     */
    protected IHttpRequestHandler httpRequestHandler = new PostHttpRequestHandler();

    public PostarShareApiClient(DefaultClientConfigModel apiClientConfig) throws FsApiException {
        super(apiClientConfig);
    }

    /**
     * 执行请求
     *
     * @param request
     * @param iApiDefinition
     * @param merchantConfigModel
     * @return
     * @throws FsApiException
     */
    @Override
    protected PostarBizRes doExecute(PostarBaseReq request, PostarShareApiEnum iApiDefinition, DefaultClientConfigModel merchantConfigModel) throws FsApiException {
        long beginTime = System.currentTimeMillis();
        LogUtil.info(log, "{} >> 执行请求开始 >> iApiDefinition={}, request={}",
                getClientInfo().getClientDesc(), iApiDefinition, request);
        // 构建请求上下文
        DefaultRequestContext requestContext = buildRequestContext(iApiDefinition, request, merchantConfigModel);
        ApiRequestModel apiRequestModel = null;
        ApiResponseModel apiResponseModel = null;
        try {
            // 入参数校验
            checkParam(request, requestContext);
            // 构建请求数据
            apiRequestModel = buildApiRequestModel(request, requestContext);

            apiResponseModel = httpRequest(apiRequestModel, requestContext);
            Object requestBody = ObjectUtils.defaultIfNull(apiRequestModel.getRequestBody(), apiRequestModel.getRequestForm());
            LogUtil.info(log, "{} >> 请求结束 >> url={}, method={}, request={}, response={}, cost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), iApiDefinition, requestBody,
                    apiResponseModel.getResponseBody(), System.currentTimeMillis() - beginTime);
            // 反序列化构建响应结果
            return buildApiResponse(apiResponseModel, apiRequestModel, requestContext);
        } catch (FsApiException e) {
            LogUtil.error(log, "{} >> 请求异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, request, apiRequestModel, apiResponseModel);
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, request, apiRequestModel, apiResponseModel);
            throw new FsApiException(e.getMessage(), e);
        }
    }

    /**
     * 构建请求参数
     *
     * @param request
     * @param context
     * @return
     */
    @Override
    protected ApiRequestModel buildApiRequestModel(PostarBaseReq request, DefaultRequestContext context) {
        // 请求内容
        String requestBody = "";
        try {
            ApiRequestModel apiRequestModel = new ApiRequestModel();
            // 设置网关地址
            apiRequestModel.setApiURL(context.getApiClientConfig().getApiParentURL() + context.getIApiDefinition().getApiURI());
            // 请求内容
            JSONObject bodyJson = (JSONObject) JSONObject.toJSON(request);
            bodyJson.put("agetId", context.getApiClientConfig().getAppId());
            bodyJson.put("version", context.getIApiDefinition().getVersion());
            List<String> params = new ArrayList<>(bodyJson.size());
            bodyJson.forEach((s, o) -> {
                if (o != null) {
                    params.add(s + "=" + o);
                }
            });
            Collections.sort(params);
            String sign = SignUtil.encrypt(apiClientConfig.getPayCompanyPublicKey(), SignUtil.getSHA256(StringUtils.join(params, "&")));
            bodyJson.put("sign", sign);
            apiRequestModel.setRequestBody(JSONObject.toJSONString(bodyJson));
            Map<String, String> headMap = new HashMap<>();
            headMap.put("Accept-Encoding", "identity");
            apiRequestModel.setHeadMap(headMap);
            return apiRequestModel;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求国通星驿异常 >> iApiDefinition={}, request={}, requestBody={}", e,
                    getClientInfo().getClientDesc(), context.getIApiDefinition(), request, requestBody);
            throw new RuntimeException(e);
        }
    }

    /**
     * 处理客户端信息
     *
     * @return
     */
    @Override
    protected ClientInfoModel getClientInfo() {
        ClientInfoModel clientInfoModel = new ClientInfoModel();
        clientInfoModel.setClientCode("postar-sdk");
        clientInfoModel.setClientName("国通星驿");
        return clientInfoModel;
    }

    @Override
    protected PostarBizRes buildApiResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        Object dataObj = null;
        try {
            // 响应内容
            String responseBody = apiResponseModel.getResponseBody();
            JSONObject resJson = JSONObject.parseObject(responseBody);
            dataObj = resJson.get("data");
            if (dataObj instanceof String) {
                return JSONObject.parseObject(resJson.toJSONString(), requestContext.getIApiDefinition().getResponseClass());
            } else if (dataObj instanceof JSONObject) {
                JSONObject dataJson = (JSONObject) dataObj;
                resJson.remove("data");
                dataJson.putAll(resJson);
                return JSONObject.parseObject(dataJson.toJSONString(), requestContext.getIApiDefinition().getResponseClass());
            } else if (dataObj instanceof JSONArray) {
                return resJson.toJavaObject(requestContext.getIApiDefinition().getResponseClass());
            }
            throw new FsApiException(resJson.getString("code") + " : " + resJson.getString("msg"));
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 国通星驿响应异常 >> iApiDefinition={}, request={}, 业务响应报文dataJson={}", e,
                    getClientInfo().getClientDesc(), requestContext.getIApiDefinition(), apiRequestModel.getRequest(), dataObj);
            throw new RuntimeException(e);
        }
    }

    @Override
    protected ApiResponseModel httpRequest(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws IOException {
        return httpRequestHandler.httpRequest(apiRequestModel, requestContext);
    }

    @Override
    public PostarBizRes execute(PostarBaseReq request, PostarShareApiEnum apiDefinition) throws FsApiException {
        return doExecute(request, apiDefinition);
    }

    @Override
    public PostarBizRes execute(PostarBaseReq request, PostarShareApiEnum apiDefinition, DefaultClientConfigModel configModel) throws FsApiException {
        return doExecute(request, apiDefinition, configModel);
    }
}
