/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package
        com.fshows.llmcode;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.io.file.FileWriter;
import cn.hutool.core.util.RandomUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSONObject;
import com.fshows.llmcode.base.enums.ApiTableTitleEnum;
import com.fshows.llmcode.base.enums.FieldMustEunm;
import com.fshows.llmcode.base.enums.UmpayFieldTypeEunm;
import com.fshows.llmcode.base.model.ApiFieldModel;
import com.fshows.llmcode.base.model.ApiModel;
import com.fshows.llmcode.web.WebRequest;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.beetl.core.Configuration;
import org.beetl.core.GroupTemplate;
import org.beetl.core.Template;
import org.beetl.core.resource.ClasspathResourceLoader;
import com.fshows.llmcode.llm.QwenLlmRequest;


import java.io.IOException;

import java.lang.reflect.Field;
import java.util.*;
import java.util.stream.Collectors;

/**
 * reqeust/response代码生成
 *
 * @author liluqing
 * @version GenerateClient.java, v 0.1 2022-03-06 19:47
 */
@Slf4j
public class AIGenerateClient {

    public static String API_TEMPALTE = "{}(\n" +
            "            \"{}\",\n" +
            "            \"{}\",\n" +
            "            FuStewardApiTypeEnum.XXX,\n" +
            "            {}.class,\n" +
            "            {}.class\n" +
            "    ),";

    /**
     * 基于AI大模型的
     *
     * @param args
     * @throws Exception
     */
    public static void main(String[] args) throws Exception {
        String desc ="5.4退款\n" +
                "5.4.1发起方  \n" +
                "商户系统\n" +
                "5.4.2业务流程\n" +
                "支持部分退款与全额退款。\n" +
                "微信交易提交退款申请成功后，退款有一定延时，用零钱支付的退款20分钟内到账，银行卡支付的退款3个工作日内到账。\n" +
                "注意：当原交易参与优惠活动时，通道仅支持全额退款。实际情况以通道返回为准。\n" +
                "5.4.3请求报文\n" +
                "序号\t域名\t中文名称\t长度\t必输\t备注\n" +
                "1.\tmerchantId\t一级服务商机构编码\t15\tN\t\n" +
                "字段说明见3.1.10\n" +
                "2.\tmerchantNo\t商户号\t15\tY\t字段说明见3.1.11\n" +
                "3.\trefundOrderNo\t商户退款流水号\t32\tY\t合作方自定义流水号，唯一标识一笔交易（建议根据当前系统日期和时间加随机序列生成，确保唯一 ） \n" +
                "4.\torigOrderNo \t原交易流水号\t32\tY\t需要退款的原交易流水号 \n" +
                "5.\tamount \t退款金额\t10\tY\t单位为元，两位小数，该金额不能大于订单金额\n" +
                "6.\torgPcsDate\t原交易日期\t8\tY\tyyyyMMdd\n" +
                "7.\toccurAdd\t网络地址（终端IP）\t40\tN\tIpv4格式：xx.xx.xx.xx\n" +
                "Ipv6格式：\n" +
                "xx:xx:xx:xx:xx:xx:xx:xx\n" +
                "注:1.Ipv6需转换为16进制;\n" +
                "2.数字人民币退款交易为必填\n" +
                "8.\tGPS\t交易设备GPS\t32\tN\t字段说明见3.1.12。\n" +
                "9.\trefundReason\t退款原因\t128\tN\t填写退款描述，最多64个汉字。数币专用。\n" +
                "不要使用`~!@#$%^&+=|{}';'<>/?~！#￥%&+|{}‘；”“'。？等特殊字符\n" +
                "10.\ttermId\t终端编号\t8\tN\t上送规则如下:\n" +
                "1.上送终端编号为已在我行系统注册的且终端状态启用，上送终端编号应与商户编号相互匹配。\n" +
                "2.条码交易允许跨终端退款，即A终端发生的交易，可以通过B终端发起退款\n" +
                "5.4.4返回报文\n" +
                "返回字段说明：\n" +
                "返回码：“000000”时为退款交易成功。\n" +
                "“10400”时为订单处理中，商户需调用“交易查询”接口查询这笔退款交易最终状态。\n" +
                "序号\t域名\t中文名\t长度\t必填\t说明\t\n" +
                "1.\trespCode\t返回码\t6\tY\t详见数据字典“返回码”\t\n" +
                "2.\trespMsg\t返回信息\t256\tY\t详见数据字典“返回信息”\t\n" +
                "3.\ttradeNo\t平台流水号\t64\tY\t微信、支付宝、银联、数研所等平台返回的退款订单号。\t返回码为“000000”时返回左侧字段。\n" +
                "4.\tsuccessTime \t成功时间\t16\tY\tyyyyMMddHHmmss \t\n" +
                "5.\tchannelOrderNo \t商户通道退款订单号\t50\tY\t银行系统生成，发往微信支付宝银联、数研所通道的通道退款交易订单号\t\n" +
                "6.\tamount\t退款金额\t10\tY\t本次退款金额。\t\n" +
                "7.\torigChannelNo\t原商户通道交易订单号\t50\tY\t需要退款的原商户通道交易订单号\t\n" +
                "8.\tylDctDetail\t银联-商品优惠明细内容\t6000\tN\t详情参见要素说明\t\n" +
                "9.\tylCouponInfo\t银联-优惠信息\t500\tN\t详情参见要素说明\t\n" +
                "10.\twxSettlementRefundFee\t微信-退款金额\t10\tN\t微信-去掉非充值代金券退款金额后的退款金额，退款金额=申请退款金额-非充值代金券退款金额，退款金额<=申请退款金额\t\n" +
                "11.\twxCashRefundFee \t微信-现金退款金额 \t10\tN\t微信-现金退款金额，单位为元 \t\n" +
                "12.\twxCouponRefundFee \t微信-代金券退款总金额 \t10\tN\t微信-代金券退款金额<=退款金额，退款金额-代金券或立减优惠退款金额为现金 \t\n" +
                "13.\twxRefundDetail \t微信-优惠退款详情 \t\tN\t详情参见要素说明\t\n" +
                "14.\taliPresentRefundBuyerAmount\t支付宝-买家退款金额\t10\tN\t支付宝-本次退款金额中买家退款金额\t\n" +
                "15.\taliPresentRefundDiscountAmount\t支付宝-平台优惠退款金额\t10\tN\t支付宝-本次退款金额中平台优惠退款金额\t\n" +
                "16.\taliPresentRefundMdiscountAmount\t支付宝-商家优惠退款金额\t10\tN\t支付宝-本次退款金额中商家优惠退款金额\t";

        WebRequest request = new WebRequest();
        request.setTextDesc(desc);
        request.setInterfaceCode("UniouOrderCreate");
        request.setRequestObjectAbsolutePath("/Users/zhaoxumin/project/llm-code-generation/src/main/java/com/fshows/llmcode/generation/request");
        request.setResponseObjectAbsolutePath("/Users/zhaoxumin/project/llm-code-generation/src/test/java/com/fshows/llmcode/response");
//        request.setIgnoreParamList("merchantId,merchantNo,respCode,respDesc");
        request.setIgnoreParamList(null);
        request.setRequestObjectPrefix("ZxmPrefix");
        request.setRequestObjectSuffix("ZxmSuffix");
        request.setResponseObjectPrefix("ZzmPrefix");
        request.setResponseObjectSuffix("ZzmSuffix");

        // 执行代码生成
        execute(request, null);
    }

    /**
     * 执行代码生成
     *
     * @param webRequest 客户端请求参数
     * @throws IOException
     */
    public static void execute(WebRequest webRequest, String apiModelStr) throws IOException {
        trimStringProperties(webRequest);
        // 通过ai模型解析出文档里的字段
        ApiModel apiModel = null;
        if (StringUtils.isBlank(apiModelStr)) {
            apiModel = QwenLlmRequest.analysisText(webRequest.getTextDesc());
        } else {
            apiModel = JSONObject.parseObject(apiModelStr, ApiModel.class);
        }

        String interfaceCode = StringUtils.defaultIfBlank(webRequest.getInterfaceCode(), apiModel.getInterfaceCode());
        String requestName = webRequest.getRequestObjectPrefix() + interfaceCode + webRequest.getRequestObjectSuffix();
        String responseName = webRequest.getResponseObjectPrefix() + interfaceCode + webRequest.getResponseObjectSuffix();

        // 过滤忽略的字段
        List<String> ignoreList = Arrays.stream(webRequest.getIgnoreParamList().split(","))
                .map(String::trim)
                .collect(Collectors.toList());
        apiModel.setRequest(apiModel.getRequest().stream().filter(e -> !ignoreList.contains(e.getFieldName())).collect(Collectors.toList()));
        apiModel.setResponse(apiModel.getResponse().stream().filter(e -> !ignoreList.contains(e.getFieldName())).collect(Collectors.toList()));

        // 生成request
        generateCode(ApiTableTitleEnum.REQUEST, webRequest.getRequestObjectAbsolutePath(), requestName, apiModel.getInterfaceName(), apiModel.getRequest());
        // 生成response
        generateCode(ApiTableTitleEnum.RESPONSE, webRequest.getResponseObjectAbsolutePath(), responseName, apiModel.getInterfaceName(), apiModel.getResponse());
//  todo 枚举待生成      System.out.println("apiEnum:\n" + outApiTemplate(apiModel, interfaceCode, requestName, responseName));
    }

    /**
     *
     *
     * @param apiModel
     * @return
     */
    public static String outApiTemplate(ApiModel apiModel, String interfaceCode, String requestName, String responseName) {
        String apiName = StrUtil.toUnderlineCase(interfaceCode).toUpperCase();
        String apiEnumStr = StrUtil.format(API_TEMPALTE,
                apiName,
                apiModel.getInterfaceName(),
                apiModel.getInterfaceURL(),
                requestName,
                responseName);
        return apiEnumStr;
    }

    /**
     * 生成代码
     *
     * @param fieldList          待解析的html目录
     * @param apiTableTitleEnum 表哥头类型
     * @param path              文件路径
     * @param className         类名
     * @throws IOException
     */
    public static void generateCode(ApiTableTitleEnum apiTableTitleEnum, String path, String className, String interfaceName, List<ApiFieldModel> fieldList) throws IOException {
        // 解析出文档里的字段
        if (CollectionUtil.isEmpty(fieldList)) {
            return;
        }
        // 填充数据
        compensateApiField(fieldList);
        // 执行模板引擎生成代码
        String content = doGenerateCode(fieldList, apiTableTitleEnum, true,
                className, interfaceName, path);
        // 输出到文件
        outToFile(content, path, className);
        log.info("执行成功");
    }

    /**
     * 输出内容到文件
     *
     * @param content
     * @param path
     * @param fileName
     */
    private static void outToFile(String content, String path, String fileName) {
        String filePath = StringUtils.endsWith(path, "\\")
                ? path + fileName + ".java" : path + "\\" + fileName + ".java";
        FileWriter writer = new FileWriter(filePath);
        writer.write(content);
    }

    /**
     * 数据补偿
     *
     * @param fieldList
     */
    public static void compensateApiField(List<ApiFieldModel> fieldList) {
        if (CollectionUtil.isEmpty(fieldList)) {
            return;
        }
        for (ApiFieldModel apiFieldModel : fieldList) {
            // 设置最大长度
            apiFieldModel.setMaxLength(getMaxLength(apiFieldModel.getMaxLength(), apiFieldModel.getFieldType()));
            apiFieldModel.setFieldType(UmpayFieldTypeEunm.getByValue(apiFieldModel.getFieldType()).getJavaType());
            // 设置Java属性名称（api文档中的字段名称转化为驼峰命名）
            apiFieldModel.setJavaFieldName(StrUtil.toCamelCase(apiFieldModel.getFieldName()));
            // 设置是否必填
            apiFieldModel.setMust(FieldMustEunm.getByValue(apiFieldModel.getMust()).getName());
        }
    }

    /**
     * 字段类型
     *
     * @return
     */
    private static String getMaxLength(String maxLength, String fileType) {
        if (StringUtils.isNotBlank(maxLength)) {
            return maxLength;
        }
        if (StringUtils.isBlank(fileType)
                || !StringUtils.contains(fileType, "(")
                || !StringUtils.contains(fileType, ")")) {
            return null;
        }
        String maxLengthStr = StringUtils.substring(fileType, fileType.indexOf("(")+1, fileType.indexOf(")"));
        if (maxLengthStr.indexOf("~") > 0) {
            String[] strArr = StringUtils.split(maxLengthStr, "~");
            return strArr[strArr.length-1];
        }
        return maxLengthStr;
    }

    /**
     * 执行模版渲染
     *
     * @param fieldList
     * @param isGenerateAnnotation 是否生成校验注解
     * @throws IOException
     */
    public static String doGenerateCode(List<ApiFieldModel> fieldList, ApiTableTitleEnum apiTableTitleEnum
            , Boolean isGenerateAnnotation, String className, String interfaceName, String path) throws IOException {
        //初始化代码
        ClasspathResourceLoader resourceLoader = new ClasspathResourceLoader();
        Configuration cfg = Configuration.defaultConfiguration();
        GroupTemplate gt = new GroupTemplate(resourceLoader, cfg);
        //获取模板
        Template t = gt.getTemplate(apiTableTitleEnum.getTemplate());
        t.binding("fieldList", fieldList);
        t.binding("isGenerateAnnotation", isGenerateAnnotation);
        t.binding("className", className);
        t.binding("package", getPackage(path));
        t.binding("version", Long.toString(RandomUtil.randomLong(100000000, 999999999)));
        t.binding("date", DateUtil.formatDateTime(new Date()));
        t.binding("interfaceName", interfaceName);
        //渲染结果
        String str = t.render();
        return str;
    }

    /**
     * 根据路径解析出package
     * @param filePath
     * @return
     */
    public static String getPackage(String filePath) {
        int startIndex = StringUtils.lastIndexOf(filePath, "com\\");
        if (startIndex < 0) {
            startIndex = StringUtils.lastIndexOf(filePath, "com/");
        }
        if (startIndex < 0) {
            return "";
        }

        String path = StringUtils.substring(filePath, startIndex);
        path = StringUtils.replaceAll(path, "\\\\", ".");
        path = StringUtils.replaceAll(path, "/", ".");
        if (StringUtils.endsWith(path,".")) {
            path =  StringUtils.substring(filePath, 0, path.length() - 1);
        }
        return path;
    }

    /**
     * 将对象String类型的属性，去除前后空格
     *
     * @param obj
     */
    public static void trimStringProperties(Object obj) {
        Field[] fields = obj.getClass().getDeclaredFields();
        for (Field field : fields) {
            field.setAccessible(true);
            if (field.getType() == String.class) {
                try {
                    String value = (String) field.get(obj);
                    if (value != null) {
                        // 使用 trim() 方法去除字符串两端的空格
                        String trimmedValue = value.trim();
                        field.set(obj, trimmedValue);
                    } else {
                        // null值默认设置空串
                        field.set(obj, StrUtil.EMPTY);
                    }
                } catch (IllegalAccessException e) {
                    e.printStackTrace();
                }
            }
        }
    }
}