/**
 * fshows.com
 * Copyright (C) 2013-2020 All Rights Reserved.
 */
package com.fshows.leshuapay.sdk.client.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.fshows.leshuapay.sdk.client.LeshuaSubsidyClient;
import com.fshows.leshuapay.sdk.exception.LeshuaException;
import com.fshows.leshuapay.sdk.request.LeshuaBizRequest;
import com.fshows.leshuapay.sdk.response.LeshuaBaseResponse;
import com.fshows.leshuapay.sdk.util.SignUtil;
import com.fshows.leshuapay.sdk.util.ValidateUtil;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.BasicResponseHandler;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;

import java.util.LinkedHashMap;


/**
 * @author linzy
 * @version LeshuaShareClientImpl.java, v 0.1 20210-05-06 11:40 AM linzy
 */
@Slf4j
@Data
public class LeshuaSubsidyClientImpl implements LeshuaSubsidyClient {

    /**
     * 从connectManager获取Connection 超时时间
     */
    private static final int CONNECTION_REQUEST_TIMEOUT = 10000;

    /**
     * 读取数据的超时时间
     */
    private static final int SOCKET_TIMEOUT = 10000;

    /**
     * 连接超时时间
     */
    private static final int CONNECT_TIMEOUT = 10000;
    /**
     * 非对象返回
     */
    private static final String NO_DATA_URL = "/open-api/agent-subsidy/query-agent-balance";
    /**
     * 商户服务地址
     */
    private String serverUrl;

    /**
     * 服务商编号
     */
    private String agentId;

    /**
     * 服务商私钥
     */
    private String privateKey;

    /**
     * 请求超时时间，单位毫秒
     */
    private Integer timeout;

    /**
     * @param serverUrl  商户地址
     * @param agentId    服务商编号
     * @param privateKey 服务商私钥
     * @param timeout    毫秒
     */
    public LeshuaSubsidyClientImpl(String serverUrl, String agentId, String privateKey, Integer timeout) {
        this.serverUrl = serverUrl;
        this.agentId = agentId;
        this.privateKey = privateKey;
        this.timeout = timeout;
    }

    /**
     * 调用商户相关请求
     *
     * @param request 商户入参
     * @param url     实际业务请求地址
     * @param timeout 超时设置，单位毫秒
     * @return
     * @throws LeshuaException
     */
    @Override
    public <T> LeshuaBaseResponse<T> execute(LeshuaBizRequest<T> request, String url, Integer timeout) throws LeshuaException {
        ValidateUtil.validateWithThrow(request);

        String postUrl = serverUrl + url;
        //构建请求对象
        JSONObject object = new JSONObject(new LinkedHashMap());
        object.put("version", "0.0.1");
        object.put("agentId", agentId);
        if (!url.equals(NO_DATA_URL)) {
            object.put("data", request);
        }
        //获取签名值
        String sign = SignUtil.getSubsidySign(object.toString(), privateKey);
        object.put("sign", sign);
        RequestConfig requestConfig = null;
        if (timeout == null || timeout <= 0) {
            requestConfig = RequestConfig.custom().setConnectionRequestTimeout(CONNECTION_REQUEST_TIMEOUT)
                    .setSocketTimeout(SOCKET_TIMEOUT).setConnectTimeout(CONNECT_TIMEOUT).build();
        } else {
            requestConfig = RequestConfig.custom().setConnectionRequestTimeout(timeout)
                    .setSocketTimeout(timeout).setConnectTimeout(timeout).build();
        }

        HttpPost httpPost = new HttpPost(postUrl);
        httpPost.setConfig(requestConfig);
        CloseableHttpClient httpClient = null;
        try {
            httpClient = HttpClients.createDefault();
            BasicResponseHandler handler = new BasicResponseHandler();
            StringEntity entity = new StringEntity(object.toJSONString(), "utf-8");
            entity.setContentEncoding("UTF-8");
            entity.setContentType("application/json");
            httpPost.setEntity(entity);
            String responseStr = httpClient.execute(httpPost, handler);
            log.info("乐刷请求返回  param={};response={}", JSON.toJSON(request), responseStr);
            JSONObject jsonObject = JSON.parseObject(responseStr);
            LeshuaBaseResponse<T> response = new LeshuaBaseResponse<>();
            response.setReqSerialNo(jsonObject.getString("reqSerialNo"));
            String data = jsonObject.toString();
            if (StringUtils.isBlank(data)) {
                return response;
            }
            response.setData(JSON.parseObject(data, request.getResponseClass()));
            return response;
        } catch (Exception e) {
            log.error("调用乐刷异常,request={};msg={}", JSON.toJSON(request), e.getMessage());
            throw new LeshuaException("调用失败", e);

        } finally {
            try {
                httpClient.close();
            } catch (Exception e) {
                log.error("关闭httpClient出错，错误信息,request={};msg={}", JSON.toJSON(request), e.getMessage());
            }
        }
    }

}
