/**
 * fshows.com
 * Copyright (C) 2013-2023 All Rights Reserved.
 */
package
        com.fshows.kqbill;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.fshows.kqbill.apienum.KqbillTradeApiEnum;
import com.fshows.kqbill.handler.KqbillClientConfigModel;
import com.fshows.kqbill.handler.KqbillPostHttpWithSSLRequestHandler;
import com.fshows.kqbill.request.KqbillBizReq;
import com.fshows.kqbill.request.KqbillHeaderReq;
import com.fshows.kqbill.response.KqbillBizRes;
import com.fshows.kqbill.util.SignUtil;
import com.fshows.sdk.core.client.base.AbstractApiClient;
import com.fshows.sdk.core.client.base.handler.IHttpRequestHandler;
import com.fshows.sdk.core.client.base.handler.ISerializableHandler;
import com.fshows.sdk.core.client.base.model.*;
import com.fshows.sdk.core.client.component.serializable.JsonSerializableHandler;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import com.fshows.sdk.core.util.ReqIdUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

/**
 * 嵌套参数客户端模版
 *
 * @author liluqing
 * @version NestTemplateApiClinet.java, v 0.1 2023-12-12 15:40
 */
@Slf4j
public class KqbillTradeApiClient extends AbstractApiClient<KqbillBizReq, KqbillBizRes, KqbillTradeApiEnum> {
    private KqbillClientConfigModel kqbillClientConfig;
    private SignUtil signUtil;

    /**
     * 序列化处理器
     */
    protected ISerializableHandler paramSerializable = new JsonSerializableHandler();

    /**
     * 请求执行器
     */
    protected IHttpRequestHandler httpRequestHandler = new KqbillPostHttpWithSSLRequestHandler();

    public KqbillTradeApiClient(DefaultClientConfigModel apiClientConfig) throws FsApiException {
        super(apiClientConfig);
        this.kqbillClientConfig = (KqbillClientConfigModel) apiClientConfig;

        signUtil = new SignUtil(apiClientConfig);
    }

    /**
     * 执行请求
     *
     * @param request
     * @param iApiDefinition
     * @param merchantConfigModel
     * @return
     * @throws FsApiException
     */
    @Override
    protected KqbillBizRes doExecute(KqbillBizReq request, KqbillTradeApiEnum iApiDefinition, DefaultClientConfigModel merchantConfigModel) throws FsApiException {
        long beginTime = System.currentTimeMillis();
        LogUtil.info(log, "{} >> 执行请求开始 >> iApiDefinition={}, request={}",
                getClientInfo().getClientDesc(), iApiDefinition, request);
        // 构建请求上下文
        DefaultRequestContext requestContext = buildRequestContext(iApiDefinition, request, merchantConfigModel);
        ApiRequestModel apiRequestModel = null;
        ApiResponseModel apiResponseModel = null;
        try {
            // 入参数校验
            checkParam(request, requestContext);
            // 构建请求数据
            apiRequestModel = buildApiRequestModel(request, requestContext);

            apiResponseModel = httpRequest(apiRequestModel, requestContext);
            Object requestBody = ObjectUtils.defaultIfNull(apiRequestModel.getRequestBody(), apiRequestModel.getRequestForm());
            LogUtil.info(log, "{} >> 请求结束[密文] >> url={}, method={}, request={}, response={}, cost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), iApiDefinition, requestBody,
                    apiResponseModel.getResponseBody(), System.currentTimeMillis() - beginTime);
            // 反序列化构建响应结果
            KqbillBizRes result = buildApiResponse(apiResponseModel, apiRequestModel, requestContext);
            LogUtil.info(log, "{} >> 请求结束[明文] >> url={}, method={}, request={}, response={}, cost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), iApiDefinition, JSONObject.toJSONString(request),
                    JSONObject.toJSONString(result), System.currentTimeMillis() - beginTime);
            return result;
        } catch (FsApiException e) {
            LogUtil.error(log, "{} >> 请求异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, request, apiRequestModel, apiResponseModel);
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, request, apiRequestModel, apiResponseModel);
            throw new FsApiException(e.getMessage(), e);
        }
    }

    /**
     * 构建请求参数
     *
     * @param request
     * @param context
     * @return
     */
    @Override
    protected ApiRequestModel buildApiRequestModel(KqbillBizReq request, DefaultRequestContext context) {
        // 请求内容
        String requestBody = "";
        try {
            ApiRequestModel apiRequestModel = new ApiRequestModel();
            // 设置网关地址
            apiRequestModel.setApiURL(context.getApiClientConfig().getApiParentURL());

            // header
            KqbillHeaderReq headerReq = new KqbillHeaderReq();
            headerReq.setVersion(context.getIApiDefinition().getVersion());
            headerReq.setMemberCode(context.getApiClientConfig().getAppId());
//            headerReq.setVendorMemberCode(context.getApiClientConfig().getAppId());
            headerReq.setMessageType(context.getIApiDefinition().getApiURI());
            headerReq.setExternalRefNumber(request.getExternalRefNumber());

            // 加密body
            JSONObject encryptBody = signUtil.encrypt(JSON.toJSONString(request));

            // 最终请求报文
            JSONObject requestString = new JSONObject();
            requestString.put("head", headerReq);
            requestString.put("requestBody", encryptBody);

            apiRequestModel.setRequestBody(requestString.toJSONString());
            Map<String, String> headMap = new HashMap<>();
            headMap.put("Accept-Encoding", "identity");
            apiRequestModel.setHeadMap(headMap);

            return apiRequestModel;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求快钱支付异常 >> iApiDefinition={}, request={}, requestBody={}", e,
                    getClientInfo().getClientDesc(), context.getIApiDefinition(), request, requestBody);
            throw new RuntimeException(e);
        }
    }

    /**
     * 处理客户端信息
     *
     * @return
     */
    @Override
    protected ClientInfoModel getClientInfo() {
        ClientInfoModel clientInfoModel = new ClientInfoModel();
        clientInfoModel.setClientCode("kqbill-sdk");
        clientInfoModel.setClientName("快钱支付");
        return clientInfoModel;
    }

    @Override
    protected KqbillBizRes buildApiResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        JSONObject dataJson = null;
        try {
            // 响应内容
            dataJson = JSONObject.parseObject(apiResponseModel.getResponseBody());

            // 返回对象
            KqbillBizRes bizRes = new KqbillBizRes();

            JSONObject resBody = dataJson.getJSONObject("responseBody");
            if (resBody != null && resBody.size() > 0) {
                String decrptBody = signUtil.decrypt(resBody);
                bizRes = JSONObject.parseObject(decrptBody, requestContext.getIApiDefinition().getResponseClass());
            }

            KqbillBizRes.KqbillHeaderRes headerRes = JSONObject.parseObject(dataJson.getString("head"), KqbillBizRes.KqbillHeaderRes.class);
            bizRes.setHead(headerRes);
            return bizRes;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 快钱支付响应异常 >> iApiDefinition={}, request={}, 业务响应报文dataJson={}", e,
                    getClientInfo().getClientDesc(), requestContext.getIApiDefinition(), apiRequestModel.getRequest(), dataJson);
            throw new RuntimeException(e);
        }
    }

    @Override
    protected ApiResponseModel httpRequest(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws IOException {
        return httpRequestHandler.httpRequest(apiRequestModel, requestContext);
    }

    @Override
    public KqbillBizRes execute(KqbillBizReq request, KqbillTradeApiEnum apiDefinition) throws FsApiException {
        return doExecute(request, apiDefinition);
    }

    @Override
    public KqbillBizRes execute(KqbillBizReq request, KqbillTradeApiEnum apiDefinition, DefaultClientConfigModel configModel) throws FsApiException {
        return doExecute(request, apiDefinition, configModel);
    }

    @Override
    protected DefaultRequestContext buildRequestContext(KqbillTradeApiEnum tradeApiDefinitionEnum, KqbillBizReq request, DefaultClientConfigModel merchantConfig) {
        DefaultRequestContext context = new DefaultRequestContext();
        context.setRequestId(ReqIdUtil.getId());
        context.setIApiDefinition(tradeApiDefinitionEnum);
        // 复制默认参数
        KqbillClientConfigModel clientConfig = this.kqbillClientConfig;
        KqbillClientConfigModel config = copyKqbillConfig(clientConfig);
        if (merchantConfig != null) {
            // merchantId
            config.setAgentId(StringUtils.isEmpty(merchantConfig.getAgentId()) ? clientConfig.getAgentId() : merchantConfig.getAgentId());
            // merchantId
            config.setAppId(StringUtils.isEmpty(merchantConfig.getAppId()) ? clientConfig.getAppId() : merchantConfig.getAppId());
            // 私钥
            config.setFubeiPrivateKey(StringUtils.isEmpty(merchantConfig.getFubeiPrivateKey()) ? clientConfig.getFubeiPrivateKey() : merchantConfig.getFubeiPrivateKey());
            // 公钥
            config.setPayCompanyPublicKey(StringUtils.isEmpty(merchantConfig.getPayCompanyPublicKey()) ? clientConfig.getPayCompanyPublicKey() : merchantConfig.getPayCompanyPublicKey());
        }
        context.setApiClientConfig(config);

        context.setClientInfoModel(getClientInfo());
        return context;
    }

    private KqbillClientConfigModel copyKqbillConfig(KqbillClientConfigModel clientConfigModel) {
        KqbillClientConfigModel configModel = new KqbillClientConfigModel();
        configModel.setAppId(clientConfigModel.getAppId());
        configModel.setAgentId(clientConfigModel.getAgentId());
        configModel.setFubeiPrivateKey(clientConfigModel.getFubeiPrivateKey());
        configModel.setPayCompanyPublicKey(clientConfigModel.getPayCompanyPublicKey());
        configModel.setPassword(clientConfigModel.getPassword());
        configModel.setApiParentURL(clientConfigModel.getApiParentURL());
        configModel.setMd5SignKey(clientConfigModel.getMd5SignKey());
        configModel.setSignTypeEnum(clientConfigModel.getSignTypeEnum());
        configModel.setVrifySignResponse(clientConfigModel.isVrifySignResponse());
        configModel.setCheckParam(clientConfigModel.isCheckParam());
        configModel.setConnectionTimeout(clientConfigModel.getConnectionTimeout());
        configModel.setReadTimeout(clientConfigModel.getReadTimeout());
        configModel.setCharset(clientConfigModel.getCharset());
        configModel.setHump(clientConfigModel.isHump());
        configModel.setExtendParam(new HashMap<>(clientConfigModel.getExtendParam()));
        configModel.setSingIgnoreNull(clientConfigModel.isSingIgnoreNull());
        // SSL设置
        configModel.setSSLVersion(clientConfigModel.getSSLVersion());
        configModel.setSSLCertPath(clientConfigModel.getSSLCertPath());
        configModel.setSSLKeyStore(clientConfigModel.getSSLKeyStore());
        configModel.setSSLKeyPass(clientConfigModel.getSSLKeyPass());
        return configModel;
    }

    /**
     * 加密
     *
     * @param data
     * @return
     * @throws Exception
     */
    public JSONObject encodeStr(String data) throws Exception {
        return signUtil.encrypt(data);
    }

    /**
     * 解密
     *
     * @param data
     * @return
     * @throws Exception
     */
    public String decodeStr(JSONObject data) throws Exception {
        return signUtil.decrypt(data);
    }
}