/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.fuiou.client.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.RandomUtil;
import cn.hutool.core.util.XmlUtil;
import com.alibaba.fastjson.JSONObject;
import com.fshows.fuiou.client.base.ApiClientConfig;
import com.fshows.fuiou.client.base.IFuiouApiDefinition;
import com.fshows.fuiou.client.base.IHttpRequest;
import com.fshows.fuiou.client.base.ISigner;
import com.fshows.fuiou.client.base.SignParam;
import com.fshows.fuiou.enums.settlement.FuiouSettlementApiDefinitionEnum;
import com.fshows.fuiou.exception.FuiouApiException;
import com.fshows.fuiou.request.base.FuiouBizRequest;
import com.fshows.fuiou.response.base.FuiouBizResponse;
import com.fshows.fuiou.response.settlement.FuiouQuerySettlementResponse;
import com.fshows.fuiou.response.settlement.FuiouQuerySettlementTxnResponse;
import com.fshows.fuiou.util.FuiouRequestUtils;
import com.fshows.fuiou.util.LogUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 富友结算接口客户端
 *
 * @author liluqing
 * @version SettlementFuiouApiClientImpl.java, v 0.1 2022-03-04 15:02
 */
@Slf4j
public class SettlementFuiouApiClientImpl extends AbstractFuiouApiClientImpl {

    public SettlementFuiouApiClientImpl(ApiClientConfig apiClientConfig) throws FuiouApiException {
        super(apiClientConfig);
    }

    public SettlementFuiouApiClientImpl(ApiClientConfig apiClientConfig, IHttpRequest httpRequest, ISigner signer) throws FuiouApiException {
        super(apiClientConfig, httpRequest, signer);
    }

    @Override
    public <T extends FuiouBizResponse, R extends IFuiouApiDefinition> T execute(FuiouBizRequest<T, R> request, R settlementApiDefinitionEnum) throws FuiouApiException {
        return (T) super.doExecute(request, settlementApiDefinitionEnum);
    }

    /**
     * 添加公共参数构建完整的请求
     *
     * @param bizRequest 业务请求参数
     * @param settlementApiDefinitionEnum
     * @return
     */
    @Override
    protected FuiouBizRequest buildFuiouBizRequest(FuiouBizRequest bizRequest, IFuiouApiDefinition settlementApiDefinitionEnum) {
        // 补充参数
        bizRequest.setInsCd(apiClientConfig.getAppId());
        bizRequest.setRandomStr(IdUtil.simpleUUID());
        return bizRequest;
    }

    @Override
    protected SignParam buildSignParam(FuiouBizRequest baseRequest) {
        SignParam signParam = new SignParam();
        signParam.setAppid(apiClientConfig.getAppId());
        signParam.setNonceStr(RandomUtil.randomString(12));
        signParam.setRequest(baseRequest);
        return signParam;
    }

    @Override
    protected String serializableRequest(FuiouBizRequest fuiouBaseRequest, String sign, IFuiouApiDefinition settlementApiDefinitionEnum) {
        Map<String, String> requstMap = FuiouRequestUtils.toMap(fuiouBaseRequest);
        return FuiouRequestUtils.generateXml(requstMap);
    }

    @Override
    protected FuiouBizResponse parseResponse(String resBody, IFuiouApiDefinition settlementApiDefinitionEnum) throws FuiouApiException {
        try {
            if (FuiouSettlementApiDefinitionEnum.QUERY_SETTLEMENT.equals(settlementApiDefinitionEnum)) {
                Map<String, Object> resMap = XmlUtil.xmlToMap(resBody);
                FuiouQuerySettlementResponse response = BeanUtil.toBean(resMap, settlementApiDefinitionEnum.getResponseClass());

                List<FuiouQuerySettlementTxnResponse> txnList = Lists.newArrayList();
                if (resMap.get("txnList") != null) {
                    // 目前只兼容查询一条的数据
                    HashMap hashMap = (HashMap) resMap.get("txnList");
                    FuiouQuerySettlementTxnResponse txn = JSONObject.parseObject(JSONObject.toJSONString(hashMap.get("txn")), FuiouQuerySettlementTxnResponse.class);
                    txnList.add(txn);
                    response.setTxnList(txnList);
                }
                return response;
            } else {
                Map<String, Object> resMap = XmlUtil.xmlToMap(resBody);
                return BeanUtil.toBean(resMap, settlementApiDefinitionEnum.getResponseClass());
            }
        } catch (Exception e) {
            LogUtil.error(log, "【fuiou-sdk】响应结果反序列化异常 >> resBody={}", e, resBody);
            throw new FuiouApiException("[fuiou-sdk]响应结果反序列化异常");
        }
    }
}