/**
 * fshows.com
 * Copyright (C) 2013-2023 All Rights Reserved.
 */
package
        com.fshows.sdk.core.client.component.signer;

import cn.hutool.core.codec.Base64;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.crypto.asymmetric.Sign;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.client.base.handler.IApiSignHandler;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import com.fshows.sdk.core.util.SignerUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.Map;

/**
 * rsa数据加签（基于对于Map<String, Object> requestMap的加签验签实现）
 * （如果参与加签验签的属性只有一层，没有多层嵌套的结果，可直接使用该实现）
 *
 * @author liluqing
 * @version RsaSignerImpl.java, v 0.1 2023-12-12 18:53
 */
@Slf4j
public class RsaSignHandlerImpl implements IApiSignHandler {

    @Override
    public String sign(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws FsApiException {
        String waitSignStr = null;
        try {
            // 默认配置
            DefaultClientConfigModel apiClientConfig = requestContext.getApiClientConfig();
            // 获取验签公钥
            String privateKey = apiClientConfig.getFubeiPrivateKey();
            // 获取待加签字符串
            Map<String, Object> data = apiRequestModel.getParamMap();
            // 获取待加签字符串
            waitSignStr = SignerUtil.getWaitSignStr(data, requestContext.getNoNeedSignWords(), requestContext.getApiClientConfig().isSingIgnoreNull());
            // 创建加签对象
            Sign sign = new Sign(
                    apiClientConfig.getSignTypeEnum().getAlgorithm(),
                    SecureUtil.decode(privateKey),
                    null);
            // 执行加签操作
            byte[] signed = sign.sign(waitSignStr.getBytes(apiClientConfig.getCharset()));
            return Base64.encode(signed);
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求加签失败 >> apiRequestModel={}, waitSignStr={}", e, requestContext.getClientInfoModel().getClientDesc(), apiRequestModel, waitSignStr);
            throw new FsApiException("请求加签失败", e);
        }
    }

    @Override
    public Boolean verifySign(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws FsApiException {
        String waitSignStr = null;
        try {
            // 默认配置
            DefaultClientConfigModel apiClientConfig = requestContext.getApiClientConfig();
            // 获取验签公钥
            String publicKey = apiClientConfig.getPayCompanyPublicKey();
            // 获取待加签字符串
            Map<String, Object> data = apiResponseModel.getResponseMap();
            // 获取待加签字符串
            waitSignStr = SignerUtil.getWaitSignStr(data,requestContext.getNoNeedSignWords(),requestContext.getApiClientConfig().isSingIgnoreNull()
            );
            // 创建加签对象
            Sign sign = new Sign(
                    apiClientConfig.getSignTypeEnum().getAlgorithm(),
                    null,
                    SecureUtil.decode(publicKey));
            // 执行加签操作
            return sign.verify(waitSignStr.getBytes(apiClientConfig.getCharset()), Base64.decode(apiResponseModel.getResponseSign()));
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 响应验签失败 >> apiRequestModel={}, waitSignStr={}", e, requestContext.getClientInfoModel().getClientDesc(), apiResponseModel, waitSignStr);
            throw new FsApiException("请求加签失败", e);
        }
    }
}