/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package
        com.fshows.sdk.core.client.base;

import cn.hutool.core.bean.BeanUtil;
import com.fshows.sdk.core.client.base.definition.IApiDefinition;
import com.fshows.sdk.core.client.base.definition.IRequestDefinition;
import com.fshows.sdk.core.client.base.definition.IResponseDefinition;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.ClientInfoModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.util.ReqIdUtil;
import com.fshows.sdk.core.util.ValidateUtil;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.util.HashMap;

/**
 * 默认api客户端实现
 *
 * @author liluqing
 * @version AbstractSwiftApiClientImpl.java, v 0.1 2022-03-02 11:59
 */
@Setter
@Slf4j
public abstract class AbstractApiClient<REQ extends IRequestDefinition, RES extends IResponseDefinition, API extends IApiDefinition> implements IApiClient<REQ, RES, API> {

    /**
     * 客户端配置
     */
    protected DefaultClientConfigModel apiClientConfig;

    public AbstractApiClient(DefaultClientConfigModel apiClientConfig) throws FsApiException {
        ValidateUtil.validateWithThrow(apiClientConfig);
        this.apiClientConfig = apiClientConfig;
    }

    /**
     * @param request
     * @param iApiDefinition
     * @return {@link RES}
     * @throws FsApiException
     */
    protected RES doExecute(REQ request, API iApiDefinition) throws FsApiException {
        return doExecute(request,iApiDefinition,null);
    }


    /**
     * 执行请求
     *
     * @param request
     * @param iApiDefinition
     * @param merchantConfigModel
     * @return
     * @throws FsApiException
     */
    protected RES doExecute(REQ request, API iApiDefinition, DefaultClientConfigModel merchantConfigModel) throws FsApiException {
        long beginTime = System.currentTimeMillis();
        LogUtil.info(log, "{} >> 执行请求开始 >> iApiDefinition={}, request={}",
                getClientInfo().getClientDesc(), iApiDefinition, request);
        // 构建请求上下文
        DefaultRequestContext requestContext = buildRequestContext(iApiDefinition, request, merchantConfigModel);
        ApiRequestModel apiRequestModel = null;
        ApiResponseModel apiResponseModel = null;
        try {
            // 入参数校验
            checkParam(request, requestContext);
            // 构建请求数据
            apiRequestModel = buildApiRequestModel(request, requestContext);
            // 执行post请求
            apiResponseModel = httpRequest(apiRequestModel, requestContext);
            Object requestBody = ObjectUtils.defaultIfNull(apiRequestModel.getRequestBody(), apiRequestModel.getRequestForm());
            LogUtil.info(log, "{} >> 请求结束 >> url={}, method={}, request={}, response={}, cost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), iApiDefinition, requestBody,
                    apiResponseModel.getResponseBody(), System.currentTimeMillis() - beginTime);
            // 反序列化构建响应结果
            return buildApiResponse(apiResponseModel, apiRequestModel, requestContext);
        } catch (FsApiException e) {
            LogUtil.error(log, "{} >> 请求异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc() ,
                    iApiDefinition, request, apiRequestModel, apiResponseModel);
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc() ,
                    iApiDefinition, request, apiRequestModel, apiResponseModel);
            throw new FsApiException(e.getMessage(), e);
        }
    }

    /**
     * 构建请求上下文
     *
     * @param tradeApiDefinitionEnum
     * @param request
     * @return
     */
    protected DefaultRequestContext buildRequestContext(API tradeApiDefinitionEnum, REQ request, DefaultClientConfigModel merchantConfig) {
        DefaultRequestContext context = new DefaultRequestContext();
        context.setRequestId(ReqIdUtil.getId());
        context.setIApiDefinition(tradeApiDefinitionEnum);
        // 复制默认参数
        DefaultClientConfigModel clientConfig = this.apiClientConfig;
        DefaultClientConfigModel config = BeanUtil.copyProperties(clientConfig, DefaultClientConfigModel.class);
        merchantConfig.setExtendParam(new HashMap<>(clientConfig.getExtendParam()));
        // merchantId
        config.setAppId(StringUtils.isEmpty(merchantConfig.getAppId()) ? clientConfig.getAppId() : merchantConfig.getAppId());
        // 私钥
        config.setFubeiPrivateKey(StringUtils.isEmpty(merchantConfig.getFubeiPrivateKey())? clientConfig.getFubeiPrivateKey(): merchantConfig.getFubeiPrivateKey());
        // 公钥
        config.setPayCompanyPublicKey(StringUtils.isEmpty(merchantConfig.getPayCompanyPublicKey()) ? clientConfig.getPayCompanyPublicKey(): merchantConfig.getPayCompanyPublicKey());
        context.setApiClientConfig(config);

        context.setClientInfoModel(getClientInfo());
        return context;
    }

    /**
     * 构建api请求数据
     *
     * @param context
     * @param request
     * @return
     */
    protected abstract ApiRequestModel buildApiRequestModel(REQ request, DefaultRequestContext context);

    /**
     * 获取客户端信息
     *
     * @return
     */
    protected abstract ClientInfoModel getClientInfo();

    /**
     * 请求参数校验
     *
     * @param bizRequest
     * @return
     */
    protected void checkParam(REQ bizRequest, DefaultRequestContext defaultRequestContext) throws FsApiException {
        ValidateUtil.notNull(bizRequest, "request请求参数不能为空");
        ValidateUtil.notNull(defaultRequestContext.getIApiDefinition(), "request请求方法不能为空");
        boolean checkRequestClass = StringUtils.equals(bizRequest.getClass().getCanonicalName(), defaultRequestContext.getIApiDefinition().getRequestClass().getCanonicalName());
        if (!checkRequestClass) {
            throw new FsApiException("请求参数类型不正确");
        }
        // 当校验开关打开时才开启主动前置参数校验
        if (this.apiClientConfig.isCheckParam()) {
            ValidateUtil.validateWithThrow(bizRequest);
        }
    }

    /**
     * 构建响应结果
     *
     * @param apiRequestModel
     * @param requestContext
     * @throws FsApiException
     */
    protected abstract RES buildApiResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext);

    /**
     * 发送http请求
     */
    protected abstract ApiResponseModel httpRequest(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws IOException;
}