/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package com.fshows.fsframework.web.auth;

import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTVerifier;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.exceptions.JWTCreationException;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.fshows.fsframework.core.constants.StringPool;
import com.fshows.fsframework.core.utils.LogUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.joda.time.DateTime;

import java.util.Date;
import java.util.UUID;

/**
 * @author lixiang
 * @version JwtTokenManager.java, v 0.1 2018-12-11 11:26 lixiang
 */
@Slf4j
public class JwtTokenManager {

    /**
     * 获取token信息
     *
     * @param subject 主题（一般为密钥）
     * @param token token
     * @return
     */
    public static JwtToken getToken(String subject, String token) {
        DecodedJWT jwt = parseToken(subject, token);
        JwtToken jwtToken = new JwtToken();
        jwtToken.setIssuedAt(jwt.getIssuedAt());
        jwtToken.setExpiresAt(jwt.getExpiresAt());
        jwtToken.setUserId(jwt.getAudience().get(0));
        return jwtToken;
    }

    /**
     * 验证token，返回token解码信息
     *
     * @param subject 主题（一般为密钥）
     * @param token   token
     * @return DecodedJWT
     */
    public static DecodedJWT parseToken(String subject, String token) {
        Algorithm algorithm = Algorithm.HMAC512(subject);
        JWTVerifier verifier = JWT.require(algorithm)
                .withIssuer(StringPool.FSHOWS)
                .build();
        return verifier.verify(token);
    }

    /**
     * @param userId  用户ID
     * @param subject 密钥
     * @param minutes 过期时间
     * @return
     */
    public static String generateToken(String userId, String subject, int minutes) {
        try {
            Date date = new Date();
            DateTime dateTime = new DateTime(date);
            Date expiredTime = dateTime.plusMinutes(minutes).toDate();
            Algorithm algorithm = Algorithm.HMAC512(subject);
            return JWT.create()
                    .withIssuer(StringPool.FSHOWS)
                    .withIssuedAt(date)
                    .withExpiresAt(expiredTime)
                    .withJWTId(UUID.randomUUID().toString())
                    .withAudience(userId)
                    .sign(algorithm);
        } catch (JWTCreationException ex) {
            LogUtil.error(log, "generateToken>>生成token异常：{}", ExceptionUtils.getStackTrace(ex));
            return StringPool.EMPTY;
        }
    }

}