/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package com.fshows.fsframework.web.service;


import com.fshows.fsframework.core.BaseParam;
import com.fshows.fsframework.web.bean.ApplicationContextHelper;
import com.fshows.fsframework.web.domain.ApiContainer;
import com.fshows.fsframework.web.domain.ApiDescriptor;
import com.fshows.fsframework.web.exception.ApiInvokeException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Map;

/**
 * API调用服务
 *
 * @author buhao
 * @version ApiGatewayService.java, v 0.1 2018-06-08 14:54 buhao
 */
@Component
public abstract class AbstractApiInvokeService {
    /**
     * API调用类
     */
    @Autowired
    protected IApiClient apiClient;

    @Autowired
    protected ApiContainer apiContainer;

    /**
     * 调用API
     *
     * @param appId
     * @param sign
     * @param apiMethodName
     * @param content
     * @param <P>
     * @param <R>
     * @return
     */
    @SuppressWarnings("unchecked")
    public <P extends BaseParam, R> R invoke(String appId, String sign, String apiMethodName, Map<String, Object> params, String content) throws Throwable {
        // 获得方法信息
        ApiDescriptor apiDescriptor = apiContainer.get(apiMethodName);
        if (null == apiDescriptor) {
            throw ApiInvokeException.API_NOT_EXIST;
        }
        // 获得 bean
        String beanName = apiDescriptor.getBeanName();
        Object bean = ApplicationContextHelper.getBean(beanName);
        if (null == bean) {
            throw ApiInvokeException.API_NOT_EXIST;
        }
        Boolean hasPermission = checkPermission(appId, apiMethodName);
        if (!hasPermission) {
            throw ApiInvokeException.DO_NOT_HAS_PERMISSION;
        }
        Boolean checkSign = checkSign(params, appId);
        if (!checkSign) {
            throw ApiInvokeException.API_INVALID_SIGIN;
        }
        // 调用方法
        return apiClient.invoke(apiMethodName, content, (ApiInvoker<P, R>) (method, param) ->
                (R) apiDescriptor.getMethod().invoke(bean, param));
    }

    /**
     * 调用API
     *
     * @param appId
     * @param sign
     * @param apiMethodName
     * @param content
     * @param <P>
     * @param <R>
     * @return
     */
    @SuppressWarnings("unchecked")
    public <P extends BaseParam, R> R invoke2(String appId, String sign, String apiMethodName, String traceId, Map<String, Object> params, String content) throws Throwable {
        // 获得方法信息
        ApiDescriptor apiDescriptor = apiContainer.get(apiMethodName);
        if (null == apiDescriptor) {
            throw ApiInvokeException.API_NOT_EXIST;
        }
        // 获得 bean
        String beanName = apiDescriptor.getBeanName();
        Object bean = ApplicationContextHelper.getBean(beanName);
        if (null == bean) {
            throw ApiInvokeException.API_NOT_EXIST;
        }
        Boolean hasPermission = checkPermission(appId, apiMethodName);
        if (!hasPermission) {
            throw ApiInvokeException.DO_NOT_HAS_PERMISSION;
        }
        Boolean checkSign = checkSign(params, appId);
        if (!checkSign) {
            throw ApiInvokeException.API_INVALID_SIGIN;
        }
        // 调用方法
        return apiClient.invoke2(apiMethodName, content, traceId, (ApiInvoker<P, R>) (method, param) ->
                (R) apiDescriptor.getMethod().invoke(bean, param));
    }

    /**
     * 权限验证
     *
     * @param appId
     * @param apiMethodName
     * @return
     */
    protected abstract boolean checkPermission(String appId, String apiMethodName);

    /**
     * 签名检查
     *
     * @param params
     * @param appId
     * @return
     */
    protected abstract boolean checkSign(Map<String, Object> params, String appId);

}