/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package com.fshows.fsframework.extend.aliyun.event;

import com.aliyun.openservices.cms.CMSClient;
import com.aliyun.openservices.cms.exception.CMSException;
import com.aliyun.openservices.cms.model.impl.CustomEvent;
import com.aliyun.openservices.cms.request.CustomEventUploadRequest;
import com.aliyun.openservices.cms.response.CustomEventUploadResponse;
import com.fshows.fsframework.core.utils.LogUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.concurrent.BasicThreadFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * @author liujing01
 * @version AliyunEventUtil.java, v 0.1 2018-09-25 14:13
 */
@Slf4j
public class AliyunEventUtil implements Runnable{
    /**
     * oss 外网地址
     */
    private static final String OSS_ENDPOINT_OUT = "http://metrichub-cms-cn-hangzhou.aliyuncs.com";
    /**
     * 初始化queue与Executors
     */
    private LinkedBlockingQueue<EventEntry> eventQueue;
    private ScheduledExecutorService schedule;
    private CMSClient cmsClient;

    private static final String SUCCESS_CODE = "200";

    private static final int MAX_POLL_SIZE = 99;
    /**
     * 使用volatile关键字保其可见性
     */
    private static volatile   AliyunEventUtil instance = null;
    public static AliyunEventUtil getInstance(String accessKey, String secretKey) {
        if(instance==null){
            synchronized (AliyunEventUtil.class){
                if(instance==null){
                    instance = new AliyunEventUtil(accessKey,secretKey);
                }
            }
        }
        return instance;
    }
    private AliyunEventUtil(){}
    private AliyunEventUtil(String accessKey, String secretKey){
        cmsClient = new CMSClient(OSS_ENDPOINT_OUT, accessKey,secretKey);
        eventQueue = new LinkedBlockingQueue<>(10000);
        schedule = new ScheduledThreadPoolExecutor(1,new BasicThreadFactory.Builder().namingPattern("aliyun-event-pool-%d").daemon(true).build());
        schedule.scheduleAtFixedRate(this, 5, 1, TimeUnit.SECONDS);
    }


    /**
     * 上报事件：
     * 每一个事件都包含事件的名称与事件的内容，名称用于识别事件，内容是事件的详细信息，支持全文搜索。
     * @param event
     */
    public boolean put(EventEntry event) {
        if(cmsClient==null){
            return false;
        }
        //这里事件队列满后将直接丢弃，可以根据自己的情况调整这个策略。
        boolean b = eventQueue.offer(event);
        if (!b) {
            LogUtil.warn(log, "事件队列已满，丢弃事件：{}", event);
            return false;
        }
        return true;
    }


    /**
     * 上报事件
     */
    public  boolean uploadEvent(EventEntry event) {
        try{
            if(cmsClient==null){
                return false;
            }
            if(event==null){
                return false;
            }
            CustomEventUploadRequest request = CustomEventUploadRequest.builder()
                    .append(CustomEvent.builder()
                            .setContent(event.getContent())
                            .setGroupId(event.getGroupId())
                            .setName(event.getName()).build())
                    .build();
            CustomEventUploadResponse response = cmsClient.putCustomEvent(request);
            if(response==null){
                LogUtil.error(log, "AliyunEventUtil ---- >> uploadEvent 上传异常, 请求参数：{}", event.toString());
                return false;
            }
            if(!SUCCESS_CODE.equals(response.getCode())){
                LogUtil.error(log, "AliyunEventUtil ---- >> uploadEvent 上传异常,请求参数：{}, 返回结果：requestId={}, code = {},message={}", event.toString(), response.getRequestId(), response.getCode(), response.getMessage());
                return false;
            }
            return true;
        }catch (CMSException e) {
            LogUtil.error(log, "AliyunEventUtil ---- >> uploadEvent 上传异常, 参数: param = {}, 异常: Ex = {}",event.getContent(),e);
            return false;
        }
    }


    /**
     * 批量上报事件
     */
    public  boolean uploadEventBatch(List<EventEntry> events) {
        if(cmsClient==null){
            return false;
        }
        List<CustomEvent> eventList = new ArrayList<>();
        for(EventEntry event: events){
            eventList.add(CustomEvent.builder()
                    .setContent(event.getContent())
                    .setGroupId(event.getGroupId())
                    .setName(event.getName()).build());
        }
        return uploadEventBatch0(eventList);
    }

    private boolean uploadEventBatch0(List<CustomEvent> eventList){
        try {
            if(eventList==null || eventList.isEmpty()){
                return false;
            }
            CustomEventUploadRequest request = CustomEventUploadRequest.builder()
                    .setEventList(eventList).build();
            CustomEventUploadResponse response = cmsClient.putCustomEvent(request);
            if (response == null) {
                return false;
            }
            if (!SUCCESS_CODE.equals(response.getCode())) {
                LogUtil.error(log, "AliyunEventUtil ---- >> uploadEventBatch 上传异常,返回结果：requestId={}, code = {},message={}", response.getRequestId(), response.getCode(), response.getMessage());
                return false;
            }
            return true;
        } catch (Exception e) {
            LogUtil.error(log, "AliyunEventUtil ---- >> uploadEvent 上传异常: Ex = {}", e);
            return false;
        }
    }

    @Override
    public void run() {
        do {
            batchPut();
        } while (this.eventQueue.size() > 10);
    }

    private void batchPut() {
        /**
         *  从队列中取出99条事件，用于批量上报
         */
        List<CustomEvent> events = new ArrayList<>();
        for (int i = 0; i < MAX_POLL_SIZE; i++) {
            EventEntry e = this.eventQueue.poll();
            if (e == null) {
                break;
            }
            events.add(CustomEvent.builder().setContent(e.getContent()).setName(e.getName()).setGroupId(e.getGroupId()).build());
        }
        if (events.isEmpty()) {
            return;
        }
        uploadEventBatch0(events);
    }
}
