/**
 * fshows.com
 * Copyright (C) 2013-2025 All Rights Reserved.
 */
package com.fshows.fsframework.extend.aliyun.mq.util;

import com.aliyun.openservices.ons.api.PropertyKeyConst;
import org.apache.commons.lang3.StringUtils;

import java.util.Properties;

/**
 * MQ实例工具类
 * 提供MQ实例相关的工具方法
 *
 * @author liluqing
 * @version MqInstanceUtil.java, v 0.1 2025-01-10 15:30
 */
public class MqInstanceUtil {

    /**
     * 生成Producer实例的唯一标识
     *
     * @param properties MQ配置
     * @param instance   实例对象
     * @return 实例标识
     */
    public static String generateProducerInstanceKey(Properties properties, Object instance) {
        StringBuilder keyBuilder = new StringBuilder("producer");

        // 添加ProducerId
        String producerId = properties.getProperty(PropertyKeyConst.ProducerId);
        if (StringUtils.isBlank(producerId)) {
            producerId = properties.getProperty(PropertyKeyConst.GROUP_ID);
        }

        if (StringUtils.isNotBlank(producerId)) {
            keyBuilder.append("_").append(producerId);
        }
        
        // 添加实例hashCode
        keyBuilder.append("_").append(System.identityHashCode(instance));
        
        // 添加时间戳
        keyBuilder.append("_").append(System.currentTimeMillis());
        
        return keyBuilder.toString();
    }

    /**
     * 生成Consumer实例的唯一标识
     *
     * @param properties MQ配置
     * @param instance   实例对象
     * @return 实例标识
     */
    public static String generateConsumerInstanceKey(Properties properties, Object instance) {
        StringBuilder keyBuilder = new StringBuilder("consumer");
        
        // 添加ConsumerId
        String consumerId = properties.getProperty(PropertyKeyConst.ConsumerId);
        if (StringUtils.isBlank(consumerId)) {
            consumerId = properties.getProperty(PropertyKeyConst.GROUP_ID);
        }
        if (StringUtils.isNotBlank(consumerId)) {
            keyBuilder.append("_").append(consumerId);
        }
        
        // 添加实例hashCode
        keyBuilder.append("_").append(System.identityHashCode(instance));
        
        // 添加时间戳
        keyBuilder.append("_").append(System.currentTimeMillis());
        
        return keyBuilder.toString();
    }

    /**
     * 检查Properties是否包含必要的MQ配置
     *
     * @param properties MQ配置
     * @return 是否有效
     */
    public static boolean isValidMqProperties(Properties properties) {
        if (properties == null) {
            return false;
        }
        
        // 检查必要的配置项
        String accessKey = properties.getProperty(PropertyKeyConst.AccessKey);
        String secretKey = properties.getProperty(PropertyKeyConst.SecretKey);
        String nameServer = properties.getProperty(PropertyKeyConst.NAMESRV_ADDR);
        
        return StringUtils.isNotBlank(accessKey) && 
               StringUtils.isNotBlank(secretKey) && 
               StringUtils.isNotBlank(nameServer);
    }

    /**
     * 复制Properties
     *
     * @param source 源Properties
     * @return 复制的Properties
     */
    public static Properties copyProperties(Properties source) {
        if (source == null) {
            return new Properties();
        }
        
        Properties target = new Properties();
        target.putAll(source);
        return target;
    }

    /**
     * 合并Properties，target中的配置会覆盖source中的同名配置
     *
     * @param source 源Properties
     * @param target 目标Properties
     * @return 合并后的Properties
     */
    public static Properties mergeProperties(Properties source, Properties target) {
        Properties merged = copyProperties(source);
        if (target != null) {
            merged.putAll(target);
        }
        return merged;
    }

    /**
     * 获取实例类型描述
     *
     * @param instanceKey 实例标识
     * @return 实例类型
     */
    public static String getInstanceType(String instanceKey) {
        if (StringUtils.isBlank(instanceKey)) {
            return "unknown";
        }
        
        if (instanceKey.startsWith("producer")) {
            return "Producer";
        } else if (instanceKey.startsWith("consumer")) {
            return "Consumer";
        } else {
            return "unknown";
        }
    }

    /**
     * 从实例标识中提取ID
     *
     * @param instanceKey 实例标识
     * @return ID部分
     */
    public static String extractIdFromInstanceKey(String instanceKey) {
        if (StringUtils.isBlank(instanceKey)) {
            return "";
        }
        
        String[] parts = instanceKey.split("_");
        if (parts.length >= 2) {
            return parts[1];
        }
        
        return "";
    }
}
