/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package
        com.fshows.fsframework.extend.idgen.worker;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import com.fshows.fsframework.extend.idgen.exception.WorkerIdAssignException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.Optional;
import java.util.Set;

/**
 * 基于静态配置的工作ID生成器
 *
 * @author liluqing
 * @version ConfigWorkerIdAssigner.java, v 0.1 2022-12-12 10:06
 */
@Slf4j
public class ConfigWorkerIdAssigner implements WorkerIdAssigner{

    /**
     * 机器ID配置
     * 键为IP,值为机器ID
     * 示例格式：{"127.0.0.1": 0, "192.168.3.1": 1}
     */
    private String config;

    /**
     * 本机IP
     */
    private Set<String> nativeIpSet;

    /**
     * 机器ID
     */
    private long workerId;

    public ConfigWorkerIdAssigner(String config, Set<String> nativeIpSet) {
        this.config = config;
        this.nativeIpSet = nativeIpSet;
        this.workerId = getWorkerId(config, nativeIpSet);
    }

    /**
     * 从配置中解析机器ID
     *
     * @param config
     * @param nativeIpSet
     * @return
     */
    private long getWorkerId(String config, Set<String> nativeIpSet) {
        log.info("ConfigWorkerIdAssigner >> workerId配置解析器 >> nativeIpSet={}, config={}", nativeIpSet, config);
        if (StringUtils.isBlank(config)) {
            throw new WorkerIdAssignException("workerId生成器-机器ID配置-不能为空");
        }
        if (CollectionUtil.isEmpty(nativeIpSet)) {
            throw new WorkerIdAssignException("workerId生成器-本机IP-不能为空");
        }
        JSONObject workerConfigModel = JSONObject.parseObject(config);
        // 根据本机IP从配置中匹配机器ID
        Optional<Long> workerIdOpt = nativeIpSet.stream()
                .filter(workerConfigModel::containsKey)
                .map(workerConfigModel::getLong)
                .findFirst();
        if (!workerIdOpt.isPresent()) {
            log.error("ConfigWorkerIdAssigner >> workerId配置解析异常 >> 配置中未解析到当前IP对应的机器ID >> nativeIpSet={}, config={}", nativeIpSet, config);
            throw new WorkerIdAssignException("workerId配置解析异常,配置中未解析到当前IP对应的机器ID,请检查机器ID配置");
        }
        return workerIdOpt.get();
    }

    @Override
    public long assignWorkerId() {
        return workerId;
    }
}