/**
 * fshows.com
 * Copyright (C) 2013-2025 All Rights Reserved.
 */
package com.fshows.fsframework.extend.aliyun.mq.core;

import com.ctrip.framework.apollo.model.ConfigChange;
import com.ctrip.framework.apollo.model.ConfigChangeEvent;
import com.ctrip.framework.apollo.spring.annotation.ApolloConfigChangeListener;
import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.extend.aliyun.mq.config.FsMqConfig;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Set;
import java.util.concurrent.CompletableFuture;

/**
 * Apollo配置监听器
 * 监听MQ相关配置变更，触发实例更新
 *
 * @author liluqing
 * @version ApolloConfigListener.java, v 0.1 2025-01-10 15:30
 */
@Component
@Slf4j
public class ApolloMqConfigListener {

    @Autowired
    private FsMqInstanceManager instanceManager;

    @Autowired
    private FsMqConfig fsMqConfig;

    /**
     * 监听Apollo配置变更
     * 当检测到MQ相关配置变更时，异步更新所有MQ实例
     *
     * @param changeEvent 配置变更事件
     */
    @ApolloConfigChangeListener
    public void onConfigChange(ConfigChangeEvent changeEvent) {
        // 检查是否启用动态更新
        if (!fsMqConfig.getMqClientDynamicUpdate()) {
            LogUtil.debug(log, "MQ客户端动态更新未启用，忽略配置变更");
            return;
        }

        Set<String> watchKeys = fsMqConfig.getWatchKeys();
        boolean needUpdate = false;

        // 检查是否有需要监听的配置发生变更
        for (String changeKey : changeEvent.changedKeys()) {
            ConfigChange change = changeEvent.getChange(changeKey);
            if (watchKeys.contains(change.getPropertyName())) {
                LogUtil.info(log, "检测到MQ相关配置变更: {} = {} -> {}",
                    change.getPropertyName(), change.getOldValue(), change.getNewValue());
                needUpdate = true;
            }
        }

        if (needUpdate) {
            LogUtil.info(log, "开始更新MQ实例...");

            // 异步更新，避免阻塞Apollo配置更新
            CompletableFuture.runAsync(() -> {
                try {
                    // 先更新Producer（无损策略）
                    instanceManager.updateAllProducers();

                    // 再更新Consumer（随机延迟重启策略）
                    instanceManager.updateAllConsumers();

                    LogUtil.info(log, "MQ实例更新完成，Producer: {}个, Consumer: {}个",
                        instanceManager.getProducerCount(), instanceManager.getConsumerCount());
                } catch (Exception e) {
                    LogUtil.error(log, "MQ实例更新失败", e);
                }
            }).exceptionally(throwable -> {
                LogUtil.error(log, "MQ实例更新异步任务执行失败", throwable);
                return null;
            });
        }
    }
}
