/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package com.fshows.fsframework.extend.apollo.listener;

import com.ctrip.framework.apollo.enums.PropertyChangeType;
import com.ctrip.framework.apollo.model.ConfigChange;
import com.ctrip.framework.apollo.model.ConfigChangeEvent;
import com.ctrip.framework.apollo.spring.annotation.ApolloConfigChangeListener;
import com.fshows.fsframework.core.utils.LogUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.core.config.Configurator;

import java.util.Set;
import java.util.stream.Collectors;

/**
 * 1、日志级别变更，只有变更时有效，启动时还是以log4j2.xml 中的级别为准
 * 2、约定规则
 * a. 根日志级别的 key 必须为 logger.level
 * b. 其它日志的 key 为 logger.level.{loggerName}
 * 3、不处理删除变更
 *
 * @author buhao
 * @version ConfigChannge.java, v 0.1 2018-09-14 20:35 buhao
 */
@Slf4j
public class LoggerLevelConfigChangeListener {

    /**
     * 日志级别配置前缀
     */
    private static final String LOGGER_LEVEL_PREFIX = "logger.level";

    /**
     * 日志级别变更监控
     *
     * @param changeEvent
     */
    @ApolloConfigChangeListener
    public void loggerLevelChange(ConfigChangeEvent changeEvent) {

        // 获得本次修改的 key 的集合
        Set<String> changeKeySet = changeEvent.changedKeys();

        // 获得与日志级别相关的 key
        Set<String> loggerChangeKeySet = changeKeySet.stream().filter(it -> it.startsWith(LOGGER_LEVEL_PREFIX)).collect(Collectors.toSet());

        // 修改对应的日志级别
        for (String key : loggerChangeKeySet) {
            // 获得变更内容
            ConfigChange change = changeEvent.getChange(key);
            // 变更对象不为空，并且不是删除操作
            if (change != null && !change.getChangeType().equals(PropertyChangeType.DELETED)) {
                // 获得新的级别
                String newValue = change.getNewValue();
                // 获得老的级别
                String oldValue = change.getOldValue();
                // 如果级别有变更
                if (!newValue.equals(oldValue)) {
                    String loggerName = "Root";
                    Level level = Level.getLevel(newValue.toUpperCase());
                    // 如果是日志根级别
                    if (key.equals(LOGGER_LEVEL_PREFIX)) {
                        Configurator.setRootLevel(level);
                    } else {
                        loggerName = key.replace(LOGGER_LEVEL_PREFIX + ".", "");
                        Configurator.setLevel(loggerName, level);
                    }
                    LogUtil.info(log, "{} 日志级别变更 {} to {}", loggerName, oldValue, newValue);
                }
            }

        }
    }
}