/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package com.fshows.fsframework.extend.dubbo.filter;

import com.alibaba.dubbo.common.Constants;
import com.alibaba.dubbo.common.extension.Activate;
import com.alibaba.dubbo.rpc.Filter;
import com.alibaba.dubbo.rpc.Invocation;
import com.alibaba.dubbo.rpc.Invoker;
import com.alibaba.dubbo.rpc.Result;
import com.alibaba.dubbo.rpc.RpcContext;
import com.alibaba.dubbo.rpc.RpcException;
import com.alibaba.dubbo.rpc.RpcResult;
import com.fshows.fsframework.common.annotation.NoGlobalLog;
import com.fshows.fsframework.common.exception.BaseException;
import com.fshows.fsframework.common.exception.CommonException;
import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.core.utils.SystemClock;
import com.fshows.fsframework.core.utils.TraceIdGenerator;
import com.fshows.fsframework.extend.util.SpringContextUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.MDC;
import org.springframework.core.env.Environment;

import java.lang.reflect.Method;

/**
 * dubbo 全局 trace 过滤器
 *
 * @author buhao
 * @version GlobalTraceFilter.java, v 0.1 2018-09-15 16:35 buhao
 */
@Slf4j
@Activate(group = {Constants.PROVIDER, Constants.CONSUMER}, order = -9999)
public class GlobalTraceFilter implements Filter {
    /**
     * 服务名称
     */
    public static final String DUBBO_APPLICATION_NAME = "dubbo.application.name";

    public static final String TRACE_ID = "TRACE_ID";
    public static final String SOURCE_GATEWAY = "SOURCE_GATEWAY";

    /**
     * do invoke filter.
     * <p>
     * <code>
     * // before filter
     * Result result = invoker.invoke(invocation);
     * // after filter
     * return result;
     * </code>
     *
     * @param invoker    service
     * @param invocation invocation.
     * @return invoke result.
     * @throws RpcException
     * @see Invoker#invoke(Invocation)
     */
    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {
        // 获得 mdc 上下文的 tradeId
        String mdcTraceId = invocation.getAttachment(TRACE_ID);
        // 获得来源网关
        String sourceGateway = invocation.getAttachment(SOURCE_GATEWAY);

        boolean isProvider = RpcContext.getContext().isProviderSide();
        if (isProvider) {
            if (StringUtils.isBlank(mdcTraceId)) {
                mdcTraceId = TraceIdGenerator.generate();
            }
            // 把 traceId 放入日志的上下文中
            MDC.put(TRACE_ID, mdcTraceId);
            RpcContext.getContext().setAttachment(TRACE_ID, mdcTraceId);
        }
        if (!isProvider) {
            if (StringUtils.isBlank(mdcTraceId)) {
                mdcTraceId = MDC.get(TRACE_ID);
            }
            if (StringUtils.isBlank(mdcTraceId)) {
                mdcTraceId = TraceIdGenerator.generate();
            }
            RpcContext.getContext().setAttachment(TRACE_ID, mdcTraceId);

            if (StringUtils.isBlank(sourceGateway)) {
                sourceGateway = MDC.get(SOURCE_GATEWAY);
            }
            if (StringUtils.isBlank(sourceGateway)) {
                Environment environment = SpringContextUtil.getEnvironment();
                if (environment != null) {
                    sourceGateway = environment.getProperty(DUBBO_APPLICATION_NAME);
                }
            }
        }
        if (StringUtils.isNotBlank(sourceGateway)) {
            if (isProvider) {
                MDC.put(SOURCE_GATEWAY, sourceGateway);
            }
            RpcContext.getContext().setAttachment(SOURCE_GATEWAY, sourceGateway);
        }

        // 获得 RPC 方法名
        String methodName = invoker.getUrl().getPath();
        // 判断是否记录调用日志
        boolean isLog = false;
        try {
            Method method = invoker.getInterface().getMethod(invocation.getMethodName(), invocation.getParameterTypes());
            NoGlobalLog noGlobalLog = method.getAnnotation(NoGlobalLog.class);
            isLog = noGlobalLog == null;
            methodName += "." + method.getName();
        } catch (NoSuchMethodException e) {
            LogUtil.error(log, "方法: invoke 发生异常， 参数: invoker = {}, invocation = {} ,异常: Ex = {}", invoker, invocation, e);
        }


        // 获得开始时间
        long startTime = SystemClock.millisClock().now();

        // 打印调用前日志
        if (isLog) {
            // 获得参数
            Object[] arguments = invocation.getArguments();

            LogUtil.debug(log, "RPC 接口开始 methodName = {}, agruments = {}", methodName, arguments);
        }

        // 调用接口
        Result result = invoker.invoke(invocation);

        // 打印调用后日志
        if (isLog) {
            // 抛出的异常
            Throwable exception = result.getException();
            // 返回结果
            Object value = result.getValue();
            // 打印结束日志
            if (exception != null) {
                LogUtil.error(log, "RPC 接口异常结束 methodName = {}, time = {}ms ", exception, methodName, SystemClock.millisClock().now() - startTime);
                // 如果不是服务自定义异常，就返回系统错误
                if (!(exception instanceof BaseException)) {
                    LogUtil.error(log, "RPC 接口异常结束 methodName = {},  exception = {}, time = {}ms ", methodName, exception, SystemClock.millisClock().now() - startTime);
                    if (isProvider) {
                        // 清空 MDC
                        MDC.remove(TRACE_ID);
                        MDC.remove(SOURCE_GATEWAY);
                    }
                    return new RpcResult(CommonException.SYSTEM_ERROR);
                }
            } else {
                LogUtil.debug(log, "RPC 接口结束 methodName = {},  result ={}, time = {}ms ", methodName, value, SystemClock.millisClock().now() - startTime);
            }
        }
        if (isProvider) {
            // 清空 MDC
            MDC.remove(TRACE_ID);
            MDC.remove(SOURCE_GATEWAY);
        }
        return result;
    }
}