/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package com.fshows.fsframework.extend.dubbo.filter;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.dubbo.common.Constants;
import com.alibaba.dubbo.common.extension.Activate;
import com.alibaba.dubbo.rpc.Filter;
import com.alibaba.dubbo.rpc.Invocation;
import com.alibaba.dubbo.rpc.Invoker;
import com.alibaba.dubbo.rpc.Result;
import com.alibaba.dubbo.rpc.RpcException;
import com.alibaba.fastjson.JSONObject;
import com.fshows.fsframework.core.constants.StringPool;
import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.core.utils.SystemClock;
import com.fshows.fsframework.extend.dubbo.model.DubboMethodMonitorConfigConvertModel;
import com.fshows.fsframework.extend.dubbo.model.DubboMethodMonitorConfigModel;
import com.fshows.fsframework.extend.util.SpringContextUtil;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.env.Environment;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;

/**
 * dubbo慢方法监控
 *
 * @author zhaoxumin
 * @version GlobalDubboMethodMonitor.java, v 0.1 2023-05-31 10:42 zhaoxumin
 */
@Slf4j
@Activate(group = {Constants.PROVIDER}, order = -9999)
public class GlobalDubboMethodMonitor implements Filter {

    /**
     * 来源网关
     */
    public static final String SOURCE_GATEWAY = "SOURCE_GATEWAY";

    /**
     * dubbo方法监控开关. 0-关闭 1-开启 apollo中配置
     */
    private final static String DUBBO_METHOD_MONITOR_SWITCH = "dubbo.method.monitor.switch";

    /**
     * dubbo方法监控参数配置. apollo中配置
     */
    private final static String DUBBO_METHOD_MONITOR_CONFIG = "dubbo.method.monitor.config";

    private static volatile DubboMethodMonitorConfigConvertModel configModel = null;

    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {
        Environment environment = SpringContextUtil.getEnvironment();
        if (environment != null) {
            if (StringPool.ONE.equals(environment.getProperty(DUBBO_METHOD_MONITOR_SWITCH))) {
                DubboMethodMonitorConfigConvertModel methodMonitorConfigModel = null;
                try {
                    methodMonitorConfigModel = getMethodMonitorConfigModel();
                } catch (Throwable e) {
                    LogUtil.error(log, "record bad-facade info get monitor config error >> dubbo慢调用监控，加载配置参数异常", e);
                }

                if (methodMonitorConfigModel != null) {
                    Map<String, List<DubboMethodMonitorConfigModel.MethodMonitorLevelConfigModel>> methodMonitorConfigMap = methodMonitorConfigModel.getMethodMonitorConfigMap();
                    if (CollUtil.isNotEmpty(methodMonitorConfigMap)) {
                        // 获得 RPC 方法名
                        String methodName = invoker.getUrl().getPath() + "." + invocation.getMethodName();
                        List<DubboMethodMonitorConfigModel.MethodMonitorLevelConfigModel> methodMonitorLevelConfigModels = methodMonitorConfigMap.get(methodName);
                        if (CollUtil.isEmpty(methodMonitorLevelConfigModels)) {
                            // 为空，则按默认配置处理
                            methodMonitorLevelConfigModels = methodMonitorConfigMap.get("default");
                        }

                        if (CollUtil.isNotEmpty(methodMonitorLevelConfigModels)) {
                            // 获得开始时间
                            long rpcStartTime = SystemClock.millisClock().now();
                            // 调用接口
                            Result result = invoker.invoke(invocation);
                            // 获得结束时间
                            long rpcEndTime = SystemClock.millisClock().now();
                            // 计算耗时
                            long timeConsuming = rpcEndTime - rpcStartTime;

                            Object[] arguments = invocation.getArguments();
                            for (DubboMethodMonitorConfigModel.MethodMonitorLevelConfigModel methodMonitorLevelConfigModel : methodMonitorLevelConfigModels) {
                                if (timeConsuming > methodMonitorLevelConfigModel.getLowerThreshold()) {
                                    LogUtil.warn(log, "record bad-facade info >> timeConsumingLevel={}, methodName={}, arguments={}, time={}ms, sourceGateway={}",
                                            methodMonitorLevelConfigModel.getLevel(), methodName, arguments, timeConsuming, invocation.getAttachment(SOURCE_GATEWAY));
                                    break;
                                }
                            }
                            return result;
                        }
                    }
                }
            } else {
                configModel = null;
            }
        }
        return invoker.invoke(invocation);
    }

    /**
     * 获取方法监控配置
     *
     * @return
     */
    private DubboMethodMonitorConfigConvertModel getMethodMonitorConfigModel() {
        if (configModel == null) {
            synchronized (this) {
                if (configModel == null) {
                    Environment environment = SpringContextUtil.getEnvironment();
                    if (environment == null) {
                        return null;
                    }

                    DubboMethodMonitorConfigConvertModel methodMonitorConfigConvertModel = new DubboMethodMonitorConfigConvertModel();
                    // 解析dubbo方法监控点参数配置
                    String methodMonitorConfigStr = environment.getProperty(DUBBO_METHOD_MONITOR_CONFIG);
                    if (StrUtil.isNotBlank(methodMonitorConfigStr)) {
                        ConcurrentMap<String, List<DubboMethodMonitorConfigModel.MethodMonitorLevelConfigModel>> methodMonitorConfigMap = Maps.newConcurrentMap();
                        DubboMethodMonitorConfigModel config = JSONObject.parseObject(methodMonitorConfigStr, DubboMethodMonitorConfigModel.class);
                        if (config != null) {
                            List<DubboMethodMonitorConfigModel.MethodMonitorPointConfigModel> methodMonitorPointConfigList = config.getMethodMonitorPointConfigList();
                            if (CollUtil.isNotEmpty(methodMonitorPointConfigList)) {
                                for (DubboMethodMonitorConfigModel.MethodMonitorPointConfigModel methodMonitorPointConfigModel : methodMonitorPointConfigList) {
                                    List<DubboMethodMonitorConfigModel.MethodMonitorLevelConfigModel> methodMonitorLevelConfig = methodMonitorPointConfigModel.getMethodMonitorLevelConfigList().stream()
                                            .sorted(Comparator.comparingInt(DubboMethodMonitorConfigModel.MethodMonitorLevelConfigModel::getLowerThreshold).reversed())
                                            .collect(Collectors.toList());
                                    for (String singleMethodName : methodMonitorPointConfigModel.getMethodName().split(StringPool.COMMA)) {
                                        methodMonitorConfigMap.put(singleMethodName, methodMonitorLevelConfig);
                                    }
                                }
                            }
                        }
                        methodMonitorConfigConvertModel.setMethodMonitorConfigMap(methodMonitorConfigMap);
                    }
                    configModel = methodMonitorConfigConvertModel;
                }
            }
        }
        return configModel;
    }
}