/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.fsframework.extend.extension;

import lombok.Data;

import java.io.Serializable;

/**
 * BusinessScenario(业务场景) = businessId（业务id） + useCase（用例） + scenario（场景）
 * 标识唯一的用户业务场景，注册、查找扩展点会用到
 *
 * @author liangyuanping
 * @version BusinessScenario.java, v 0.1 2022-07-16 4:19 下午 liangyuanping
 */
@Data
public class BusinessScenario implements Serializable {

    private static final long serialVersionUID = -5440151557874573533L;

    public final static String DEFAULT_BUSINESS_ID = "defaultBusinessId";
    public final static String DEFAULT_USE_CASE = "defaultUseCase";
    public final static String DEFAULT_SCENARIO = "defaultScenario";
    private final static String DOT_SEPARATOR = ".";

    /**
     * businessId 用于识别业务，如果只有一个业务，则可以为空。例如：tmall（天猫）、
     */
    private String businessId;
    /**
     * 用例 用于标识一个场景用例，不能为null。例如："place order（下单）"
     */
    private String useCase;
    /**
     * 场景 用于标识一个用例的场景，不能为null。例如"88vip"，"normal"
     */
    private String scenario;

    public BusinessScenario() {
        this.businessId = DEFAULT_BUSINESS_ID;
        this.useCase = DEFAULT_USE_CASE;
        this.scenario = DEFAULT_SCENARIO;
    }

    public BusinessScenario(String businessId, String useCase, String scenario) {
        this.businessId = businessId;
        this.useCase = useCase;
        this.scenario = scenario;
    }

    /**
     * 构造用例场景
     *
     * @param useCase  用例
     * @param scenario 场景
     */
    public static BusinessScenario valueOf(String useCase, String scenario) {
        return new BusinessScenario(useCase, scenario);
    }

    public BusinessScenario(String useCase, String scenario) {
        this(DEFAULT_BUSINESS_ID, useCase, scenario);
    }

    public BusinessScenario(String scenario) {
        this(DEFAULT_BUSINESS_ID, DEFAULT_USE_CASE, scenario);
    }

    /**
     * 构建业务场景
     *
     * @param businessId
     * @param useCase
     * @param scenario
     * @return
     */
    public static BusinessScenario valueOf(String businessId, String useCase, String scenario) {
        return new BusinessScenario(businessId, useCase, scenario);
    }

    /**
     * 获取唯一用户业务场景身份
     * <p>
     * 例如：用户业务场景身份可以提供"tmall.placeOrder.normal"场景以外（"tmall.placeOrder.88vip"）的扩展处理。
     *
     * @return
     */
    public String getUniqueIdentity() {
        return businessId + DOT_SEPARATOR + useCase + DOT_SEPARATOR + scenario;
    }

    @Override
    public String toString() {
        return "BusinessScenario{" +
                "businessId='" + businessId + '\'' +
                ", useCase='" + useCase + '\'' +
                ", scenario='" + scenario + '\'' +
                '}';
    }
}