/*
 * ailike.com
 * Copyright (C) 2022-2024 All Rights Reserved.
 */
package com.fshows.fsframework.extend.dubbo.filter;

import com.alibaba.dubbo.common.Constants;
import com.alibaba.dubbo.common.extension.Activate;
import com.alibaba.dubbo.common.extension.ExtensionLoader;
import com.alibaba.dubbo.common.store.DataStore;
import com.alibaba.dubbo.rpc.*;
import com.fshows.fsframework.core.utils.LogUtil;
import lombok.extern.slf4j.Slf4j;

import java.math.BigDecimal;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * dubbo线程池超过阈值
 * 接近阈值的时候，我们需要告警出来，第一个是可以提前知道线程池是否快要满了，第二个是以这个为依据看是否要调大线程池数量。
 * @author zhangling
 * @version GlobalDubboThreadMonitorFilter.java, v 0.1 2024-07-24 16:32 zhangling
 */
@Slf4j
@Activate(group = {Constants.PROVIDER, Constants.CONSUMER}, order = -9999)
public class GlobalDubboThreadMonitorFilter implements Filter {

    public static final String DUBBO_THREADPOOL = "DUBBO.ThreadPoolExecutor";

    /**
     * 阈值：超过50%就告警
     */
    private static int[] thresholdList = new int[] {50, 55, 60, 65, 70, 75, 80, 85, 90, 95};

    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {
        try {
            DataStore dataStore = ExtensionLoader.getExtensionLoader(DataStore.class).getDefaultExtension();
            Map<String, Object> executors = dataStore.get(Constants.EXECUTOR_SERVICE_COMPONENT_KEY);
            for (Map.Entry<String, Object> entry : executors.entrySet()) {
                ExecutorService executor = (ExecutorService) entry.getValue();

                if (executor != null && executor instanceof ThreadPoolExecutor) {
                    ThreadPoolExecutor tp = (ThreadPoolExecutor) executor;
                    StringBuffer sb = new StringBuffer();
                    sb.append("activePoolSize: ").append(tp.getActiveCount()).append("/n");
                    sb.append("maxPoolSize: ").append(tp.getMaximumPoolSize()).append("/n");
                    sb.append("corePoolSize: ").append(tp.getCorePoolSize()).append("/n");
                    sb.append("completedTask: ").append(tp.getCompletedTaskCount()).append("/n");
                    sb.append("totalTaskCount: ").append(tp.getTaskCount());

                    LogUtil.info(log, "GlobalDubboThreadMonitorFilter.invoke ip = {}, executor info {}", RpcContext.getContext().getMethodName(), sb.toString());
                    double threshold = new BigDecimal((float)tp.getActiveCount()/tp.getMaximumPoolSize() * 100).setScale(2, BigDecimal.ROUND_HALF_UP).doubleValue();
                    for (int i=0;i<thresholdList.length;i++) {
                        //和阈值比较
                        if (threshold >= thresholdList[i]) {
                            LogUtil.warn(log, "GlobalDubboThreadMonitorFilter.invoke" + "EXCEED_"+thresholdList[i]+"{}%", sb.toString());
                        }
                    }
                }
            }
        } catch(Exception e) {
            LogUtil.error(log, "GlobalDubboThreadMonitorFilter.invoke", e);
        }
        return invoker.invoke(invocation);
    }
}