/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package com.fshows.fsframework.extend.dubbo.filter;

import com.alibaba.dubbo.common.Constants;
import com.alibaba.dubbo.common.extension.Activate;
import com.alibaba.dubbo.common.logger.Logger;
import com.alibaba.dubbo.common.logger.LoggerFactory;
import com.alibaba.dubbo.common.utils.ReflectUtils;
import com.alibaba.dubbo.common.utils.StringUtils;
import com.alibaba.dubbo.rpc.Filter;
import com.alibaba.dubbo.rpc.Invocation;
import com.alibaba.dubbo.rpc.Invoker;
import com.alibaba.dubbo.rpc.Result;
import com.alibaba.dubbo.rpc.RpcContext;
import com.alibaba.dubbo.rpc.RpcException;
import com.alibaba.dubbo.rpc.RpcResult;
import com.alibaba.dubbo.rpc.filter.ExceptionFilter;
import com.alibaba.dubbo.rpc.service.GenericService;
import com.fshows.fsframework.common.exception.BaseException;

import java.lang.reflect.Method;

/**
 * Dubbo 对于异常的处理有自己的一套规则：
 * <p>
 * 1、如果是checked异常则直接抛出
 * 2、如果是unchecked异常但是在接口上有声明，也会直接抛出
 * 3、如果异常类和接口类在同一jar包里，直接抛出
 * 5、如果是JDK自带的异常，直接抛出
 * 6、如果是Dubbo的异常，直接抛出
 * 7、其余的都包装成RuntimeException然后抛出（避免异常在Client出不能反序列化问题）
 * <p>
 * 针对第7点，添加一个全局异常类的判断，如果是可以通过
 *
 * @author buhao
 * @version GlobalExceptionFilter.java, v 0.1 2018-10-05 12:57 buhao
 */
@Activate(group = Constants.PROVIDER)
public class GlobalExceptionFilter implements Filter {

    private final Logger logger;

    public GlobalExceptionFilter() {
        this(LoggerFactory.getLogger(ExceptionFilter.class));
    }

    public GlobalExceptionFilter(Logger logger) {
        this.logger = logger;
    }

    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) {
        try {
            Result result = invoker.invoke(invocation);
            if (result.hasException() && GenericService.class != invoker.getInterface()) {
                try {
                    Throwable exception = result.getException();

                    // directly throw if it's checked exception
                    if (!(exception instanceof RuntimeException) && (exception instanceof Exception)) {
                        return result;
                    }
                    // directly throw if the exception appears in the signature
                    try {
                        Method method = invoker.getInterface().getMethod(invocation.getMethodName(), invocation.getParameterTypes());
                        Class<?>[] exceptionClassses = method.getExceptionTypes();
                        for (Class<?> exceptionClass : exceptionClassses) {
                            if (exception.getClass().equals(exceptionClass)) {
                                return result;
                            }
                        }
                    } catch (NoSuchMethodException e) {
                        return result;
                    }

                    // for the exception not found in method's signature, print ERROR message in server's log.
                    logger.error("Got unchecked and undeclared exception which called by " + getErrorString(invoker, invocation, exception), exception);

                    // directly throw if exception class and interface class are in the same jar file.
                    String serviceFile = ReflectUtils.getCodeBase(invoker.getInterface());
                    String exceptionFile = ReflectUtils.getCodeBase(exception.getClass());
                    String className = exception.getClass().getName();
                    if (serviceFile == null || exceptionFile == null || serviceFile.equals(exceptionFile) || exception instanceof BaseException) {
                        return result;
                    }
                    // directly throw if it's JDK exception
                    if (className.startsWith("java.") || className.startsWith("javax.")) {
                        return result;
                    }
                    // directly throw if it's dubbo exception
                    if (exception instanceof RpcException) {
                        return result;
                    }

                    // otherwise, wrap with RuntimeException and throw back to the client
                    return new RpcResult(new RuntimeException(StringUtils.toString(exception)));
                } catch (Throwable e) {
                    logger.warn("Fail to ExceptionFilter when called by " + getErrorString(invoker, invocation, e), e);
                    return result;
                }
            }
            return result;
        } catch (RuntimeException e) {
            logger.error("Got unchecked and undeclared exception which called by " + getErrorString(invoker, invocation, e), e);
            throw e;
        }
    }

    private String getErrorString(Invoker<?> invoker, Invocation invocation, Throwable e) {
        return RpcContext.getContext().getRemoteHost()
                + ". service: " + invoker.getInterface().getName() + ", method: " + invocation.getMethodName()
                + ", exception: " + e.getClass().getName() + ": " + e.getMessage();
    }
}