/**
 * fshows.com
 * Copyright (C) 2013-2020 All Rights Reserved.
 */
package com.fshows.fbank.sdk;

import cn.hutool.core.util.StrUtil;
import cn.hutool.system.SystemUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.fshows.com.fbank.openapi.sdk.client.FBankOpenApiClient;
import com.fshows.com.fbank.openapi.sdk.client.OpenParameters;
import com.fshows.com.fbank.openapi.sdk.config.Configuration;
import com.fshows.com.fbank.openapi.sdk.config.FileConfiguration;
import com.fshows.fbank.sdk.common.FBankApiEnum;
import com.fshows.fbank.sdk.common.FBankException;
import com.fshows.fbank.sdk.common.ReqIdUtil;
import com.fshows.fbank.sdk.common.ValidateUtil;
import com.fshows.fbank.sdk.request.FbankRequest;
import com.fshows.fbank.sdk.request.FileDownloadRequest;
import com.fshows.fbank.sdk.request.item.file.BizData;
import com.fshows.fbank.sdk.response.FbankResponse;
import com.fshows.fbank.sdk.response.FbankResponseBody;
import com.fshows.fbank.sdk.response.FbankResponseHead;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 企富通请求客户端
 *
 * @author buhao
 * @version FBankRequestClient.java, v 0.1 2020-10-15 16:19 buhao
 * @date 2020/10/15
 */
public class FBankRequestClient extends FBankOpenApiClient {

    /**
     * 请求成功 code
     */
    public static final String REQUEST_SUCCESS_CODE = "000000";
    private static final Logger LOGGER = LoggerFactory.getLogger(FBankRequestClient.class);
    private static final String FILE_SEPARATOR = SystemUtil.getOsInfo().getFileSeparator();

    public FBankRequestClient(Configuration configuration) {
        super(configuration);
    }

    /**
     * 请求企富通接口
     *
     * @param apiEnum       api枚举
     * @param request       请求
     * @param responseClass 响应类
     * @return {@link FbankResponse<T>}* @throws FBankException 企富通异常
     */
    public <T extends FbankResponseBody> FbankResponse<T> request(FBankApiEnum apiEnum,
                                                                  FbankRequest request,
                                                                  Class<T> responseClass) throws FBankException {
        final String reqId = ReqIdUtil.getId();

        LOGGER.info("request >> 企富通请求开始 >>reqId = {}, apiEnum = {}, request = {}, responseClass = {}",
                reqId, apiEnum, request, responseClass);

        // 1. 生成公共参数
        request.setMerchantNo(this.getConfiguration().merchantNo());
        request.setClientSerialNo(reqId);

        // 2. 参数校验
        ValidateUtil.validateWithThrow(request);

        //3. 拼接请求参数
        final OpenParameters context = new OpenParameters.Builder()
                .apiName(apiEnum.getValue())
                .param(JSON.toJSONString(request))
                .build();

        try {
            // 4. 发布请求
            final String result = send(context);
            // 5. 转化返回值
            final FbankResponse<T> response = JSON.parseObject(result, new TypeReference<FbankResponse<T>>(responseClass) {
            });
            // 6. 响应码判断
            final FbankResponseHead head = response.getHead();
            if (!StrUtil.equalsIgnoreCase(head.getRspCode(), REQUEST_SUCCESS_CODE)) {
                LOGGER.error("request >> 企富通请求业务异常 >> apiEnum = {}, request = {}",
                        apiEnum, request);
                throw new FBankException(head.getRspCode(), head.getRspMsg())
                        .newInstance("调用企富通接口业务异常: code:{0}, msg:{1}", head.getRspCode(), head.getRspMsg());
            }
            LOGGER.info("request >> 企富通请求结束 >>reqId = {}, apiEnum = {}, request = {}, responseClass = {}, result = {}",
                    reqId, apiEnum, request, responseClass, result);
            return response;
        } catch (FBankException e) {
            throw e;
        } catch (Exception e) {
            LOGGER.error("request >> 企富通请求服务异常 >> apiEnum = {}, request = {}, exception = {}",
                    apiEnum, request, ExceptionUtils.getStackTrace(e));
            throw FBankException.SERVER_EXCEPTION;
        }
    }


    /**
     * 下载文件
     *
     * @param request 请求
     * @return {@link String}
     */
    public String downloadFile(FileDownloadRequest request) {
        FBankApiEnum apiEnum = FBankApiEnum.FILE_DOWNLOAD;

        LOGGER.info("request >> 企富通请求开始 >> apiEnum = {}, request = {}",
                apiEnum, request);

        // 1. 生成公共参数
        final BizData bizData = request.getBizData();
        bizData.setMerchantNo(this.getConfiguration().merchantNo());

        // 2. 参数校验
        ValidateUtil.validateWithThrow(request);

        //3. 拼接请求参数
        final String fileName = request.getFileName();
        final OpenParameters context = new OpenParameters.Builder()
                .apiName(apiEnum.getValue())
                .fileName(fileName)
                .param(JSON.toJSONString(request))
                .build();

        try {
            // 4. 发布请求
            downloadFile(context);
            // 拼接本地地址
            return ((FileConfiguration) this.getConfiguration()).storagePath()
                    + FILE_SEPARATOR
                    + fileName + ".zip";
        } catch (Exception e) {
            LOGGER.error("request >> 企富通下载文件异常 >> apiEnum = {}, request = {}, exception = {}",
                    apiEnum, request, ExceptionUtils.getStackTrace(e));
            throw FBankException.SERVER_EXCEPTION;
        }
    }
}