/**
 * fshows.com
 * Copyright (C) 2013-2024 All Rights Reserved
 */
package com.fshows.ccbpay.request.trade.pay;

import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.annotation.JSONField;
import com.fshows.ccbpay.enums.trade.CcbPayTradeApiDefinitionEnum;
import com.fshows.ccbpay.request.base.CcbPayJsApiBaseRequest;
import com.fshows.ccbpay.response.trade.pay.CcbWxJsApiPayResponse;
import lombok.Data;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.NotBlank;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * @author liubo
 * @version CcbWxJsApiPayRequest.java, v 0.1 2024-04-23-11:53 上午 liubo
 */
@Data
public class CcbWxJsApiPayRequest extends CcbPayJsApiBaseRequest<CcbWxJsApiPayResponse, CcbPayTradeApiDefinitionEnum> {

    private static final long serialVersionUID = -4879495627158691091L;

    /**
     * 商户号
     * 请联系开立商户的分行提供
     *
     * 示例：105421097080009
     * (必填)
     */
    @NotBlank
    @Length(max = 15, message = "merchantid长度不能超过15")
    @JSONField(name = "MERCHANTID")
    private String merchantid;

    /**
     * 柜台号
     * 请联系开立商户的分行提供
     *
     * 示例：902807340
     * (必填)
     */
    @NotBlank
    @Length(max = 9, message = "posid长度不能超过9")
    @JSONField(name = "POSID")
    private String posid;

    /**
     * 分行号
     * 请联系开立商户的分行提供（一级分行号）
     *
     * 示例：360000000
     * (必填)
     */
    @NotBlank
    @Length(max = 9, message = "branchid长度不能超过9")
    @JSONField(name = "BRANCHID")
    private String branchid;


    /**
     * 订单号
     * 由商户提供，最长 30 位
     *
     * 示例：360000000
     * (必填)
     */
    @NotBlank
    @Length(max = 30, message = "orderId长度不能超过30")
    @JSONField(name = "ORDERID")
    private String orderId;

    /**
     * 付款金额
     * 由商户提供，按实际金额给
     * 出
     * 支付完成后，请商户与收到
     * 的商户通知中的付款金额比
     * 对，确认两者金额一致；
     *
     * 示例：0.01
     * (必填)
     */
    @NotBlank
    @JSONField(name = "PAYMENT")
    private String payment;

    /**
     * 缺省为 01－人民币
     * 币种
     * 示例：1
     */
    @NotBlank
    @Length(max = 2, message = "curcode长度不能超过30")
    @JSONField(name = "CURCODE")
    private String curcode;

    /**
     * 备注1
     * 一般作为商户自定义备注信
     * 息使用，可在对账单中显示。
     * 集团主商户代理发起子商户
     * 交易时，需将集团主商户信
     * 息填入该字段，格式如下：
     * JTSH:主商户号+主柜台
     * 例如：
     * JTSH:1050000000000001234
     * 56789
     * 其中：105000000000000 为
     * 主商户号
     * 123456789 为主柜台
     */
    @Length(max = 30, message = "remark1长度不能超过30")
    @JSONField(name = "REMARK1")
    private String remark1;

    /**
     * 备注2
     * 一般作为商户自定义备注信
     * 息使用，可在对账单中显示。
     */
    @Length(max = 30, message = "remark2长度不能超过30")
    @JSONField(name = "REMARK2")
    private String remark2;

    /**
     * 交易码
     */
    @NotBlank
    @Length(max = 6, message = "txCode长度不能超过6")
    @JSONField(name = "TXCODE")
    private String txCode;

    /**
     * MAC 校验域
     * 采用标准 MD5 算法，由商
     * 户实现
     */
    @Length(max = 32, message = "mac长度不能超过32")
    @JSONField(name = "MAC")
    private String mac;

    /**
     * 接口类型
     * 分行业务人员在 P2 员工
     * 渠道后台设置防钓鱼的
     * 开关。
     * 1- 防钓鱼接口
     */
    @NotBlank
    @Length(max = 1, message = "type长度不能超过1")
    @JSONField(name = "TYPE")
    private String type;

    /**
     * 公钥后 30 位
     * 商户从建行商户服务平台下
     * 载，截取后 30 位。
     * 仅作为源串参加 MD5 摘要，
     * 不作为参数传递
     */
    @Length(max = 30, message = "pub长度不能超过30")
    @JSONField(name = "PUB")
    private String pub;

    /**
     * 网关类型
     * 默认送 0
     */
    @NotBlank
    @Length(max = 100, message = "gateWay长度不能超过100")
    @JSONField(name = "GATEWAY")
    private String gateWay;

    /**
     * 客户端 IP
     * 客户在商户系统中的 IP，即
     * 客户登陆（访问）商户系统
     * 时使用的 ip）
     */
    @Length(max = 40, message = "clientIp长度不能超过40")
    @JSONField(name = "CLIENTIP")
    private String clientIp;

    /**
     * 客户注册信
     * 息
     * 客户在商户系统中注册的信
     * 息，中文需使用 escape 编码
     */
    @Length(max = 256, message = "regInfo长度不能超过256")
    @JSONField(name = "REGINFO")
    private String regInfo;

    /**
     * 商品信息
     * 客户购买的商品
     * 中文需使用 escape 编码
     */
    @Length(max = 256, message = "proInfo长度不能超过256")
    @JSONField(name = "PROINFO")
    private String proInfo;

    /**
     * 商户url
     * 商户送空值即可；
     * 具体请看 REFERER 设置说
     */
    @Length(max = 100, message = "regInfo长度不能超过100")
    @JSONField(name = "REFERER")
    private String referer;

    /**
     * 格式：
     * YYYYMMDDHHMMSS如：
     * 20120214143005
     * 银行系统时间> TIMEOUT
     * 时拒绝交易，若送空值则不
     * 判断超时。
     * 当该字段有值时参与 MAC
     * 校验，否则不参与 MAC 校
     * 验。
     */
    @Length(max = 14, message = "timeOut长度不能超过14")
    @JSONField(name = "TIMEOUT")
    private String timeOut;

    /**
     * 交易类型
     * JSAPI-- 公 众 号 支 付 、
     * MINIPRO--小程序
     */
    @NotBlank
    @Length(max = 16, message = "tradeType长度不能超过16")
    @JSONField(name = "TRADE_TYPE")
    private String tradeType;

    /**
     * 小程序/公众
     * 号的 APPID
     * 当前调起支付的小程序/公
     * 众号 APPID
     */
    @NotBlank
    @Length(max = 32, message = "subAppId长度不能超过16")
    @JSONField(name = "SUB_APPID")
    private String subAppId;

    /**
     * 用户子标识
     * 用户在小程序/公众号 appid
     * 下的唯一标识，小程序通过
     * wx.login 获取，接口文档地
     * 址 ：
     * https://developers.weixin.qq.c
     * om/miniprogram/dev/api/apilogin.html?t=20161122
     */
    @NotBlank
    @Length(max = 128, message = "subOpenId长度不能超过128")
    @JSONField(name = "SUB_OPENID")
    private String subOpenId;

    /**
     * 渠道商号
     * 对于商户自定义的渠道商号
     * 当该字段有值时参与 MAC
     * 校验，否则不参与 MAC 校
     * 验。
     */
    @Length(max = 20, message = "wxChannelId长度不能超过20")
    @JSONField(name = "WX_CHANNELID")
    private String wxChannelId;

    /**
     * 返回信息位图
     * 共 20 位，商户通知是否返回某
     * 个字段的位图，0 或空-不返回，
     * 1-返回。
     * 第 1 位：是否返回 OPENID 和
     * SUB_OPENID
     * 第 2 位：保留位，默认送 0
     * 第 3 位：保留位，默认送 0
     * 第 4 位：是否返回支付详细信息
     * 字段
     * 示例：10000000000000000000
     */
    @Length(max = 20, message = "returnField长度不能超过20")
    @JSONField(name = "RETURN_FIELD")
    private String returnField;


    /**
     * 实名支付功能，包含类型、
     * 证件号、姓名三个子域（如果本字段
     * 出现，那么本字
     * 段包含的三个子域均需出现。详见下
     * 文说明5)USERPARAM字段说明）。
     * 当该字段有值时参与MAC校验，否则不
     * 参与MAC校验。
     */
    @Length(max = 2000, message = "userParam长度不能超过2000")
    @JSONField(name = "USERPARAM")
    private String userParam;

    /**
     * 二级商户代
     * 码
     */
    @Length(max = 15, message = "smerId长度不能超过2000")
    @JSONField(name = "SMERID")
    private String smerId;


    /**
     * 二级商户名称
     * 码
     */
    @Length(max = 100, message = "smerName长度不能超过100")
    @JSONField(name = "SMERNAME")
    private String smerName;

    /**
     * 二级商户类别代码
     *
     */
    @Length(max = 15, message = "smerTypeId长度不能超过15")
    @JSONField(name = "SMERTYPEID")
    private String smerTypeId;

    /**
     * 二级商户类别名称，汉字最
     * 长 27 个，中文需使用 escape
     * 编码
     *
     */
    @Length(max = 81, message = "smerType长度不能超过81")
    @JSONField(name = "SMERTYPE")
    private String smerType;

    /**
     * 交易类型代
     * 码
     *
     */
    @Length(max = 15, message = "tradeCode长度不能超过15")
    @JSONField(name = "TRADECODE")
    private String tradeCode;

    /**
     * 交易类型名称
     * 如消费、投资理财、信用卡
     * 还款等，中文需使用 escape
     * 编码
     *
     */
    @Length(max = 30, message = "tradeName长度不能超过30")
    @JSONField(name = "TRADENAME")
    private String tradeName;

    /**
     * 商品类别代
     * 码
     *
     */
    @Length(max = 24, message = "smeProType长度不能超过24")
    @JSONField(name = "SMEPROTYPE")
    private String smeProType;

    /**
     * 商品类别名
     * 称
     *
     */
    @Length(max = 50, message = "proname长度不能超过50")
    @JSONField(name = "PRONAME")
    private String proname;

    @Override
    public Map<String,String> getReqMap(){

        HashMap map = new LinkedHashMap();
        map.put("MERCHANTID",merchantid);
        map.put("POSID",posid);
        map.put("BRANCHID",branchid);
        map.put("ORDERID",orderId);
        map.put("PAYMENT",payment);
        map.put("CURCODE",curcode);
        map.put("TXCODE",txCode);
        map.put("REMARK1",remark1);
        map.put("REMARK2",remark2);
        map.put("TYPE",type);
        map.put("PUB",pub);
        map.put("GATEWAY",gateWay);
        map.put("CLIENTIP","");
        map.put("REGINFO","");
        map.put("PROINFO","");
        map.put("REFERER","");
        map.put("TRADE_TYPE",tradeType);
        map.put("SUB_APPID",subAppId);
        map.put("SUB_OPENID",subOpenId);

        return map;
    }


}