/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package
        com.fshows.ark.spring.boot.starter.extend.mq.configlistener;

import com.fshows.ark.spring.boot.starter.core.mq.base.consumer.FsConsumerModel;
import com.fshows.ark.spring.boot.starter.core.mq.base.consumer.FsConsumerModelProperties;
import com.fshows.ark.spring.boot.starter.core.mq.base.consumer.IFshowsConsumer;
import com.fshows.ark.spring.boot.starter.core.mq.rocketmq.consumer.AbstractConsumerContainer;
import com.fshows.ark.spring.boot.starter.exception.MQConsumerException;
import com.fshows.ark.spring.boot.starter.util.ConfigUtil;
import com.fshows.ark.spring.boot.starter.util.StrUtil;
import com.fshows.ark.spring.boot.starter.util.ThreadUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 配置监听器
 *
 * @author liluqing
 * @version ConsumeConfigChangeListener.java, v 0.1 2022-09-15 15:59
 */
@Slf4j
public class ConsumeConfigChangeListener {

    @Autowired
    private AbstractConsumerContainer consumerContainer;

    /**
     * 消费者重启最大延迟时间
     */
    @Value("${ark.mq.consume.restart.max-delay-time:60}")
    private Integer restartMaxDelayTime;

    private ConcurrentHashMap<String, List<MQPropertyModel>> map = new ConcurrentHashMap<>();

    public ConsumeConfigChangeListener(AbstractConsumerContainer consumerContainer) {
        this.consumerContainer = consumerContainer;
    }

    public void init() {
        Field[] fieldList = FsConsumerModelProperties.class.getDeclaredFields();
        // 开启字段的访问权限
        Arrays.stream(fieldList).forEach(f->f.setAccessible(true));
        List<IFshowsConsumer> list = consumerContainer.getFshowsConsumerList();
        for (IFshowsConsumer consumer : list) {
            FsConsumerModel consumerModel = consumer.getConsumerModel();
            // 存放配置key的对象
            FsConsumerModelProperties consumerConfigKey = consumerModel.getConsumerConfigKey();
            // 存放实际配置的值的对象
            FsConsumerModelProperties consumerModelProperties = consumerModel.getConsumerProperties();
            for (Field field : fieldList) {
                // 获取动态key的名称
                String key = ConfigUtil.getDynamicKey(getFieldValueStr(consumerConfigKey, field));
                // 如果key为null则说明是静态值，不是动态配置的key
                if (key ==  null) {
                    continue;
                }
                // 当前该配置的值
                String value = getFieldValueStr(consumerModelProperties, field);
                // 构建动态配置
                MQPropertyModel propertyModel = new MQPropertyModel();
                propertyModel.setTarget(consumerModelProperties);
                propertyModel.setField(field);
                propertyModel.setKey(key);
                propertyModel.setValue(value);
                propertyModel.setFshowsConsumer(consumer);

                // 将属性对象访问map缓存中
                put(key, propertyModel);
            }
        }
    }

    /**
     * 根据反射获取对象值
     *
     * @param target
     * @param field
     * @return
     */
    public String getFieldValueStr(Object target, Field field) {
        try {
            Object value = field.get(target);
            return value == null ? null : value.toString();
        } catch (IllegalAccessException e) {
            log.error("ark-spring-boot-starter >> mq消费者配置变更监听器加载异常!", e);
            throw new MQConsumerException("ark-spring-boot-starter >> mq消费者配置变更监听器加载异常！");
        }
    }

    /**
     * 将配置放入map中
     *
     * @param key
     * @param mqPropertyModel
     */
    private void put(String key, MQPropertyModel mqPropertyModel) {
        List<MQPropertyModel> propertyModelList = map.get(key);
        if (propertyModelList == null) {
            propertyModelList = new ArrayList<>();
        }
        propertyModelList.add(mqPropertyModel);
        map.put(key, propertyModelList);
    }

    /**
     * 重新加载mq配置
     *
     * @param configChangeModelList
     */
    public void reloadConfig(List<ConfigChangeModel> configChangeModelList) {
        if (configChangeModelList == null) {
            return;
        }
        // 初始化配置缓存
        if (map.size() == 0) {
            init();
        }
        // 需要重新reload的comsumer实例
        Set<IFshowsConsumer> reloadConsumerSet = new HashSet<>();
        for (ConfigChangeModel configChangeModel : configChangeModelList) {
            reloadConsumerSet.addAll(setMQPropertyNewVaule(configChangeModel));
        }
        if (reloadConsumerSet.size() == 0) {
            log.info("ark-spring-boot-starter >> 未将测到mq相关配置更新, 本次配置变更不刷新consumer实例！");
            return;
        }
        // 随机睡眠一段时间之后再重启消费者实例，避免所有节点同一时间重启导致消息堆积
        int restartDelayTime = new Random().nextInt(restartMaxDelayTime);
        log.info("ark-spring-boot-starter >> 检测到配置更新,{}秒钟后开始重启consumer实例！", restartDelayTime);
        ThreadUtil.sleep(restartDelayTime);
        // 循环重启发生了变更的消费者实例
        reloadConsumerSet.forEach(IFshowsConsumer::restart);
        log.info("ark-spring-boot-starter >> consumer实例重启完成！");
    }


    /**
     * 更新某一个指定配置
     *
     * @param configChangeModel
     * @return 返回发生了配置变更的消费者实例
     */
    public List<IFshowsConsumer> setMQPropertyNewVaule(ConfigChangeModel configChangeModel) {
        // 新值为空时则不做任何处理
        if (StrUtil.isBlank(configChangeModel.getNewValue())) {
            return new ArrayList<>();
        }
        String key = configChangeModel.getKey();
        List<MQPropertyModel> list = map.get(key);
        if (list == null) {
            return new ArrayList<>();
        }
        List<IFshowsConsumer> updateConsumer = new ArrayList<>();
        for (MQPropertyModel mqPropertyModel : list) {
            // 新值和当前值一致时不做任何变更
            if (configChangeModel.getNewValue().equals(mqPropertyModel.getValue())) {
                continue;
            }
            boolean bo = setFieldValueStr(configChangeModel.getNewValue(), mqPropertyModel.getTarget(), mqPropertyModel.getField());
            if (bo) {
                mqPropertyModel.setValue(configChangeModel.getNewValue());
                updateConsumer.add(mqPropertyModel.getFshowsConsumer());
            }
        }
        return updateConsumer;
    }

    /**
     * 根据反射获取对象值
     *
     * @param target
     * @param field
     * @return
     */
    public boolean setFieldValueStr(String value, Object target, Field field) {
        try {
            field.set(target, value);
            return true;
        } catch (IllegalAccessException e) {
            log.error("ark-spring-boot-starter >> mq消费者配置变更监听器异常 >> 动态加载变更配置异常 >> value={}", value);
        }
        return false;
    }
}