/**
 * fshows.com
 * Copyright (C) 2013-2021 All Rights Reserved.
 */
package
        com.fshows.ark.spring.boot.starter.core.mq.rocketmq.producer;

import com.fshows.ark.spring.boot.starter.annotation.mq.RocketMessage;
import com.fshows.ark.spring.boot.starter.annotation.mq.RocketMessageAsync;
import com.fshows.ark.spring.boot.starter.annotation.mq.RocketProducer;
import com.fshows.ark.spring.boot.starter.annotation.mq.TransactionMessage;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsProducerModel;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.IFshowsProducer;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.IProducerProxyFactory;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.TransactionMessageManage;
import com.fshows.ark.spring.boot.starter.core.mq.rocketmq.producer.transaction.TransactionMessageMethodModel;
import com.fshows.ark.spring.boot.starter.enums.ProducerParamTypeEnum;
import com.fshows.ark.spring.boot.starter.enums.ProducerReturnTypeEnum;
import com.fshows.ark.spring.boot.starter.enums.ProducerSendTypeEnum;
import com.fshows.ark.spring.boot.starter.enums.ProducerTypeEnum;
import com.fshows.ark.spring.boot.starter.exception.MQProducerException;
import com.fshows.ark.spring.boot.starter.util.ConfigUtil;
import com.fshows.ark.spring.boot.starter.util.NumberUtil;
import com.fshows.ark.spring.boot.starter.util.StrUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author liluqing
 * @version AbstractProducerProxyFactory.java, v 0.1 2021-08-30 19:49
 */
@Slf4j
public abstract class AbstractProducerProxyFactory implements IProducerProxyFactory, ApplicationContextAware {

    /**
     * spring上下文
     */
    protected ApplicationContext applicationContext;

    /**
     * 缓存的fs的生产者bean
     */
    protected final Map<String, IFshowsProducer> fsProducerCacheMap = new ConcurrentHashMap<>();

    /**
     * 构建rocketmq生产者代理类
     *
     * @return
     */
    @Override
    public <T> T bulidRocketProducerProxy(Class<T> producerInterface) {
        // 解析生产者接口的元信息
        FsProducerModel producerModel = resolveProducerInterface(producerInterface, applicationContext);
        // 获取事务消息管理器
        TransactionMessageManage transactionMessageManage = getTransactionMessageManage(applicationContext);
        // 获取fs生产者对象,对阿里云produceBean的封装，避免直接引用阿里云produceBean
        IFshowsProducer fshowsProducer = getFshowsProducer(producerModel, transactionMessageManage);

        // 构建代理对象
        T instance = buildProducerProxy(fshowsProducer, transactionMessageManage, producerModel, producerInterface);
        return instance;
    }

    /**
     * 获取事务消息管理器
     *
     * @param ctx
     * @return
     */
    public TransactionMessageManage getTransactionMessageManage(ApplicationContext ctx) {
        try {
            Map<String, TransactionMessageManage> map = ctx.getBeansOfType(TransactionMessageManage.class);
            if (map.size() == 0) {
                return null;
            }
            return map.entrySet().iterator().next().getValue();
        } catch (Exception e) {
            log.warn("ark-spring-boot-starter >> 当前应用未配置TransactionMessageManage,暂无法使用事务消息功能！");
        }
        return null;
    }

    /**
     * 获取fs生产者对象
     *
     * @return
     */
    private IFshowsProducer getFshowsProducer(FsProducerModel fsProducerModel, TransactionMessageManage transactionMessageManage) {
        // 先尝试根据groupId从缓存中获取
        if (fsProducerCacheMap.containsKey(fsProducerModel.getGroupId())) {
            return fsProducerCacheMap.get(fsProducerModel.getGroupId());
        }
        IFshowsProducer fshowsProducer = createIFshowsProducer(fsProducerModel, transactionMessageManage);
        fsProducerCacheMap.put(fsProducerModel.getGroupId(), fshowsProducer);
        return fshowsProducer;
    }

    /**
     * 创建fs生产者bean对象
     *
     * @param fsProducerModel
     * @return
     */
    public abstract IFshowsProducer createSimpleFshowsProducer(FsProducerModel fsProducerModel);

    /**
     * 构建生产者动态代理处理器,进行消息发送处理
     *
     * @param fshowsProducer
     * @param producerModel
     * @return
     */
    public abstract <T> T buildProducerProxy(IFshowsProducer fshowsProducer,TransactionMessageManage transactionMessageManage , FsProducerModel producerModel, Class<T> producerInterface);

    /**
     * 创建fs生产者bean对象
     *
     * @param fsProducerModel
     * @return
     */
    protected abstract IFshowsProducer createIFshowsProducer(FsProducerModel fsProducerModel, TransactionMessageManage transactionMessageManage);

    /**
     * 解析需要被代理的接口,获取注解中配置的消息源数据
     *
     * @param producerInterface
     * @return
     */
    protected FsProducerModel resolveProducerInterface(Class producerInterface, ApplicationContext ctx) {
        RocketProducer producerAnnotation = (RocketProducer) producerInterface.getAnnotation(RocketProducer.class);
        FsProducerModel fsProducerModel = new FsProducerModel();
        fsProducerModel.setProducerClass(producerInterface);
        fsProducerModel.setGroupId(ConfigUtil.getProperty(producerAnnotation.groupId(), ctx));
        fsProducerModel.setSendMsgTimeoutMillis(ConfigUtil.getProperty(producerAnnotation.sendMsgTimeoutMillis(), ctx));
        fsProducerModel.setCharsetName(ConfigUtil.getProperty(producerAnnotation.msgContentCharset(), ctx));
        fsProducerModel.setAccessKey(ConfigUtil.getProperty(producerAnnotation.accessKey(), ctx));
        fsProducerModel.setSecretKey(ConfigUtil.getProperty(producerAnnotation.secretKey(), ctx));
        fsProducerModel.setNamesrvAddr(ConfigUtil.getProperty(producerAnnotation.namesrvAddr(), ctx));
        Map<Method, RocketSendMessageMethodModel> sendMessageMethodModelMap = new HashMap<>();
        Method[] methods = producerInterface.getMethods();
        for (Method m : methods) {
            // 解析@RocketMessage注解
            RocketSendMessageMethodModel messageMethodModel = resolveRocketMessage(ctx, producerAnnotation, m);
            if (messageMethodModel == null) {
                // 解析@RocketMessageAsync注解
                messageMethodModel = resolveRocketMessageAsync(ctx, producerAnnotation, m);
            }
            if (messageMethodModel == null) {
                log.error("ark-spring-boot-starter >> {}方法没有标注@RocketMessage或者@RocketMessageAsync注解,生产者自动化配置失败", m);
                throw new MQProducerException("ark-spring-boot-starter >> 检测到" + m + "方法没有标注@RocketMessage或者@RocketMessageAsync注解的方法");
            }
            sendMessageMethodModelMap.put(m, messageMethodModel);
        }
        fsProducerModel.setSendMessageMethodModelMap(sendMessageMethodModelMap);
        // 校验生产者元信息是否合法
        checkFsProducerModel(fsProducerModel, ctx);
        return fsProducerModel;
    }

    /**
     * 校验生产者实例配置完整性
     *
     * @param fsProducerModel 生产者相关配置
     */
    protected void checkFsProducerModel(FsProducerModel fsProducerModel, ApplicationContext ctx) {
        String className = fsProducerModel.getProducerClass().getName();
        if (StrUtil.isEmpty(fsProducerModel.getGroupId())) {
            log.error("ark-spring-boot-starter >> {}接口标注没有设置groupId", className);
            throw new MQProducerException("ark-spring-boot-starter >> 检测到" + className + "类的@RocketProducer注解没有设置groupId或配置项未配置");
        }
        if (StrUtil.isEmpty(fsProducerModel.getNamesrvAddr())) {
            log.error("ark-spring-boot-starter >> 检测到mq name server地址配置项'ark.mq.name-server'未配置");
            throw new MQProducerException("ark-spring-boot-starter >> 检测到mq name server地址配置项'ark.mq.name-server'未配置");
        }
        if (!NumberUtil.isNumber(fsProducerModel.getSendMsgTimeoutMillis())) {
            log.error("ark-spring-boot-starter >> 检测到生产者'{}'类的sendMsgTimeoutMillis属性为非数字", className);
            throw new MQProducerException("ark-spring-boot-starter >> 检测到生产者'" + className + "'类的sendMsgTimeoutMillis属性为非数字");
        }
        // 发送方法元信息
        Map<Method, RocketSendMessageMethodModel> sendMessageMethodModelMap = fsProducerModel.getSendMessageMethodModelMap();
        if (sendMessageMethodModelMap == null || sendMessageMethodModelMap.size() == 0) {
            return;
        }
        // 校验方法元信息是否合法
        for (Map.Entry<Method, RocketSendMessageMethodModel> entry : sendMessageMethodModelMap.entrySet()) {
            RocketSendMessageMethodModel messageMethodModel = entry.getValue();
            Method interfaceMethod = messageMethodModel.getInterfaceMethod();
            String methodPath = className + "." + interfaceMethod.getName();
            if (StrUtil.isEmpty(messageMethodModel.getTopic())) {
                log.error("ark-spring-boot-starter >> 检测到生产者方法'{}'的topic属性未设置", methodPath);
                throw new MQProducerException("ark-spring-boot-starter >> 检测到生产者方法'" + methodPath + "'的topic属性未设置或配置项未配置");
            }
            if (StrUtil.isEmpty(messageMethodModel.getTag())) {
                log.error("ark-spring-boot-starter >> 检测到生产者方法'{}'的tag属性未设置", methodPath);
                throw new MQProducerException("ark-spring-boot-starter >> 检测到生产者方法'" + methodPath + "'的tag属性未设置或配置项未配置");
            }
            if (!NumberUtil.isNumber(messageMethodModel.getDelayTime())) {
                log.error("ark-spring-boot-starter >> 检测到生产者方法'{}'的delayTime属性为非数字", methodPath);
                throw new MQProducerException("ark-spring-boot-starter >> 检测到生产者方法'" + methodPath + "'的delayTime属性为非数");
            }
            // 事务消息属性校验
            checkTransactionMsgConfig(messageMethodModel, methodPath,ctx);
        }
    }

    /**
     * 校验事务消息配置
     */
    private void checkTransactionMsgConfig(RocketSendMessageMethodModel messageMethodModel, String methodPath, ApplicationContext ctx) {
        // 如果不是事务消息则返回
        if (!ProducerTypeEnum.TRANSACTION_MESSAGE_PRODUCER.equals(messageMethodModel.getProducerTypeEnum()) ) {
            return;
        }
        // 事务消息只支持同步发送方式
        if (!ProducerSendTypeEnum.SYNC.equals(messageMethodModel.getSendTypeEnum())) {
            log.error("ark-spring-boot-starter >> 事务消息只支持ProducerSendTypeEnum.SYNC(同步发送方式),方法'{}'配置有误", methodPath);
            throw new MQProducerException("ark-spring-boot-starter >> 事务消息只支持ProducerSendTypeEnum.SYNC(同步发送方式),方法'" + methodPath + "'配置有误");
        }
        if (messageMethodModel.getTransactionMessageMethodModel() == null
                || !NumberUtil.isNumber(messageMethodModel.getTransactionMessageMethodModel().getTransactionLocalTimeOut())) {
            log.error("ark-spring-boot-starter >> 检测到生产者事务消息方法'{}'的transactionLocalTimeOut属性未设置", methodPath);
            throw new MQProducerException("ark-spring-boot-starter >> 检测到生产者方法'" + methodPath + "'的transactionLocalTimeOut属性未设置或配置项未配置");
        }
        TransactionMessageManage transactionMessageManage = getTransactionMessageManage(ctx);
        if (transactionMessageManage == null) {
            log.error("ark-spring-boot-starter >> 因事务消息管理器未配置,'{}'上的@TransactionMessage注解配置无效,", methodPath);
            throw new MQProducerException("ark-spring-boot-starter >> '" + methodPath + "'上的@TransactionMessage注解配置无法生效,事务管理器未配置");
        }

    }

    /**
     * 处理 @RocketMessage注解源信息
     *
     * @param ctx
     * @param producerAnnotation
     * @param m
     * @return
     */
    protected RocketSendMessageMethodModel resolveRocketMessage(ApplicationContext ctx, RocketProducer producerAnnotation, Method m) {
        RocketMessage rocketMessage = m.getAnnotation(RocketMessage.class);
        if (rocketMessage == null) {
            return null;
        }
        // 分析方法校验并获取方法的入参类型和反参类型
        ProducerParamTypeEnum paramTypeEnum = getAndCheckMethodParam(m, rocketMessage.sendType());
        ProducerReturnTypeEnum returnTypeEnum = getAndCheckMethodReturn(m, rocketMessage.sendType());
        // 获取事务事务消息配置
        TransactionMessageMethodModel transactionMessageMethodModel = resolveTransactionMessage(ctx, m);

        // 如果存在事务消息配置则是事务消息，否则为普通消息
        ProducerTypeEnum producerTypeEnum = transactionMessageMethodModel == null
                ? ProducerTypeEnum.DEFAULT_NOMAL_PRODUCER : ProducerTypeEnum.TRANSACTION_MESSAGE_PRODUCER;

        // 构建发送方法原数据
        RocketSendMessageMethodModel messageMethodModel = RocketSendMessageMethodModel.builder()
                .groupId(ConfigUtil.getProperty(producerAnnotation.groupId(), ctx))
                .topic(ConfigUtil.getProperty(rocketMessage.topic(), ctx))
                .tag(ConfigUtil.getProperty(rocketMessage.tag(), ctx))
                .msgKeyPrefix(ConfigUtil.getProperty(rocketMessage.msgKeyPrefix(), ctx))
                .delayTime(ConfigUtil.getProperty(rocketMessage.delayTime(), ctx))
                .sendErrorThrowEx(rocketMessage.sendErrorThrowEx())
                .sendTypeEnum(rocketMessage.sendType())
                .interfaceMethod(m)
                .paramTypeEnum(paramTypeEnum)
                .returnTypeEnum(returnTypeEnum)
                .producerTypeEnum(producerTypeEnum)
                .transactionMessageMethodModel(transactionMessageMethodModel)
                .build();
        return messageMethodModel;
    }

    /**
     * 构建事务消息原数据
     *
     * @param ctx
     * @param m
     * @return
     */
    private TransactionMessageMethodModel resolveTransactionMessage(ApplicationContext ctx, Method m) {
        TransactionMessage transactionMessage = m.getAnnotation(TransactionMessage.class);
        if (transactionMessage == null) {
            return null;
        }
        // 构建发送方法原数据
        TransactionMessageMethodModel transactionMessageMethodModel = TransactionMessageMethodModel.builder()
                .transactionLocalTimeOut(ConfigUtil.getProperty(transactionMessage.localTimeOut(), ctx))
                .build();
        return transactionMessageMethodModel;
    }

    /**
     * 处理 @RocketMessage注解源信息
     *
     * @param ctx
     * @param producerAnnotation
     * @param m
     * @return
     */
    protected RocketSendMessageMethodModel resolveRocketMessageAsync(ApplicationContext ctx,
                                                                     RocketProducer producerAnnotation, Method m) {
        RocketMessageAsync rocketMessage = m.getAnnotation(RocketMessageAsync.class);
        if (rocketMessage == null) {
            return null;
        }
        // 分析方法校验并获取方法的入参类型和反参类型
        ProducerParamTypeEnum paramTypeEnum = getAndCheckMethodParam(m, ProducerSendTypeEnum.ASYNC);
        ProducerReturnTypeEnum returnTypeEnum = getAndCheckMethodReturn(m, ProducerSendTypeEnum.ASYNC);

        // 获取事务事务消息配置
        TransactionMessageMethodModel transactionMessageMethodModel = resolveTransactionMessage(ctx, m);

        // 如果存在事务消息配置则是事务消息，否则为普通消息
        ProducerTypeEnum producerTypeEnum = transactionMessageMethodModel == null
                ? ProducerTypeEnum.DEFAULT_NOMAL_PRODUCER : ProducerTypeEnum.TRANSACTION_MESSAGE_PRODUCER;

        RocketSendMessageMethodModel messageMethodModel = RocketSendMessageMethodModel.builder()
                .groupId(ConfigUtil.getProperty(producerAnnotation.groupId(), ctx))
                .topic(ConfigUtil.getProperty(rocketMessage.topic(), ctx))
                .tag(ConfigUtil.getProperty(rocketMessage.tag(), ctx))
                .delayTime(ConfigUtil.getProperty(rocketMessage.delayTime(), ctx))
                .sendTypeEnum(ProducerSendTypeEnum.ASYNC)
                .msgKeyPrefix(ConfigUtil.getProperty(rocketMessage.msgKeyPrefix(), ctx))
                .interfaceMethod(m)
                .paramTypeEnum(paramTypeEnum)
                .returnTypeEnum(returnTypeEnum)
                .producerTypeEnum(producerTypeEnum)
                .transactionMessageMethodModel(transactionMessageMethodModel)
                .build();
        return messageMethodModel;
    }

    /**
     * 获取并校验消息发送方法的入参
     *
     * @param m 被代理的方法
     * @param sendTypeEnum 发送方式
     */
    protected ProducerParamTypeEnum getAndCheckMethodParam(Method m, ProducerSendTypeEnum sendTypeEnum) {
        Class[] paramTypes = m.getParameterTypes();
        ProducerParamTypeEnum producerParamTypeEnum = ProducerParamTypeEnum.getByParamsClass(paramTypes);
        if (producerParamTypeEnum == null) {
            log.error("ark-spring-boot-starter >> {}方法的参数列表不合法,无法自动化配置mq生产者实例！", m);
            throw new MQProducerException("ark-spring-boot-starter >> 检测到" + m + "方法的参数列表不合法");
        }
        ProducerSendTypeEnum[] paramAllowSendType = producerParamTypeEnum.getSendTypeEnum();
        Optional<ProducerSendTypeEnum> paramSendTypeOptional = Arrays.stream(paramAllowSendType)
                .filter(e -> e.equals(sendTypeEnum)).findFirst();
        if (!paramSendTypeOptional.isPresent()) {
            log.error("ark-spring-boot-starter >> {}方法的参数列表不合法！", m);
            throw new MQProducerException("ark-spring-boot-starter >> 检测到" + m + "方法的参数列表不合法");
        }
        return producerParamTypeEnum;
    }

    /**
     * 获取并校验消息发送方法的返回值
     *
     * @param m
     * @param sendTypeEnum
     */
    protected ProducerReturnTypeEnum getAndCheckMethodReturn(Method m, ProducerSendTypeEnum sendTypeEnum) {
        ProducerReturnTypeEnum returnTypeEnum = ProducerReturnTypeEnum.getByValue(m.getReturnType().getName());
        if (returnTypeEnum == null) {
            log.error("ark-spring-boot-starter >> {}方法的返回值不合法！目前返回值仅支持void, com.fshows.ark.spring.boot.starter.core.mq.base.FsSendResult类型", m);
            throw new MQProducerException("ark-spring-boot-starter >> 检测到" + m + "方法的返回值不合法");
        }
        ProducerSendTypeEnum[] returnAllowSendType = returnTypeEnum.getSendTypeEnum();
        Optional<ProducerSendTypeEnum> returnSendTypeOptional = Arrays.stream(returnAllowSendType)
                .filter(e -> e.equals(sendTypeEnum)).findFirst();
        if (!returnSendTypeOptional.isPresent()) {
            log.error("ark-spring-boot-starter >> {}方法的返回值不合法！当前发送方式不支持'{}'类型", m, m.getReturnType().getName());
            throw new MQProducerException("ark-spring-boot-starter >> 检测到" + m + "方法的返回值不合法");
        }
        return returnTypeEnum;
    }


    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}