/**
 * fshows.com
 * Copyright (C) 2013-2021 All Rights Reserved.
 */
package
        com.fshows.ark.spring.boot.starter.core.mq.rocketmq.consumer;

import com.fshows.ark.spring.boot.starter.annotation.mq.RocketConsumer;
import com.fshows.ark.spring.boot.starter.annotation.mq.RocketListener;
import com.fshows.ark.spring.boot.starter.constant.CommonConfigConstant;
import com.fshows.ark.spring.boot.starter.constant.CommonConstant;
import com.fshows.ark.spring.boot.starter.constant.MqConfigConstant;
import com.fshows.ark.spring.boot.starter.core.mq.base.consumer.FsConsumerConfigKey;
import com.fshows.ark.spring.boot.starter.core.mq.base.consumer.FsConsumerModel;
import com.fshows.ark.spring.boot.starter.core.mq.base.consumer.FsConsumerModelProperties;
import com.fshows.ark.spring.boot.starter.core.mq.base.consumer.IFshowsConsumer;
import com.fshows.ark.spring.boot.starter.core.mq.rocketmq.interceptor.consumer.IConsumerInterceptorManagement;
import com.fshows.ark.spring.boot.starter.enums.ConsumerParamTypeEnum;
import com.fshows.ark.spring.boot.starter.enums.ConsumerReturnTypeEnum;
import com.fshows.ark.spring.boot.starter.exception.MQConsumerException;
import com.fshows.ark.spring.boot.starter.util.ConfigUtil;
import com.fshows.ark.spring.boot.starter.util.LogUtil;
import com.fshows.ark.spring.boot.starter.util.StrUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

/**
 * 抽象消费者容器
 *
 * @author liluqing
 * @version AbstractConsumerContainer.java, v 0.1 2021-09-09 17:31
 */
@Slf4j
public abstract class AbstractConsumerContainer implements ApplicationRunner, ApplicationContextAware {

    /**
     * spring容器上下文
     */
    private ApplicationContext ctx;

    /**
     * 消费者拦截器管理器
     */
    protected IConsumerInterceptorManagement consumerInterceptorManagement;

    /**
     * 消费者容器,用于存放消费者bean
     */
    private ConcurrentHashMap<String, IFshowsConsumer> fshowsConsumerMap = new ConcurrentHashMap<String, IFshowsConsumer>(16);

    /**
     * 初始化消费者
     *      [入口方法, 实现了ApplicationRunner, 在spring容器初始化之后自动执行run方法]
     *
     * @param args
     * @throws Exception
     */
    @Override
    public void run(ApplicationArguments args) throws Exception {
        // 获取标记有consumer注解的bean
        Map<String, Object> rocketConsumer = ctx.getBeansWithAnnotation(RocketConsumer.class);
        if (rocketConsumer == null || rocketConsumer.size() == 0) {
            return;
        }
        // 校验元配置信息
        checkMetaConfig();
        // 解析bean，获取consumer元信息
        List<FsConsumerModel> fsConsumerModelList = resolveRocketConsumeModel(rocketConsumer);
        if (fsConsumerModelList == null || fsConsumerModelList.size() == 0) {
            return;
        }
        // 批量创建消费者bean
        List<IFshowsConsumer> consumerBeanList = fsConsumerModelList.stream()
                                                    .map(e->createIFshowsConsumer(e))
                                                    .collect(Collectors.toList());
        // 启动消费者
        consumeStart(consumerBeanList);
        log.info("ark-spring-boot-starter >> Rocket mq started successfully!");
    }

    /**
     * 校验消费者源配置信息
     */
    public void checkMetaConfig() {
        // mq 命名服务地址
        String mqNameServer = ConfigUtil.getProperty(MqConfigConstant.NAME_SERVER_ADDR, ctx);
        if (mqNameServer == null || mqNameServer.length() == 0) {
            LogUtil.error(log, "ark-spring-boot-starter >> 配置项${ark.mq.name-server}MQ Name Server地址未配置, 启动失败！");
            throw new MQConsumerException("ark-spring-boot-starter >> MQNameServer地址未配置！");
        }
        // 如果是使用阿里云服务器,则阿里云accessKey和secretKey必须配置
        if (mqNameServer.contains(CommonConstant.ALIYUN_DOMAIN_NAME)) {
            // mq的 accessKey
            String accessKey = ConfigUtil.getProperty(CommonConfigConstant.ALIYUN_ACCESS_KEY, ctx);
            // mq的 secretKey
            String secretKey = ConfigUtil.getProperty(CommonConfigConstant.ALIYUN_SECRET_KEY, ctx);
            if (CommonConstant.NONE.equalsIgnoreCase(accessKey) || CommonConstant.NONE.equalsIgnoreCase(secretKey)) {
                LogUtil.error(log, "ark-spring-boot-starter >> 启动失败 >> 检测到使用阿里云MQ服务器,必须配置${ark.aliyun.access-key}和${ark.aliyun.secret-key}配置项！");
                throw new MQConsumerException("ark-spring-boot-starter >> rocketmq消费者创建失败！");
            }
        }
    }

    /**
     * 启动所有消费者
     */
    protected void consumeStart(List<IFshowsConsumer> consumerBeanList) {
        if (consumerBeanList == null || consumerBeanList.size() == 0) {
            return;
        }
        consumerBeanList.parallelStream().forEach(consumerBean -> consumerBean.start());
    }

    /**
     * 批量创建消费者
     *
     * @param fsConsumerModel
     * @return
     */
    protected IFshowsConsumer createIFshowsConsumer(FsConsumerModel fsConsumerModel) {
        FsConsumerModelProperties consumerProperties = fsConsumerModel.getConsumerProperties();
        IFshowsConsumer fshowsConsumer = fshowsConsumerMap.get(consumerProperties.getGroupId());
        if (fshowsConsumer != null) {
            return fshowsConsumerMap.get(consumerProperties.getGroupId());
        }
        fshowsConsumer = doCreateIFshowsConsumer(fsConsumerModel);
        fshowsConsumerMap.put(consumerProperties.getGroupId(), fshowsConsumer);
        return fshowsConsumer;
    }

    /**
     * 执行消费者创建动作
     *
     * @param fsConsumerModel
     * @return
     */
    protected abstract IFshowsConsumer doCreateIFshowsConsumer(FsConsumerModel fsConsumerModel);

    /**
     * 解析bean和注解生成消费者元信息
     *
     * @param rocketConsumer
     * @return
     */
    protected List<FsConsumerModel> resolveRocketConsumeModel(Map<String, Object> rocketConsumer) {
        List<FsConsumerModel> list = new ArrayList<>();
        if (rocketConsumer == null) {
            return list;
        }
        for (Map.Entry<String, Object> entry : rocketConsumer.entrySet()) {
            Object target = entry.getValue();
            RocketConsumer calssAnnotation = target.getClass().getAnnotation(RocketConsumer.class);
            String topic = calssAnnotation.topic();
            Method[] methods = target.getClass().getDeclaredMethods();
            for (Method m: methods) {
                RocketListener annotation = m.getAnnotation(RocketListener.class);
                if (annotation == null) {
                    continue;
                }
                // 生成消费者元信息
                list.add(createRocketConsumerModel(entry.getKey(), entry.getValue(), m, annotation, topic));
            }
        }
        return list;
    }

    /**
     * 校验相关配置是否合法
     *
     * @param target 消费者函数所在的对象
     * @param listenerMethod 监听的方法的
     */
    protected void checkRocketListener(Object target, Method listenerMethod) {
        // 判断消费函数参数数量必须为1
        if (listenerMethod.getParameterCount() != 1) {
            LogUtil.error(log, "ark-spring-boot-starter >> MQ消费函数参数数量必须为1,且类型必须为FsMessage、String、阿里云的Message其中之一! {}.{}", target.getClass().getName(),listenerMethod.getName());
            throw new MQConsumerException("ark-spring-boot-starter >> rocketmq消费者创建失败！");
        }
        // 校验入参类型
        Class<?> patameterType = listenerMethod.getParameterTypes()[0];
        if (ConsumerParamTypeEnum.getByValue(patameterType.getName()) == null) {
            LogUtil.error(log, "ark-spring-boot-starter >> MQ消费函数参数数量必须为1,且类型必须为FsMessage、String、阿里云的Message其中之一! {}.{}", target.getClass().getName(),listenerMethod.getName());
            throw new MQConsumerException("ark-spring-boot-starter >> rocketmq消费者创建失败！");
        }
        // 校验返回值类型
        Class<?> returnType = listenerMethod.getReturnType();
        if (ConsumerReturnTypeEnum.getByValue(returnType.getName()) == null) {
            LogUtil.error(log, "ark-spring-boot-starter >> MQ消费函数返回值类型必须为无返回值(通过异常来控制消息重新消费)或者阿里云的com.aliyun.openservices.ons.api.Action! {}.{}", target.getClass().getName(),listenerMethod.getName());
            throw new MQConsumerException("ark-spring-boot-starter >> rocketmq消费者创建失败！");
        }

    }

    /**
     * 解析消费者模型
     *
     * @param beanName
     * @param targetBean
     * @param listenerMethod
     * @param listenerAnnotation
     * @return
     */
    protected FsConsumerModel createRocketConsumerModel(String beanName, Object targetBean, Method listenerMethod, RocketListener listenerAnnotation, String topic) {
        // 校验消费者方法是否合法
        checkRocketListener(targetBean, listenerMethod);

        // 消费者模型
        FsConsumerModel fsConsumerModel = new FsConsumerModel();
        fsConsumerModel.setBeanName(beanName);
        fsConsumerModel.setListenerMethod(listenerMethod);
        fsConsumerModel.setTarget(targetBean);
        ConsumerParamTypeEnum paramTypeEnum = ConsumerParamTypeEnum.getByValue(listenerMethod.getParameterTypes()[0].getName());
        fsConsumerModel.setParamTypeEnum(paramTypeEnum);
        ConsumerReturnTypeEnum returnTypeEnum = ConsumerReturnTypeEnum.getByValue(listenerMethod.getReturnType().getName());
        fsConsumerModel.setReturnTypeEnum(returnTypeEnum);

        // 消费者mq属性
        FsConsumerModelProperties consumerProperties = new FsConsumerModelProperties();
        consumerProperties.setGroupId(ConfigUtil.getProperty(listenerAnnotation.groupId(), ctx));
        consumerProperties.setTopic(ConfigUtil.getProperty(topic, ctx));
        consumerProperties.setTags(ConfigUtil.getProperty(listenerAnnotation.tags(), ctx));
        consumerProperties.setConsumeTimeout(ConfigUtil.getProperty(listenerAnnotation.consumeTimeout(), ctx));
        consumerProperties.setMaxReconsumeTimes(ConfigUtil.getProperty(listenerAnnotation.maxReconsumeTimes(), ctx));
        consumerProperties.setConsumeThreadNums(ConfigUtil.getProperty(listenerAnnotation.consumeThreadNums(), ctx));
        consumerProperties.setCharsetName(ConfigUtil.getProperty(listenerAnnotation.msgContentCharset(), ctx));
        consumerProperties.setAccessKey(ConfigUtil.getProperty(listenerAnnotation.accessKey(), ctx));
        consumerProperties.setSecretKey(ConfigUtil.getProperty(listenerAnnotation.secretKey(), ctx));
        consumerProperties.setNamesrvAddr(ConfigUtil.getProperty(listenerAnnotation.namesrvAddr(), ctx));
        fsConsumerModel.setConsumerProperties(consumerProperties);

        // 校验属性是否缺失
        checkConsumerModelProperties(consumerProperties, fsConsumerModel, listenerAnnotation);

        // 消费者mq配置的key信息
        FsConsumerConfigKey consumerConfigKey = new FsConsumerConfigKey();
        consumerConfigKey.setGroupId(ConfigUtil.isDynamicConfigKey(listenerAnnotation.groupId()));
        consumerConfigKey.setTopic(ConfigUtil.isDynamicConfigKey(topic));
        consumerConfigKey.setTags(ConfigUtil.isDynamicConfigKey(listenerAnnotation.tags()));
        consumerConfigKey.setConsumeTimeout(ConfigUtil.isDynamicConfigKey(listenerAnnotation.consumeTimeout()));
        consumerConfigKey.setMaxReconsumeTimes(ConfigUtil.isDynamicConfigKey(listenerAnnotation.maxReconsumeTimes()));
        consumerConfigKey.setConsumeThreadNums(ConfigUtil.isDynamicConfigKey(listenerAnnotation.consumeThreadNums()));
        consumerConfigKey.setCharsetName(ConfigUtil.isDynamicConfigKey(listenerAnnotation.msgContentCharset()));
        consumerConfigKey.setAccessKey(ConfigUtil.isDynamicConfigKey(listenerAnnotation.accessKey()));
        consumerConfigKey.setSecretKey(ConfigUtil.isDynamicConfigKey(listenerAnnotation.secretKey()));
        consumerConfigKey.setNamesrvAddr(ConfigUtil.isDynamicConfigKey(listenerAnnotation.namesrvAddr()));
        fsConsumerModel.setConsumerConfigKey(consumerConfigKey);

        return fsConsumerModel;
    }

    /**
     * 校验消费者的属性配置
     */
    protected void checkConsumerModelProperties(FsConsumerModelProperties consumerProperties, FsConsumerModel fsConsumerModel, RocketListener listenerAnnotation) {
        if (StrUtil.isBlank(consumerProperties.getGroupId())) {
            LogUtil.error(log, "ark-spring-boot-starter >> mq消费者groupId不能为空！ >> {}.{}", fsConsumerModel.getBeanName(), fsConsumerModel.getListenerMethod());
            throw new MQConsumerException("ark-spring-boot-starter >> mq消费者groupId不能为空！");
        }

        if (StrUtil.isBlank(consumerProperties.getTags())) {
            LogUtil.error(log, "ark-spring-boot-starter >> mq消费者tag不能为空！ >> {}.{}", fsConsumerModel.getBeanName(), fsConsumerModel.getListenerMethod());
            throw new MQConsumerException("ark-spring-boot-starter >> mq消费者tag不能为空！");
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.ctx = applicationContext;
    }

    public void setConsumerInterceptorManagement(IConsumerInterceptorManagement consumerInterceptorManagement){
        this.consumerInterceptorManagement = consumerInterceptorManagement;
    }

    /**
     * 获取消费者实例管理容器中的所有消费者实例
     *
     * @return
     */
    public List<IFshowsConsumer> getFshowsConsumerList() {
        return new ArrayList<>(fshowsConsumerMap.values());
    }
}