/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.calcite.avatica.remote;

import com.aliyun.calcite.avatica.org.apache.http.NoHttpResponseException;
import com.aliyun.calcite.avatica.org.apache.http.client.ClientProtocolException;
import com.aliyun.calcite.avatica.org.apache.http.client.config.RequestConfig;
import com.aliyun.calcite.avatica.org.apache.http.client.methods.CloseableHttpResponse;
import com.aliyun.calcite.avatica.org.apache.http.client.methods.HttpPost;
import com.aliyun.calcite.avatica.org.apache.http.client.protocol.HttpClientContext;
import com.aliyun.calcite.avatica.org.apache.http.entity.ByteArrayEntity;
import com.aliyun.calcite.avatica.org.apache.http.entity.ContentType;
import com.aliyun.calcite.avatica.org.apache.http.util.EntityUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.util.Properties;

public class LindormHttpClient implements AvaticaHttpClient {
    private static final Logger LOG = LoggerFactory.getLogger(LindormHttpClient.class);

    private Properties connectionInfo = null;
    private AvaticaCommonsHttpClientImpl client = null;
    private final int connectTimeout;
    private final int connectionRequestTimeout;
    private final int socketTimeout;

    public LindormHttpClient(Properties connectionInfo, AvaticaCommonsHttpClientImpl client) {
        this.connectionInfo = connectionInfo;
        this.client = client;
        connectTimeout =  Integer.parseInt(connectionInfo.getProperty(HADriver.LINDORM_TSDB_DRIVER_CONNECT_TIMEOUT, HADriver.DEFAULT_LINDORM_TSDB_DRIVER_CONNECT_TIMEOUT));
        connectionRequestTimeout = Integer.parseInt(connectionInfo.getProperty(HADriver.LINDORM_TSDB_DRIVER_CONNECTION_REQUEST_TIMEOUT, HADriver.DEFAULT_LINDORM_TSDB_DRIVER_CONNECTION_REQUEST_TIMEOUT));
        socketTimeout = Integer.parseInt(connectionInfo.getProperty(HADriver.LINDORM_TSDB_DRIVER_SOCKET_TIMEOUT, HADriver.DEFAULT_LINDORM_TSDB_DRIVER_SOCKET_TIMEOUT));
    }

    public byte[] send(byte[] request) {
        while (true) {
            HttpClientContext context = HttpClientContext.create();
            context.setTargetHost(client.host);

            // Set the credentials if they were provided.
            if (null != client.credentials) {
                context.setCredentialsProvider(client.credentialsProvider);
                context.setAuthSchemeRegistry(client.authRegistry);
                context.setAuthCache(client.authCache);
            }

            ByteArrayEntity entity = new ByteArrayEntity(request, ContentType.APPLICATION_OCTET_STREAM);

            // Create the client with the AuthSchemeRegistry and manager
            HttpPost post = new HttpPost(client.uri);

            RequestConfig requestConfig = RequestConfig.custom()
                    .setConnectTimeout(connectTimeout).setConnectionRequestTimeout(connectionRequestTimeout)
                    .setSocketTimeout(socketTimeout).build();
            post.setConfig(requestConfig);
            post.setEntity(entity);

            try (CloseableHttpResponse response = execute(post, context)) {
                final int statusCode = response.getStatusLine().getStatusCode();
                if (HttpURLConnection.HTTP_OK == statusCode
                        || HttpURLConnection.HTTP_INTERNAL_ERROR == statusCode) {
                    return EntityUtils.toByteArray(response.getEntity());
                } else if (HttpURLConnection.HTTP_UNAVAILABLE == statusCode) {
                    LOG.debug("Failed to connect to server (HTTP/503), retrying");
                    // continue
                } else {
                    throw new RuntimeException("Failed to execute HTTP Request, got HTTP/" + statusCode);
                }
            } catch (NoHttpResponseException e) {
                // This can happen when sitting behind a load balancer and a backend server dies
                LOG.debug("The server failed to issue an HTTP response, retrying", e);
                // continue
            } catch (RuntimeException e) {
                throw e;
            } catch (Exception e) {
                LOG.debug("Failed to execute HTTP request", e);
                throw new RuntimeException(e);
            }
        }
    }

    CloseableHttpResponse execute(HttpPost post, HttpClientContext context)
            throws IOException, ClientProtocolException {
        return client.execute(post, context);
    }
}
