package com.taobao.spas.sdk.client.identity;

import com.taobao.spas.sdk.client.Constants;
import com.taobao.spas.sdk.common.config.ConfigConstants;
import com.taobao.spas.sdk.common.config.SpasConfigLoader;
import com.taobao.spas.sdk.common.log.SpasLogCode;
import com.taobao.spas.sdk.common.log.SpasLogger;

import java.io.*;
import java.net.URL;
import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;

/**
 * Created by shaopeng.csp on 2015/3/16.
 */
public class CredentialWatcher {

    private static final long REFRESH_INTERVAL = 10 * 1000;

    private CredentialService serviceInstance;
    private String appName;
    private String propertyPath;
    private TimerTask watcher;

    public CredentialWatcher(String appName, CredentialService serviceInstance) {
        this.appName = appName;
        this.serviceInstance = serviceInstance;
        loadCredential(true);
        watcher = new TimerTask() {
            private Timer timer = new Timer(true);
            private long modified = 0;

            {
                timer.schedule(this, REFRESH_INTERVAL, REFRESH_INTERVAL);
            }

            @Override
            public void run() {
                boolean reload = false;
                if (propertyPath == null) {
                    reload = true;
                } else {
                    File file = new File(propertyPath);
                    long lastModified = file.lastModified();
                    if (modified != lastModified) {
                        reload = true;
                        modified = lastModified;
                    }
                }
                if (reload) {
                    loadCredential(false);
                }
            }
        };
    }

    private void loadCredential(boolean init) {

        boolean logWarn = init || !serviceInstance.getCredential().valid();
        if (propertyPath == null) {
            URL url = ClassLoader.getSystemResource(Constants.PROPERTIES_FILENAME);
            if (url != null) {
                propertyPath = url.getPath();
            }
            if (propertyPath == null || propertyPath.isEmpty()) {
                if (logWarn) {
                    SpasLogger.info(appName, "Unable to load credential file from classpath: " + Constants.PROPERTIES_FILENAME);
                }
                propertyPath = SpasConfigLoader.getProperty(ConfigConstants.ENV_KEY_FILE);
                if (propertyPath == null) {
                    if (logWarn) {
                        SpasLogger.info(appName, "Undefined credential file: -D" + ConfigConstants.ENV_KEY_FILE);
                    }
                    if (appName == null) {
                        if (logWarn) {
                            SpasLogger.warn(SpasLogCode.SPAS0063, appName, "Missing property " + ConfigConstants.ENV_APP_NAME);
                        }
                        return;
                    }
                    propertyPath = Constants.CREDENTIAL_PATH + appName;
                }
            }
        }

        InputStream propertiesIS;
        try {
            propertiesIS = new FileInputStream(propertyPath);
        } catch (FileNotFoundException e) {
            if (logWarn) {
                SpasLogger.info(appName, "No credential file from path: " + propertyPath);
            }
            return;
        }

        Properties properties = new Properties();
        try {
            properties.load(propertiesIS);
        } catch (IOException e) {
            SpasLogger.error(SpasLogCode.SPAS0026, appName, "Unable to load credential file", e);
            return;
        } finally {
            try {
                propertiesIS.close();
            } catch (IOException e) {
                SpasLogger.error(SpasLogCode.SPAS0027, appName, "Unable to close credential file", e);
            }
        }

        String accessKey = null;
        String secretKey = null;
        if (properties.containsKey(Constants.ACCESS_KEY)) {
            accessKey = properties.getProperty(Constants.ACCESS_KEY);
            if (accessKey != null) {
                accessKey = accessKey.trim();
            }
        }
        if (properties.containsKey(Constants.SECRET_KEY)) {
            secretKey = properties.getProperty(Constants.SECRET_KEY);
            if (secretKey != null) {
                secretKey = secretKey.trim();
            }
        }

        Credentials credential = new Credentials(accessKey, secretKey);
        if (!credential.valid()) {
            SpasLogger.warn(SpasLogCode.SPAS0001, appName, "Credential file missing " + Constants.ACCESS_KEY + " or " + Constants.SECRET_KEY);
            //return;
        }

        serviceInstance.setCredential(credential);
    }
}
