/*
 * Decompiled with CFR 0.152.
 */
package com.lambdaworks.redis.protocol;

import com.lambdaworks.redis.ClientOptions;
import com.lambdaworks.redis.ConnectionEvents;
import com.lambdaworks.redis.RedisChannelHandler;
import com.lambdaworks.redis.internal.LettuceAssert;
import com.lambdaworks.redis.protocol.ChannelLogDescriptor;
import com.lambdaworks.redis.protocol.CommandHandler;
import com.lambdaworks.redis.protocol.ReconnectionHandler;
import com.lambdaworks.redis.protocol.ReconnectionListener;
import com.lambdaworks.redis.resource.Delay;
import io.netty.bootstrap.Bootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.util.Timeout;
import io.netty.util.Timer;
import io.netty.util.TimerTask;
import io.netty.util.concurrent.EventExecutorGroup;
import io.netty.util.internal.logging.InternalLogLevel;
import io.netty.util.internal.logging.InternalLogger;
import io.netty.util.internal.logging.InternalLoggerFactory;
import java.net.SocketAddress;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;

@ChannelHandler.Sharable
public class ConnectionWatchdog
extends ChannelInboundHandlerAdapter
implements TimerTask {
    public static final long LOGGING_QUIET_TIME_MS = TimeUnit.MILLISECONDS.convert(5L, TimeUnit.SECONDS);
    private static final InternalLogger logger = InternalLoggerFactory.getInstance(ConnectionWatchdog.class);
    private final Delay reconnectDelay;
    private final Bootstrap bootstrap;
    private final EventExecutorGroup reconnectWorkers;
    private final ReconnectionHandler reconnectionHandler;
    private final ReconnectionListener reconnectionListener;
    private Channel channel;
    private final Timer timer;
    private SocketAddress remoteAddress;
    private long lastReconnectionLogging = -1L;
    private CommandHandler<?, ?> commandHandler;
    private final AtomicBoolean reconnectSchedulerSync;
    private volatile int attempts;
    private volatile boolean listenOnChannelInactive;
    private volatile Timeout reconnectScheduleTimeout;
    private volatile String logPrefix;

    public ConnectionWatchdog(Delay reconnectDelay, ClientOptions clientOptions, Bootstrap bootstrap, Timer timer, EventExecutorGroup reconnectWorkers, final Supplier<SocketAddress> socketAddressSupplier, ReconnectionListener reconnectionListener) {
        LettuceAssert.notNull(reconnectDelay, "Delay must not be null");
        LettuceAssert.notNull(clientOptions, "ClientOptions must not be null");
        LettuceAssert.notNull(bootstrap, "Bootstrap must not be null");
        LettuceAssert.notNull(timer, "Timer must not be null");
        LettuceAssert.notNull(reconnectWorkers, "ReconnectWorkers must not be null");
        LettuceAssert.notNull(reconnectionListener, "ReconnectionListener must not be null");
        this.reconnectDelay = reconnectDelay;
        this.bootstrap = bootstrap;
        this.timer = timer;
        this.reconnectWorkers = reconnectWorkers;
        this.reconnectionListener = reconnectionListener;
        this.reconnectSchedulerSync = new AtomicBoolean(false);
        Supplier<SocketAddress> wrappedSocketAddressSupplier = new Supplier<SocketAddress>(){

            @Override
            public SocketAddress get() {
                if (socketAddressSupplier != null) {
                    try {
                        ConnectionWatchdog.this.remoteAddress = (SocketAddress)socketAddressSupplier.get();
                    }
                    catch (RuntimeException e) {
                        logger.warn("Cannot retrieve the current address from socketAddressSupplier: " + e.toString() + ", reusing old address " + ConnectionWatchdog.this.remoteAddress);
                    }
                }
                return ConnectionWatchdog.this.remoteAddress;
            }
        };
        this.reconnectionHandler = new ReconnectionHandler(clientOptions, bootstrap, wrappedSocketAddressSupplier, timer, (ExecutorService)reconnectWorkers);
        this.resetReconnectDelay();
    }

    private void resetReconnectDelay() {
        if (this.reconnectDelay instanceof Delay.StatefulDelay) {
            ((Delay.StatefulDelay)((Object)this.reconnectDelay)).reset();
        }
    }

    public void userEventTriggered(ChannelHandlerContext ctx, Object evt) throws Exception {
        logger.debug("{} userEventTriggered({}, {})", new Object[]{this.logPrefix(), ctx, evt});
        if (evt instanceof ConnectionEvents.PrepareClose) {
            ConnectionEvents.PrepareClose prepareClose = (ConnectionEvents.PrepareClose)evt;
            this.prepareClose(prepareClose);
        }
        if (evt instanceof ConnectionEvents.Activated) {
            this.attempts = 0;
            this.resetReconnectDelay();
        }
        super.userEventTriggered(ctx, evt);
    }

    void prepareClose(ConnectionEvents.PrepareClose prepareClose) {
        this.setListenOnChannelInactive(false);
        this.setReconnectSuspended(true);
        prepareClose.getPrepareCloseFuture().complete(true);
        Timeout reconnectScheduleTimeout = this.reconnectScheduleTimeout;
        if (reconnectScheduleTimeout != null && !reconnectScheduleTimeout.isCancelled()) {
            reconnectScheduleTimeout.cancel();
        }
        this.reconnectionHandler.prepareClose();
    }

    public void channelActive(ChannelHandlerContext ctx) throws Exception {
        if (this.commandHandler == null) {
            this.commandHandler = (CommandHandler)ctx.pipeline().get(CommandHandler.class);
        }
        this.reconnectSchedulerSync.set(false);
        this.reconnectScheduleTimeout = null;
        this.logPrefix = null;
        this.channel = ctx.channel();
        this.remoteAddress = this.channel.remoteAddress();
        logger.debug("{} channelActive({})", (Object)this.logPrefix(), (Object)ctx);
        super.channelActive(ctx);
    }

    public void channelInactive(ChannelHandlerContext ctx) throws Exception {
        logger.debug("{} channelInactive({})", (Object)this.logPrefix(), (Object)ctx);
        this.channel = null;
        if (this.listenOnChannelInactive && !this.reconnectionHandler.isReconnectSuspended()) {
            RedisChannelHandler channelHandler = (RedisChannelHandler)ctx.pipeline().get(RedisChannelHandler.class);
            if (channelHandler != null) {
                this.reconnectionHandler.setTimeout(channelHandler.getTimeout());
                this.reconnectionHandler.setTimeoutUnit(channelHandler.getTimeoutUnit());
            }
            this.scheduleReconnect();
        } else {
            logger.debug("{} Reconnect scheduling disabled", (Object)this.logPrefix(), (Object)ctx);
        }
        super.channelInactive(ctx);
    }

    public void scheduleReconnect() {
        logger.debug("{} scheduleReconnect()", (Object)this.logPrefix());
        if (!this.isEventLoopGroupActive()) {
            logger.debug("isEventLoopGroupActive() == false");
            return;
        }
        if (this.commandHandler != null && this.commandHandler.isClosed()) {
            logger.debug("Skip reconnect scheduling, CommandHandler is closed");
            return;
        }
        if ((this.channel == null || !this.channel.isActive()) && this.reconnectSchedulerSync.compareAndSet(false, true)) {
            ++this.attempts;
            int attempt = this.attempts;
            int timeout = (int)this.reconnectDelay.getTimeUnit().toMillis(this.reconnectDelay.createDelay(attempt));
            logger.debug("Reconnect attempt {}, delay {}ms", (Object)attempt, (Object)timeout);
            this.reconnectScheduleTimeout = this.timer.newTimeout(it -> {
                this.reconnectScheduleTimeout = null;
                if (!this.isEventLoopGroupActive()) {
                    logger.warn("Cannot execute scheduled reconnect timer, reconnect workers are terminated");
                    return;
                }
                this.reconnectWorkers.submit(() -> {
                    this.run(it);
                    return null;
                });
            }, (long)timeout, TimeUnit.MILLISECONDS);
            if (!this.reconnectSchedulerSync.get()) {
                this.reconnectScheduleTimeout = null;
            }
        } else {
            logger.debug("{} Skipping scheduleReconnect() because I have an active channel", (Object)this.logPrefix());
        }
    }

    public void run(Timeout timeout) throws Exception {
        this.reconnectSchedulerSync.set(false);
        this.reconnectScheduleTimeout = null;
        if (!this.isEventLoopGroupActive()) {
            logger.debug("isEventLoopGroupActive() == false");
            return;
        }
        if (this.commandHandler != null && this.commandHandler.isClosed()) {
            logger.debug("Skip reconnect scheduling, CommandHandler is closed");
            return;
        }
        if (this.isReconnectSuspended()) {
            logger.debug("Skip reconnect scheduling, reconnect is suspended");
            return;
        }
        boolean shouldLog = this.shouldLog();
        InternalLogLevel infoLevel = InternalLogLevel.INFO;
        InternalLogLevel warnLevel = InternalLogLevel.WARN;
        if (shouldLog) {
            this.lastReconnectionLogging = System.currentTimeMillis();
        } else {
            warnLevel = InternalLogLevel.DEBUG;
            infoLevel = InternalLogLevel.DEBUG;
        }
        InternalLogLevel warnLevelToUse = warnLevel;
        try {
            this.reconnectionListener.onReconnect(new ConnectionEvents.Reconnect(this.attempts));
            logger.log(infoLevel, "Reconnecting, last destination was {}", (Object)this.remoteAddress);
            ChannelFuture future = this.reconnectionHandler.reconnect();
            future.addListener(it -> {
                if (it.isSuccess() || it.cause() == null) {
                    return;
                }
                Throwable throwable = it.cause();
                if (ReconnectionHandler.isExecutionException(throwable)) {
                    logger.log(warnLevelToUse, "Cannot reconnect: {}", (Object)throwable.toString());
                } else {
                    logger.log(warnLevelToUse, "Cannot reconnect: {}", (Object)throwable.toString(), (Object)throwable);
                }
                if (!this.isReconnectSuspended()) {
                    this.scheduleReconnect();
                }
            });
        }
        catch (Exception e) {
            logger.log(warnLevel, "Cannot reconnect: {}", (Object)e.toString());
        }
    }

    private boolean isEventLoopGroupActive() {
        return ConnectionWatchdog.isEventLoopGroupActive((EventExecutorGroup)this.bootstrap.group()) && ConnectionWatchdog.isEventLoopGroupActive(this.reconnectWorkers);
    }

    private static boolean isEventLoopGroupActive(EventExecutorGroup executorService) {
        return !executorService.isShuttingDown();
    }

    private boolean shouldLog() {
        long quietUntil = this.lastReconnectionLogging + LOGGING_QUIET_TIME_MS;
        return quietUntil <= System.currentTimeMillis();
    }

    public void setListenOnChannelInactive(boolean listenOnChannelInactive) {
        this.listenOnChannelInactive = listenOnChannelInactive;
    }

    public boolean isListenOnChannelInactive() {
        return this.listenOnChannelInactive;
    }

    public boolean isReconnectSuspended() {
        return this.reconnectionHandler.isReconnectSuspended();
    }

    public void setReconnectSuspended(boolean reconnectSuspended) {
        this.reconnectionHandler.setReconnectSuspended(reconnectSuspended);
    }

    ReconnectionHandler getReconnectionHandler() {
        return this.reconnectionHandler;
    }

    private String logPrefix() {
        String buffer;
        if (this.logPrefix != null) {
            return this.logPrefix;
        }
        this.logPrefix = buffer = "[" + ChannelLogDescriptor.logDescriptor(this.channel) + ", last known addr=" + this.remoteAddress + ']';
        return this.logPrefix;
    }
}

