/*
 * Decompiled with CFR 0.152.
 */
package org.apache.dubbo.common.cache;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.channels.FileLock;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import org.apache.dubbo.common.cache.FileCacheStoreFactory;
import org.apache.dubbo.common.logger.Logger;
import org.apache.dubbo.common.logger.LoggerFactory;
import org.apache.dubbo.common.utils.CollectionUtils;

public class FileCacheStore {
    private static final Logger logger = LoggerFactory.getLogger(FileCacheStore.class);
    private String cacheFilePath;
    private File cacheFile;
    private File lockFile;
    private FileLock directoryLock;

    private FileCacheStore(String cacheFilePath, File cacheFile, File lockFile, FileLock directoryLock) {
        this.cacheFilePath = cacheFilePath;
        this.cacheFile = cacheFile;
        this.lockFile = lockFile;
        this.directoryLock = directoryLock;
    }

    public synchronized Map<String, String> loadCache(int entrySize) throws IOException {
        HashMap<String, String> properties = new HashMap<String, String>();
        try (BufferedReader reader = new BufferedReader(new FileReader(this.cacheFile));){
            int count = 1;
            String line = reader.readLine();
            while (line != null && count <= entrySize) {
                if (!line.startsWith("#") && line.contains("=")) {
                    String[] pairs = line.split("=");
                    properties.put(pairs[0], pairs[1]);
                    ++count;
                }
                line = reader.readLine();
            }
            if (count > entrySize) {
                logger.warn("Cache file was truncated for exceeding the maximum entry size " + entrySize);
            }
        }
        catch (IOException e) {
            logger.warn("Load cache failed ", e);
            throw e;
        }
        return properties;
    }

    private void unlock() {
        if (this.directoryLock != null && this.directoryLock.isValid()) {
            try {
                this.directoryLock.release();
                this.directoryLock.channel().close();
                FileCacheStore.deleteFile(this.lockFile);
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to release cache path's lock file:" + this.lockFile, e);
            }
        }
    }

    public synchronized void refreshCache(Map<String, String> properties, String comment, long maxFileSize) {
        if (CollectionUtils.isEmptyMap(properties)) {
            return;
        }
        try (LimitedLengthBufferedWriter bw = new LimitedLengthBufferedWriter((Writer)new OutputStreamWriter((OutputStream)new FileOutputStream(this.cacheFile, false), StandardCharsets.UTF_8), maxFileSize);){
            bw.write("#" + comment);
            bw.newLine();
            bw.write("#" + new Date());
            bw.newLine();
            for (Map.Entry<String, String> e : properties.entrySet()) {
                String key = e.getKey();
                String val = e.getValue();
                bw.write(key + "=" + val);
                bw.newLine();
            }
            bw.flush();
            long remainSize = bw.getRemainSize();
            if (remainSize < 0L) {
                logger.info("Cache file was truncated for exceeding the maximum file size " + maxFileSize + " byte. Exceeded by " + -remainSize + " byte.");
            }
        }
        catch (IOException e) {
            logger.warn("Update cache error.");
        }
    }

    private static void deleteFile(File f) {
        if (!f.delete()) {
            logger.debug("Failed to delete file " + f.getAbsolutePath());
        }
    }

    public synchronized void destroy() {
        this.unlock();
        FileCacheStoreFactory.removeCache(this.cacheFilePath);
    }

    @Deprecated
    protected String getCacheFilePath() {
        return this.cacheFilePath;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    private static class LimitedLengthBufferedWriter
    extends BufferedWriter {
        private long remainSize;

        public LimitedLengthBufferedWriter(Writer out, long maxSize) {
            super(out);
            this.remainSize = maxSize == 0L ? Long.MAX_VALUE : maxSize;
        }

        @Override
        public void write(String str) throws IOException {
            this.remainSize -= (long)str.getBytes(StandardCharsets.UTF_8).length;
            if (this.remainSize < 0L) {
                return;
            }
            super.write(str);
        }

        public long getRemainSize() {
            return this.remainSize;
        }
    }

    protected static class Empty
    extends FileCacheStore {
        private Empty(String cacheFilePath) {
            super(cacheFilePath, null, null, null);
        }

        public static Empty getInstance(String cacheFilePath) {
            return new Empty(cacheFilePath);
        }

        @Override
        public Map<String, String> loadCache(int entrySize) throws IOException {
            return Collections.emptyMap();
        }

        @Override
        public void refreshCache(Map<String, String> properties, String comment, long maxFileSize) {
        }
    }

    public static class Builder {
        private String cacheFilePath;
        private File cacheFile;
        private File lockFile;
        private FileLock directoryLock;

        private Builder() {
        }

        public Builder cacheFilePath(String cacheFilePath) {
            this.cacheFilePath = cacheFilePath;
            return this;
        }

        public Builder cacheFile(File cacheFile) {
            this.cacheFile = cacheFile;
            return this;
        }

        public Builder lockFile(File lockFile) {
            this.lockFile = lockFile;
            return this;
        }

        public Builder directoryLock(FileLock directoryLock) {
            this.directoryLock = directoryLock;
            return this;
        }

        public FileCacheStore build() {
            return new FileCacheStore(this.cacheFilePath, this.cacheFile, this.lockFile, this.directoryLock);
        }
    }
}

