/*
 * Decompiled with CFR 0.152.
 */
package com.intellij.util.io;

import com.intellij.openapi.diagnostic.Logger;
import com.intellij.util.io.PagedFileStorage;
import com.intellij.util.io.RandomAccessPagedDataInput;
import com.intellij.util.io.RecordDataOutput;
import gnu.trove.TIntArrayList;
import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.jetbrains.annotations.NonNls;

public class PagedMemoryMappedFile {
    private static final Logger LOG = Logger.getInstance("#com.intellij.util.io.PagedMemoryMappedFile");
    private final int START_FILE_SIZE;
    private final int CLUSTER_SIZE;
    private short COMPACTED_FILE_FREEPAGE_SIZE = (short)64;
    private static final int PAGE_LINK_OFFSET = 0;
    private static final int CLUSTERS_COUNT_OFFSET = 4;
    private static final int BYTE_COUNT_OFFSET = 6;
    public static final int HEADER_SIZE = 7;
    private static final int MAX_CLUSTERS_PER_PAGE = 32000;
    private static final int DEFFERED_FORCE_DELAY = 1000;
    private static final ScheduledExecutorService ourForceAlarm = Executors.newSingleThreadScheduledExecutor(new ThreadFactory(){

        public Thread newThread(Runnable r) {
            Thread t = new Thread(r);
            t.setName("PagedMemoryMappedFile flusher");
            return t;
        }
    });
    private final File myFile;
    private final PagedFileStorage myStorage;
    private boolean myIsTestMode;
    private long myFileLength;
    private boolean myIsDisposed = false;
    private volatile Runnable myActiveFlushRequest;
    private volatile int myModificationCount = 0;
    private PagedDataOutputStream myOutputPool = null;
    private RandomAccessPagedDataInput myInputPool = null;

    public PagedMemoryMappedFile(File file, int initialSize, int pageSize) throws IOException {
        this.START_FILE_SIZE = Math.min(initialSize, pageSize * 32000);
        this.CLUSTER_SIZE = pageSize;
        this.myFile = file;
        if (!file.exists()) {
            file.getParentFile().mkdirs();
            file.createNewFile();
        }
        this.myStorage = new PagedFileStorage(file);
        this.remap(false);
        this.myIsTestMode = false;
    }

    public PagedMemoryMappedFile(File file) throws IOException {
        this(file, 1024, 32);
    }

    public void setCompactedFileFreePageSize(short freePageSize) {
        this.COMPACTED_FILE_FREEPAGE_SIZE = freePageSize;
    }

    void setTestMode(boolean testMode) {
        this.myIsTestMode = testMode;
    }

    private void remap(boolean extend) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Remapping: " + this.myFile.getAbsolutePath());
        }
        int length = (int)this.myStorage.length();
        int freePage = length / this.CLUSTER_SIZE;
        int increment = 0;
        if (length == 0) {
            increment = this.START_FILE_SIZE;
        } else if (extend) {
            increment = Math.min(length, 131072);
        }
        int newSize = length + increment;
        this.myStorage.resize(newSize);
        this.myFileLength = newSize;
        if (increment > 0) {
            if (freePage == 0) {
                freePage = 1;
                --increment;
            }
            this.addFreePage(freePage, (short)(increment / this.CLUSTER_SIZE));
        }
    }

    public void compact() throws IOException {
        int freePage = this.getFirstFreePage();
        while (freePage > 0 && !this.isLastPage(freePage)) {
            freePage = this.getNextPageLink(freePage);
        }
        if (freePage > 0 && this.getClustersCount(freePage) > this.COMPACTED_FILE_FREEPAGE_SIZE) {
            int next = this.getNextPageLink(freePage);
            int prev = this.getPrevPageLink(freePage);
            this.setClustersCount(freePage, this.COMPACTED_FILE_FREEPAGE_SIZE);
            this.markPageFree(freePage, true);
            this.linkPages(freePage, next);
            this.linkPages(prev, freePage);
            this.myFileLength = (freePage + this.COMPACTED_FILE_FREEPAGE_SIZE) * this.CLUSTER_SIZE;
            this.myStorage.resize((int)this.myFileLength);
        }
    }

    private boolean isLastPage(int page) {
        return this.getNextSiblingPage(page) == -1;
    }

    public RecordDataOutput createRecord() {
        return this.getWriter(this.getFreePage());
    }

    public RandomAccessPagedDataInput getReader(int page) {
        RandomAccessPagedDataInput pooled = this.myInputPool;
        if (pooled != null) {
            this.myInputPool = null;
            pooled.setup(page);
            return pooled;
        }
        return new RandomAccessPagedDataInput(new PagedInputStream(page));
    }

    public void returnToPool(RandomAccessPagedDataInput input) {
        this.myInputPool = input;
    }

    private void returnToPool(PagedDataOutputStream stream) {
        this.myOutputPool = stream;
    }

    public RecordDataOutput getWriter(int page) {
        PagedDataOutputStream pooled = this.myOutputPool;
        if (pooled != null) {
            this.myOutputPool = null;
            pooled.setup(page);
            return pooled;
        }
        return new PagedDataOutputStream(page);
    }

    public void delete(int page) {
        PagedMemoryMappedFile.assertTrue(page > 0);
        this.uncheckedDelete(page);
        this.checkConsistency(false);
    }

    private void uncheckedDelete(int page) {
        if (page == 0) {
            return;
        }
        PagedMemoryMappedFile.assertTrue(!this.isPageFree(page));
        int next = this.getNextPageLink(page);
        this.deleteFromList(page);
        this.addFreePage(page, this.getClustersCount(page));
        this.uncheckedDelete(next);
    }

    private int getPrevSiblingPage(int page) {
        int prevSublingPage = -1;
        if (page > 1) {
            prevSublingPage = page - this.myStorage.getShort(page * this.CLUSTER_SIZE - 7 + 4);
        }
        return prevSublingPage;
    }

    private int getNextSiblingPage(int page) {
        int nextSublingPage = page + this.getClustersCount(page);
        if ((long)(nextSublingPage * this.CLUSTER_SIZE) >= this.myFileLength) {
            nextSublingPage = -1;
        }
        return nextSublingPage;
    }

    private synchronized int getFreePage() {
        int page = this.getFirstFreePage();
        if (page == 0) {
            try {
                this.remap(true);
            }
            catch (IOException e) {
                LOG.error(e);
            }
            return this.getFreePage();
        }
        this.deleteFromList(page);
        return page;
    }

    private short getClustersCount(int page) {
        short count = this.myStorage.getShort(page * this.CLUSTER_SIZE + 4);
        PagedMemoryMappedFile.assertTrue(count > 0);
        return count;
    }

    private void setClustersCount(int page, short count) {
        PagedMemoryMappedFile.assertTrue(page > 0);
        PagedMemoryMappedFile.assertTrue(count > 0 && count < 32000);
        this.myStorage.putShort(page * this.CLUSTER_SIZE + 4, count);
        this.myStorage.putShort((page + count) * this.CLUSTER_SIZE - 7 + 4, count);
    }

    private int getPrevPageLink(int page) {
        return this.myStorage.getInt(page * this.CLUSTER_SIZE + 0);
    }

    private int getNextPageLink(int page) {
        return this.myStorage.getInt((page + this.getClustersCount(page)) * this.CLUSTER_SIZE - 7 + 0);
    }

    private void linkPages(int prevPage, int nextPage) {
        PagedMemoryMappedFile.assertTrue(prevPage == 0 || prevPage != nextPage);
        if (nextPage != 0) {
            this.myStorage.putInt(nextPage * this.CLUSTER_SIZE + 0, prevPage);
            PagedMemoryMappedFile.assertTrue(prevPage == 0 || this.getNextPageLink(nextPage) != prevPage);
        }
        if (prevPage != 0) {
            int rightSide = prevPage + this.getClustersCount(prevPage);
            this.myStorage.putInt(rightSide * this.CLUSTER_SIZE - 7 + 0, nextPage);
            if (nextPage != 0 && nextPage > prevPage && nextPage < rightSide) {
                LOG.error("nextPage = " + nextPage + ", prevPage = " + prevPage + ", rightSide = " + rightSide);
                PagedMemoryMappedFile.assertTrue(false);
            }
        }
    }

    private void closePage(int page, int bytesWritten) {
        this.splitPage(page, bytesWritten);
        this.markPageFree(page, false);
    }

    private void splitPage(int page, int bytesWritten) {
        short oldClustersCount = this.getClustersCount(page);
        if (oldClustersCount > 1 && bytesWritten + 14 < oldClustersCount * this.CLUSTER_SIZE) {
            short newClustersCount = (short)((bytesWritten + 14) / this.CLUSTER_SIZE);
            if ((bytesWritten + 14) % this.CLUSTER_SIZE > 0) {
                newClustersCount = (short)(newClustersCount + 1);
            }
            if (oldClustersCount > newClustersCount) {
                this.setClustersCount(page, newClustersCount);
                this.markPageFree(page, false);
                int newPage = page + newClustersCount;
                this.setClustersCount(newPage, (short)(oldClustersCount - newClustersCount));
                this.linkPages(newPage, 0);
                this.linkPages(0, newPage);
                this.addFreePage(newPage, (short)(oldClustersCount - newClustersCount));
            }
        }
    }

    private void deleteFromList(int page) {
        PagedMemoryMappedFile.assertTrue(page > 0);
        int prev = this.getPrevPageLink(page);
        int next = this.getNextPageLink(page);
        if (this.getFirstFreePage() == page) {
            PagedMemoryMappedFile.assertTrue(prev == 0);
            this.setFirstFreePage(next);
        }
        this.linkPages(prev, next);
        this.linkPages(0, page);
        this.linkPages(page, 0);
    }

    private void mergeWithNext(int page) {
        int nextSubling = this.getNextSiblingPage(page);
        PagedMemoryMappedFile.assertTrue(nextSubling != -1 && this.isPageFree(nextSubling));
        this.deleteFromList(page);
        int prev = this.getPrevPageLink(nextSubling);
        int next = this.getNextPageLink(nextSubling);
        short count = this.getClustersCount(nextSubling);
        count = (short)(count + this.getClustersCount(page));
        this.setClustersCount(page, count);
        if (this.getFirstFreePage() == nextSubling) {
            PagedMemoryMappedFile.assertTrue(prev == 0);
            this.setFirstFreePage(page);
        }
        this.linkPages(prev, page);
        this.linkPages(page, next);
    }

    private void mergeWithPrev(int page) {
        boolean canMergeAgain;
        int prevSubling = this.getPrevSiblingPage(page);
        PagedMemoryMappedFile.assertTrue(prevSubling != -1 && this.isPageFree(prevSubling));
        this.deleteFromList(page);
        int prev = this.getPrevPageLink(prevSubling);
        int next = this.getNextPageLink(prevSubling);
        short count = this.getClustersCount(prevSubling);
        count = (short)(count + this.getClustersCount(page));
        this.setClustersCount(prevSubling, count);
        if (this.getFirstFreePage() == prevSubling) {
            PagedMemoryMappedFile.assertTrue(prev == 0);
            this.setFirstFreePage(prevSubling);
        }
        this.linkPages(prev, prevSubling);
        boolean bl = canMergeAgain = this.getPrevSiblingPage(next) == prevSubling && count + this.getClustersCount(next) < 32000;
        if (!canMergeAgain) {
            this.linkPages(prevSubling, next);
        } else {
            this.mergeWithPrev(next);
        }
    }

    private synchronized void addFreePage(int page, short count) {
        int nextSubling;
        if (count == 0) {
            return;
        }
        this.setClustersCount(page, count);
        this.markPageFree(page, true);
        int prevSubling = this.getPrevSiblingPage(page);
        if (prevSubling != -1 && this.isPageFree(prevSubling) && this.getClustersCount(page) + this.getClustersCount(prevSubling) < 32000) {
            this.mergeWithPrev(page);
            page = prevSubling;
            this.markPageFree(page, true);
        }
        if ((nextSubling = this.getNextSiblingPage(page)) != -1 && this.isPageFree(nextSubling) && this.getClustersCount(page) + this.getClustersCount(nextSubling) < 32000) {
            this.mergeWithNext(page);
            this.markPageFree(page, true);
        }
        if (this.getPrevPageLink(page) == 0 && page != this.getFirstFreePage()) {
            this.linkPages(page, this.getFirstFreePage());
            this.setFirstFreePage(page);
            this.markPageFree(page, true);
        }
    }

    private int getFirstFreePage() {
        return this.myStorage.getInt(0);
    }

    private void setFirstFreePage(int page) {
        this.myStorage.putInt(0, page);
    }

    public int getVersion() {
        return this.myStorage.getInt(4);
    }

    public void setVersion(int ver) {
        this.myStorage.putInt(4, ver);
    }

    private void markPageFree(int page, boolean isFree) {
        PagedMemoryMappedFile.assertTrue(page > 0);
        this.myStorage.put(page * this.CLUSTER_SIZE + 6, (byte)(!isFree ? 1 : 0));
        this.myStorage.put((page + this.getClustersCount(page)) * this.CLUSTER_SIZE - 7 + 6, (byte)(!isFree ? 1 : 0));
    }

    public boolean isPageFree(int page) {
        return this.myStorage.get(page * this.CLUSTER_SIZE + 6) == 0;
    }

    public void dispose() {
        this.myIsDisposed = true;
        this.myStorage.close();
    }

    private void checkConsistency(boolean shouldBeFree) {
        if (this.myIsTestMode) {
            int i;
            boolean[] map = new boolean[(int)(this.myFileLength / (long)this.CLUSTER_SIZE)];
            Arrays.fill(map, false);
            int firstFreePage = this.getFirstFreePage();
            if (firstFreePage != 0) {
                this.checkTraverseLinks(firstFreePage, map, true, new TIntArrayList());
            }
            for (i = 1; i < map.length; ++i) {
                boolean b = map[i];
                if (b) continue;
                PagedMemoryMappedFile.assertTrue(!shouldBeFree);
                this.checkTraverseLinks(i, map, false, new TIntArrayList());
            }
            for (i = 1; i < map.length; ++i) {
                PagedMemoryMappedFile.assertTrue(map[i]);
            }
        }
    }

    private void checkTraverseLinks(int page, boolean[] map, boolean shouldBeFree, TIntArrayList stack) {
        if (stack.indexOf(page) >= 0) {
            LOG.error("Short circute.");
            return;
        }
        int clustersCount = this.getClustersCount(page);
        for (int i = 0; i < clustersCount; ++i) {
            map[page + i] = true;
        }
        PagedMemoryMappedFile.assertTrue(this.isPageFree(page) == shouldBeFree);
        int next = this.getNextPageLink(page);
        if (next == 0) {
            return;
        }
        int nextSibling = this.getNextSiblingPage(page);
        PagedMemoryMappedFile.assertTrue(nextSibling != next || this.getClustersCount(page) + this.getClustersCount(nextSibling) >= 32000);
        int prevSibling = this.getPrevSiblingPage(page);
        int prev = this.getPrevPageLink(page);
        PagedMemoryMappedFile.assertTrue(prevSibling != prev || this.getClustersCount(page) + this.getClustersCount(prevSibling) >= 32000);
        PagedMemoryMappedFile.assertTrue(this.getPrevPageLink(next) == page);
        stack.add(page);
        this.checkTraverseLinks(next, map, shouldBeFree, stack);
        stack.remove(stack.size() - 1);
    }

    public void immediateForce() {
        try {
            this.myStorage.flush();
        }
        catch (Exception e) {
            this.deferredForce();
        }
    }

    public void deferredForce() {
        final int modCount = ++this.myModificationCount;
        if (this.myActiveFlushRequest == null) {
            Runnable request;
            this.myActiveFlushRequest = request = new Runnable(){

                public void run() {
                    if (PagedMemoryMappedFile.this.myIsDisposed) {
                        return;
                    }
                    if (modCount == PagedMemoryMappedFile.this.myModificationCount) {
                        PagedMemoryMappedFile.this.immediateForce();
                        PagedMemoryMappedFile.this.myActiveFlushRequest = null;
                    } else {
                        PagedMemoryMappedFile.this.myActiveFlushRequest = null;
                        PagedMemoryMappedFile.this.deferredForce();
                    }
                }
            };
            ourForceAlarm.schedule(request, 1000L, TimeUnit.MILLISECONDS);
        }
    }

    public void assertFree() {
        this.checkConsistency(true);
    }

    private static void assertTrue(boolean b) {
        PagedMemoryMappedFile.assertTrue(b, "");
    }

    private static void assertTrue(boolean b, @NonNls String message) {
        if (!b) {
            throw new PagedFileAssertionException(message);
        }
    }

    private static class PagedFileAssertionException
    extends RuntimeException {
        private PagedFileAssertionException(String message) {
            super(message);
        }
    }

    private class PagedDataOutputStream
    extends DataOutputStream
    implements RecordDataOutput {
        private int myStartPage;

        public PagedDataOutputStream(int startPage) {
            super(new PagedOutputStream(startPage));
            this.myStartPage = startPage;
        }

        public void setup(int startPage) {
            this.myStartPage = startPage;
            ((PagedOutputStream)this.out).setup(startPage);
        }

        public int getRecordId() {
            return this.myStartPage;
        }

        public void close() throws IOException {
            super.close();
            PagedMemoryMappedFile.this.checkConsistency(false);
            PagedMemoryMappedFile.this.returnToPool(this);
        }
    }

    class PagedInputStream
    extends InputStream {
        private int myCurrentPage;
        private int myCursor;
        private int myPageStart;
        private int myPosition;
        private int myPageEnd;

        public PagedInputStream(int page) {
            this.setup(page);
        }

        public void setup(int page) {
            PagedMemoryMappedFile.assertTrue(page > 0);
            this.myCurrentPage = page;
            this.setCursor(this.myCurrentPage);
            this.myPosition = 0;
        }

        private void setCursor(int page) {
            if (page <= 0) {
                PagedMemoryMappedFile.assertTrue(page > 0);
            }
            this.myPageStart = this.myCursor = page * PagedMemoryMappedFile.this.CLUSTER_SIZE + 7;
            this.myPageEnd = (page + PagedMemoryMappedFile.this.getClustersCount(page)) * PagedMemoryMappedFile.this.CLUSTER_SIZE - 7;
            PagedMemoryMappedFile.assertTrue(this.myPageStart < this.myPageEnd);
            PagedMemoryMappedFile.assertTrue(this.myPageStart > 0);
        }

        public int getPosition() {
            return this.myPosition;
        }

        public void setPosition(int pos) {
            int delta;
            while ((delta = pos - this.myPosition) != 0) {
                int posDelta;
                PagedMemoryMappedFile.assertTrue(this.myCurrentPage > 0);
                if (delta > 0 && this.myCursor + delta <= this.myPageEnd || delta < 0 && this.myCursor + delta >= this.myPageStart) {
                    this.myCursor += delta;
                    this.myPosition = pos;
                    return;
                }
                if (delta > 0) {
                    posDelta = this.myPageEnd - this.myCursor;
                    this.goNextPage();
                    this.myPosition += posDelta;
                    continue;
                }
                posDelta = this.myCursor - this.myPageStart;
                this.goPrevPage();
                this.myPosition -= posDelta + (this.myPageEnd - this.myPageStart);
            }
            return;
        }

        public long skip(long n) throws IOException {
            this.setPosition(this.myPosition + (int)n);
            return n;
        }

        public int read() throws IOException {
            if (this.myCursor >= this.myPageEnd) {
                this.goNextPage();
            }
            byte b = PagedMemoryMappedFile.this.myStorage.get(this.myCursor);
            ++this.myCursor;
            ++this.myPosition;
            return 0xFF & b;
        }

        public int read(byte[] b, int off, int len) throws IOException {
            int stopOffset = off + len;
            while (off < stopOffset) {
                if (this.myCursor >= this.myPageEnd) {
                    this.goNextPage();
                }
                int bufferSizeLeft = stopOffset - off;
                int pageSizeLeft = this.myPageEnd - this.myCursor;
                int dataToRead = Math.min(bufferSizeLeft, pageSizeLeft);
                PagedMemoryMappedFile.this.myStorage.get(this.myCursor, b, off, dataToRead);
                this.myCursor += dataToRead;
                this.myPosition += dataToRead;
                off += dataToRead;
            }
            PagedMemoryMappedFile.assertTrue(off == stopOffset);
            return len;
        }

        private void goNextPage() {
            this.myCurrentPage = PagedMemoryMappedFile.this.getNextPageLink(this.myCurrentPage);
            this.setCursor(this.myCurrentPage);
        }

        private void goPrevPage() {
            this.myCurrentPage = PagedMemoryMappedFile.this.getPrevPageLink(this.myCurrentPage);
            this.setCursor(this.myCurrentPage);
        }

        PagedMemoryMappedFile getOwnerFile() {
            return PagedMemoryMappedFile.this;
        }
    }

    private class PagedOutputStream
    extends OutputStream {
        private int myCurrentPage;
        private int myCursor;
        private int myCursorLimit;
        private int myLocalCounter;

        public PagedOutputStream(int startPage) {
            this.setup(startPage);
        }

        private void setup(int startPage) {
            PagedMemoryMappedFile.assertTrue(startPage > 0);
            this.myCurrentPage = startPage;
            this.setCursor(startPage);
        }

        public void write(int b) throws IOException {
            if (this.myCursor >= this.myCursorLimit) {
                this.allocatePage();
            }
            PagedMemoryMappedFile.this.myStorage.put(this.myCursor, (byte)b);
            ++this.myLocalCounter;
            ++this.myCursor;
        }

        public void write(byte[] b, int off, int len) throws IOException {
            int stopOffset = off + len;
            do {
                int bufferSizeLeft = stopOffset - off;
                int pageSizeLeft = this.myCursorLimit - this.myCursor;
                if (pageSizeLeft <= 0) {
                    this.allocatePage();
                    continue;
                }
                int dataToWrite = Math.min(bufferSizeLeft, pageSizeLeft);
                PagedMemoryMappedFile.this.myStorage.put(this.myCursor, b, off, dataToWrite);
                this.myCursor += dataToWrite;
                this.myLocalCounter += dataToWrite;
                off += dataToWrite;
            } while (off < stopOffset);
            PagedMemoryMappedFile.assertTrue(off == stopOffset);
        }

        private void allocatePage() {
            short currentClustersCount;
            int nextFromCurrent = PagedMemoryMappedFile.this.getNextPageLink(this.myCurrentPage);
            int nextSublingPage = PagedMemoryMappedFile.this.getNextSiblingPage(this.myCurrentPage);
            if (nextSublingPage != -1 && PagedMemoryMappedFile.this.isPageFree(nextSublingPage) && PagedMemoryMappedFile.this.getClustersCount(this.myCurrentPage) + PagedMemoryMappedFile.this.getClustersCount(nextSublingPage) < 32000) {
                PagedMemoryMappedFile.this.linkPages(this.myCurrentPage, 0);
                PagedMemoryMappedFile.this.markPageFree(this.myCurrentPage, true);
                PagedMemoryMappedFile.this.mergeWithPrev(nextSublingPage);
                PagedMemoryMappedFile.this.markPageFree(this.myCurrentPage, false);
                this.myCursorLimit = (this.myCurrentPage + PagedMemoryMappedFile.this.getClustersCount(this.myCurrentPage)) * PagedMemoryMappedFile.this.CLUSTER_SIZE - 7;
                return;
            }
            PagedMemoryMappedFile.this.closePage(this.myCurrentPage, this.myLocalCounter);
            int newPage = nextFromCurrent;
            if (newPage == 0) {
                newPage = PagedMemoryMappedFile.this.getFreePage();
            }
            if (this.myCurrentPage + (currentClustersCount = PagedMemoryMappedFile.this.getClustersCount(this.myCurrentPage)) == newPage && currentClustersCount + PagedMemoryMappedFile.this.getClustersCount(newPage) < 32000) {
                PagedMemoryMappedFile.this.linkPages(this.myCurrentPage, 0);
                PagedMemoryMappedFile.this.markPageFree(this.myCurrentPage, true);
                PagedMemoryMappedFile.this.mergeWithPrev(newPage);
                PagedMemoryMappedFile.this.markPageFree(this.myCurrentPage, false);
                this.myCursorLimit = (this.myCurrentPage + PagedMemoryMappedFile.this.getClustersCount(this.myCurrentPage)) * PagedMemoryMappedFile.this.CLUSTER_SIZE - 7;
                return;
            }
            PagedMemoryMappedFile.this.linkPages(this.myCurrentPage, newPage);
            PagedMemoryMappedFile.this.markPageFree(this.myCurrentPage, false);
            this.myCurrentPage = newPage;
            this.setCursor(this.myCurrentPage);
        }

        private void setCursor(int page) {
            PagedMemoryMappedFile.assertTrue(page > 0);
            this.myCursor = page * PagedMemoryMappedFile.this.CLUSTER_SIZE + 7;
            this.myCursorLimit = (page + PagedMemoryMappedFile.this.getClustersCount(page)) * PagedMemoryMappedFile.this.CLUSTER_SIZE - 7;
            this.myLocalCounter = 0;
        }

        public void close() throws IOException {
            super.close();
            PagedMemoryMappedFile.assertTrue(this.myLocalCounter > 0);
            PagedMemoryMappedFile.this.closePage(this.myCurrentPage, this.myLocalCounter);
            PagedMemoryMappedFile.this.linkPages(this.myCurrentPage, 0);
            PagedMemoryMappedFile.this.checkConsistency(false);
        }
    }
}

