/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.cache.query;

import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.util.Accountable;
import org.apache.lucene.util.RamUsageEstimator;
import org.elasticsearch.ElasticsearchIllegalArgumentException;
import org.elasticsearch.action.search.SearchType;
import org.elasticsearch.cluster.ClusterService;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.cache.Cache;
import org.elasticsearch.common.cache.CacheBuilder;
import org.elasticsearch.common.cache.RemovalListener;
import org.elasticsearch.common.cache.RemovalNotification;
import org.elasticsearch.common.cache.Weigher;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.hppc.ObjectOpenHashSet;
import org.elasticsearch.common.hppc.ObjectSet;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.io.stream.BytesStreamOutput;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.MemorySizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardState;
import org.elasticsearch.search.internal.SearchContext;
import org.elasticsearch.search.internal.ShardSearchRequest;
import org.elasticsearch.search.query.QueryPhase;
import org.elasticsearch.search.query.QuerySearchResult;
import org.elasticsearch.threadpool.ThreadPool;

public class IndicesQueryCache
extends AbstractComponent
implements RemovalListener<Key, Value> {
    public static final String INDEX_CACHE_QUERY_ENABLED = "index.cache.query.enable";
    public static final String INDICES_CACHE_QUERY_CLEAN_INTERVAL = "indices.cache.query.clean_interval";
    public static final String INDICES_CACHE_QUERY_SIZE = "indices.cache.query.size";
    public static final String INDICES_CACHE_QUERY_EXPIRE = "indices.cache.query.expire";
    public static final String INDICES_CACHE_QUERY_CONCURRENCY_LEVEL = "indices.cache.query.concurrency_level";
    private final ThreadPool threadPool;
    private final ClusterService clusterService;
    private final TimeValue cleanInterval;
    private final Reaper reaper;
    final ConcurrentMap<CleanupKey, Boolean> registeredClosedListeners = ConcurrentCollections.newConcurrentMap();
    final Set<CleanupKey> keysToClean = ConcurrentCollections.newConcurrentSet();
    private final String size;
    private final TimeValue expire;
    private final int concurrencyLevel;
    private volatile Cache<Key, Value> cache;

    @Inject
    public IndicesQueryCache(Settings settings, ClusterService clusterService, ThreadPool threadPool) {
        super(settings);
        this.clusterService = clusterService;
        this.threadPool = threadPool;
        this.cleanInterval = settings.getAsTime(INDICES_CACHE_QUERY_CLEAN_INTERVAL, TimeValue.timeValueSeconds(60L));
        this.size = settings.get(INDICES_CACHE_QUERY_SIZE, "1%");
        this.expire = settings.getAsTime(INDICES_CACHE_QUERY_EXPIRE, null);
        this.concurrencyLevel = settings.getAsInt(INDICES_CACHE_QUERY_CONCURRENCY_LEVEL, (Integer)16);
        if (this.concurrencyLevel <= 0) {
            throw new ElasticsearchIllegalArgumentException("concurrency_level must be > 0 but was: " + this.concurrencyLevel);
        }
        this.buildCache();
        this.reaper = new Reaper();
        threadPool.schedule(this.cleanInterval, "same", this.reaper);
    }

    private void buildCache() {
        long sizeInBytes = MemorySizeValue.parseBytesSizeValueOrHeapRatio(this.size).bytes();
        CacheBuilder<Key, Value> cacheBuilder = CacheBuilder.newBuilder().maximumWeight(sizeInBytes).weigher(new QueryCacheWeigher()).removalListener(this);
        cacheBuilder.concurrencyLevel(this.concurrencyLevel);
        if (this.expire != null) {
            cacheBuilder.expireAfterAccess(this.expire.millis(), TimeUnit.MILLISECONDS);
        }
        this.cache = cacheBuilder.build();
    }

    public void close() {
        this.reaper.close();
        this.cache.invalidateAll();
    }

    public void clear(IndexShard shard) {
        if (shard == null) {
            return;
        }
        this.keysToClean.add(new CleanupKey(shard, -1L));
        this.logger.trace("{} explicit cache clear", shard.shardId());
        this.reaper.reap();
    }

    @Override
    public void onRemoval(RemovalNotification<Key, Value> notification) {
        if (notification.getKey() == null) {
            return;
        }
        notification.getKey().shard.queryCache().onRemoval(notification);
    }

    public boolean canCache(ShardSearchRequest request, SearchContext context) {
        if (Strings.hasLength(request.templateSource())) {
            return false;
        }
        if (context.searchType() != SearchType.COUNT) {
            return false;
        }
        IndexMetaData index = this.clusterService.state().getMetaData().index(request.index());
        if (index == null) {
            return false;
        }
        if (request.queryCache() == null ? index.settings().getAsBoolean(INDEX_CACHE_QUERY_ENABLED, Boolean.FALSE) == false : request.queryCache() == false) {
            return false;
        }
        if (!(context.searcher().getIndexReader() instanceof DirectoryReader)) {
            return false;
        }
        return !context.nowInMillisUsed();
    }

    public void loadIntoContext(ShardSearchRequest request, SearchContext context, QueryPhase queryPhase) throws Exception {
        assert (this.canCache(request, context));
        Key key = IndicesQueryCache.buildKey(request, context);
        Loader loader = new Loader(queryPhase, context, key);
        Value value = this.cache.get(key, loader);
        if (loader.isLoaded()) {
            Boolean previous;
            key.shard.queryCache().onMiss();
            CleanupKey cleanupKey = new CleanupKey(context.indexShard(), ((DirectoryReader)context.searcher().getIndexReader()).getVersion());
            if (!this.registeredClosedListeners.containsKey(cleanupKey) && (previous = this.registeredClosedListeners.putIfAbsent(cleanupKey, Boolean.TRUE)) == null) {
                context.searcher().getIndexReader().addReaderClosedListener((IndexReader.ReaderClosedListener)cleanupKey);
            }
        } else {
            key.shard.queryCache().onHit();
            QuerySearchResult result = context.queryResult();
            result.readFromWithId(context.id(), value.reference.streamInput());
            result.shardTarget(context.shardTarget());
        }
    }

    private static Key buildKey(ShardSearchRequest request, SearchContext context) throws Exception {
        return new Key(context.indexShard(), ((DirectoryReader)context.searcher().getIndexReader()).getVersion(), request.cacheKey());
    }

    private class Reaper
    implements Runnable {
        private final ObjectSet<CleanupKey> currentKeysToClean = ObjectOpenHashSet.newInstance();
        private final ObjectSet<IndexShard> currentFullClean = ObjectOpenHashSet.newInstance();
        private volatile boolean closed;

        private Reaper() {
        }

        void close() {
            this.closed = true;
        }

        @Override
        public void run() {
            if (this.closed) {
                return;
            }
            if (IndicesQueryCache.this.keysToClean.isEmpty()) {
                this.schedule();
                return;
            }
            try {
                IndicesQueryCache.this.threadPool.executor("generic").execute(new Runnable(){

                    @Override
                    public void run() {
                        Reaper.this.reap();
                        Reaper.this.schedule();
                    }
                });
            }
            catch (EsRejectedExecutionException ex) {
                IndicesQueryCache.this.logger.debug("Can not run ReaderCleaner - execution rejected", ex, new Object[0]);
            }
        }

        private void schedule() {
            try {
                IndicesQueryCache.this.threadPool.schedule(IndicesQueryCache.this.cleanInterval, "same", this);
            }
            catch (EsRejectedExecutionException ex) {
                IndicesQueryCache.this.logger.debug("Can not schedule ReaderCleaner - execution rejected", ex, new Object[0]);
            }
        }

        synchronized void reap() {
            this.currentKeysToClean.clear();
            this.currentFullClean.clear();
            Iterator<CleanupKey> iterator = IndicesQueryCache.this.keysToClean.iterator();
            while (iterator.hasNext()) {
                CleanupKey cleanupKey = iterator.next();
                iterator.remove();
                if (cleanupKey.readerVersion == -1L || cleanupKey.indexShard.state() == IndexShardState.CLOSED) {
                    this.currentFullClean.add(cleanupKey.indexShard);
                    continue;
                }
                this.currentKeysToClean.add(cleanupKey);
            }
            if (!this.currentKeysToClean.isEmpty() || !this.currentFullClean.isEmpty()) {
                CleanupKey lookupKey = new CleanupKey(null, -1L);
                Iterator iterator2 = IndicesQueryCache.this.cache.asMap().keySet().iterator();
                while (iterator2.hasNext()) {
                    Key key = (Key)iterator2.next();
                    if (this.currentFullClean.contains(key.shard)) {
                        iterator2.remove();
                        continue;
                    }
                    lookupKey.indexShard = key.shard;
                    lookupKey.readerVersion = key.readerVersion;
                    if (!this.currentKeysToClean.contains(lookupKey)) continue;
                    iterator2.remove();
                }
            }
            IndicesQueryCache.this.cache.cleanUp();
            this.currentKeysToClean.clear();
            this.currentFullClean.clear();
        }
    }

    private class CleanupKey
    implements IndexReader.ReaderClosedListener {
        IndexShard indexShard;
        long readerVersion;

        private CleanupKey(IndexShard indexShard, long readerVersion) {
            this.indexShard = indexShard;
            this.readerVersion = readerVersion;
        }

        public void onClose(IndexReader reader) {
            Boolean remove = (Boolean)IndicesQueryCache.this.registeredClosedListeners.remove(this);
            if (remove != null) {
                IndicesQueryCache.this.keysToClean.add(this);
            }
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            CleanupKey that = (CleanupKey)o;
            if (this.readerVersion != that.readerVersion) {
                return false;
            }
            return this.indexShard.equals(that.indexShard);
        }

        public int hashCode() {
            int result = this.indexShard.hashCode();
            result = 31 * result + (int)(this.readerVersion ^ this.readerVersion >>> 32);
            return result;
        }
    }

    public static class Key
    implements Accountable {
        public final IndexShard shard;
        public final long readerVersion;
        public final BytesReference value;

        Key(IndexShard shard, long readerVersion, BytesReference value) {
            this.shard = shard;
            this.readerVersion = readerVersion;
            this.value = value;
        }

        public long ramBytesUsed() {
            return RamUsageEstimator.NUM_BYTES_OBJECT_REF + 8 + this.value.length();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            Key key = (Key)o;
            if (this.readerVersion != key.readerVersion) {
                return false;
            }
            if (!this.shard.equals(key.shard)) {
                return false;
            }
            return this.value.equals(key.value);
        }

        public int hashCode() {
            int result = this.shard.hashCode();
            result = 31 * result + (int)(this.readerVersion ^ this.readerVersion >>> 32);
            result = 31 * result + this.value.hashCode();
            return result;
        }
    }

    public static class Value
    implements Accountable {
        final BytesReference reference;
        final long ramBytesUsed;

        public Value(BytesReference reference, long ramBytesUsed) {
            this.reference = reference;
            this.ramBytesUsed = ramBytesUsed;
        }

        public long ramBytesUsed() {
            return this.ramBytesUsed;
        }
    }

    private static class Loader
    implements Callable<Value> {
        private final QueryPhase queryPhase;
        private final SearchContext context;
        private final Key key;
        private boolean loaded;

        Loader(QueryPhase queryPhase, SearchContext context, Key key) {
            this.queryPhase = queryPhase;
            this.context = context;
            this.key = key;
        }

        public boolean isLoaded() {
            return this.loaded;
        }

        @Override
        public Value call() throws Exception {
            this.queryPhase.execute(this.context);
            int expectedSizeInBytes = 512;
            try (BytesStreamOutput out = new BytesStreamOutput(512);){
                this.context.queryResult().writeToNoId(out);
                BytesReference reference = out.bytes();
                this.loaded = true;
                Value value = new Value(reference, out.ramBytesUsed());
                this.key.shard.queryCache().onCached(this.key, value);
                Value value2 = value;
                return value2;
            }
        }
    }

    private static class QueryCacheWeigher
    implements Weigher<Key, Value> {
        private QueryCacheWeigher() {
        }

        @Override
        public int weigh(Key key, Value value) {
            return (int)(key.ramBytesUsed() + value.ramBytesUsed());
        }
    }
}

