/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.network;

import java.io.Closeable;
import java.net.DatagramPacket;
import java.net.InetAddress;
import java.net.MulticastSocket;
import java.net.SocketAddress;
import java.net.SocketTimeoutException;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicBoolean;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.collect.Maps;
import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.logging.ESLoggerFactory;
import org.elasticsearch.common.settings.ImmutableSettings;
import org.elasticsearch.common.util.concurrent.EsExecutors;

public abstract class MulticastChannel
implements Closeable {
    protected final Listener listener;
    private AtomicBoolean closed = new AtomicBoolean();
    public static final String SHARED_CHANNEL_NAME = "#shared#";

    public static MulticastChannel getChannel(String name, boolean shared, Config config, Listener listener) throws Exception {
        if (!shared) {
            return new Plain(listener, name, config);
        }
        return Shared.getSharedChannel(listener, config);
    }

    protected MulticastChannel(Listener listener) {
        this.listener = listener;
    }

    public abstract void send(BytesReference var1) throws Exception;

    @Override
    public void close() {
        if (this.closed.compareAndSet(false, true)) {
            this.close(this.listener);
        }
    }

    protected abstract void close(Listener var1);

    private static class Plain
    extends MulticastChannel {
        private final ESLogger logger;
        private final Config config;
        private volatile MulticastSocket multicastSocket;
        private final DatagramPacket datagramPacketSend;
        private final DatagramPacket datagramPacketReceive;
        private final Object sendMutex = new Object();
        private final Object receiveMutex = new Object();
        private final Receiver receiver;
        private final Thread receiverThread;

        Plain(Listener listener, String name, Config config) throws Exception {
            super(listener);
            this.logger = ESLoggerFactory.getLogger(name);
            this.config = config;
            this.datagramPacketReceive = new DatagramPacket(new byte[config.bufferSize], config.bufferSize);
            this.datagramPacketSend = new DatagramPacket(new byte[config.bufferSize], config.bufferSize, InetAddress.getByName(config.group), config.port);
            this.multicastSocket = this.buildMulticastSocket(config);
            this.receiver = new Receiver();
            this.receiverThread = EsExecutors.daemonThreadFactory(ImmutableSettings.builder().put("name", name).build(), "discovery#multicast#receiver").newThread(this.receiver);
            this.receiverThread.start();
        }

        private MulticastSocket buildMulticastSocket(Config config) throws Exception {
            MulticastSocket multicastSocket = new MulticastSocket(config.port);
            try {
                multicastSocket.setTimeToLive(config.ttl);
                multicastSocket.setInterface(config.multicastInterface);
                multicastSocket.joinGroup(InetAddress.getByName(config.group));
                multicastSocket.setReceiveBufferSize(config.bufferSize);
                multicastSocket.setSendBufferSize(config.bufferSize);
                multicastSocket.setSoTimeout(60000);
            }
            catch (Throwable e) {
                try {
                    multicastSocket.close();
                }
                catch (Throwable throwable) {
                    // empty catch block
                }
                if (e instanceof Exception) {
                    throw (Exception)e;
                }
                throw new ElasticsearchException(e.getMessage(), e);
            }
            return multicastSocket;
        }

        public Config getConfig() {
            return this.config;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void send(BytesReference data) throws Exception {
            Object object = this.sendMutex;
            synchronized (object) {
                this.datagramPacketSend.setData(data.toBytes());
                this.multicastSocket.send(this.datagramPacketSend);
            }
        }

        @Override
        protected void close(Listener listener) {
            this.receiver.stop();
            this.receiverThread.interrupt();
            if (this.multicastSocket != null) {
                try {
                    this.multicastSocket.close();
                }
                catch (Throwable throwable) {
                    // empty catch block
                }
                this.multicastSocket = null;
            }
            try {
                this.receiverThread.join(10000L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }

        private class Receiver
        implements Runnable {
            private volatile boolean running = true;

            private Receiver() {
            }

            public void stop() {
                this.running = false;
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                while (this.running) {
                    try {
                        Object object = Plain.this.receiveMutex;
                        synchronized (object) {
                            try {
                                Plain.this.multicastSocket.receive(Plain.this.datagramPacketReceive);
                            }
                            catch (SocketTimeoutException ignore) {
                                continue;
                            }
                            catch (Exception e) {
                                if (this.running) {
                                    if (Plain.this.multicastSocket.isClosed()) {
                                        Plain.this.logger.warn("multicast socket closed while running, restarting...", new Object[0]);
                                        Plain.this.multicastSocket = Plain.this.buildMulticastSocket(Plain.this.config);
                                    } else {
                                        Plain.this.logger.warn("failed to receive packet, throttling...", e, new Object[0]);
                                        Thread.sleep(500L);
                                    }
                                }
                                continue;
                            }
                        }
                        if (Plain.this.datagramPacketReceive.getData().length <= 0) continue;
                        Plain.this.listener.onMessage(new BytesArray(Plain.this.datagramPacketReceive.getData()), Plain.this.datagramPacketReceive.getSocketAddress());
                    }
                    catch (Throwable e) {
                        if (!this.running) continue;
                        Plain.this.logger.warn("unexpected exception in multicast receiver", e, new Object[0]);
                    }
                }
            }
        }
    }

    private static final class Delegate
    extends MulticastChannel {
        private final MulticastChannel channel;

        Delegate(Listener listener, MulticastChannel channel) {
            super(listener);
            this.channel = channel;
        }

        @Override
        public void send(BytesReference data) throws Exception {
            this.channel.send(data);
        }

        @Override
        protected void close(Listener listener) {
            this.channel.close(listener);
        }
    }

    private static final class Shared
    extends MulticastChannel {
        private static final Map<Config, Shared> sharedChannels = Maps.newHashMap();
        private static final Object mutex = new Object();
        final Plain channel;
        private int refCount = 1;

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        static MulticastChannel getSharedChannel(Listener listener, Config config) throws Exception {
            Object object = mutex;
            synchronized (object) {
                Shared shared = sharedChannels.get(config);
                if (shared != null) {
                    shared.incRef();
                    ((MultiListener)shared.listener).add(listener);
                } else {
                    MultiListener multiListener = new MultiListener();
                    multiListener.add(listener);
                    shared = new Shared(multiListener, new Plain(multiListener, MulticastChannel.SHARED_CHANNEL_NAME, config));
                    sharedChannels.put(config, shared);
                }
                return new Delegate(listener, shared);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        static void close(Shared shared, Listener listener) {
            Object object = mutex;
            synchronized (object) {
                boolean removed = ((MultiListener)shared.listener).remove(listener);
                assert (removed) : "a listener should be removed";
                if (shared.decRef() == 0) {
                    assert (((MultiListener)shared.listener).listeners.isEmpty());
                    sharedChannels.remove(shared.channel.getConfig());
                    shared.channel.close();
                }
            }
        }

        Shared(MultiListener listener, Plain channel) {
            super(listener);
            this.channel = channel;
        }

        private void incRef() {
            ++this.refCount;
        }

        private int decRef() {
            --this.refCount;
            assert (this.refCount >= 0) : "illegal ref counting, close called multiple times";
            return this.refCount;
        }

        @Override
        public void send(BytesReference data) throws Exception {
            this.channel.send(data);
        }

        @Override
        public void close() {
            assert (false) : "Shared references should never be closed directly, only via Delegate";
        }

        @Override
        protected void close(Listener listener) {
            Shared.close(this, listener);
        }
    }

    public static class MultiListener
    implements Listener {
        private final CopyOnWriteArrayList<Listener> listeners = new CopyOnWriteArrayList();

        public void add(Listener listener) {
            this.listeners.add(listener);
        }

        public boolean remove(Listener listener) {
            return this.listeners.remove(listener);
        }

        @Override
        public void onMessage(BytesReference data, SocketAddress address) {
            for (Listener listener : this.listeners) {
                listener.onMessage(data, address);
            }
        }
    }

    public static interface Listener {
        public void onMessage(BytesReference var1, SocketAddress var2);
    }

    public static final class Config {
        public final int port;
        public final String group;
        public final int bufferSize;
        public final int ttl;
        public final InetAddress multicastInterface;

        public Config(int port, String group, int bufferSize, int ttl, InetAddress multicastInterface) {
            this.port = port;
            this.group = group;
            this.bufferSize = bufferSize;
            this.ttl = ttl;
            this.multicastInterface = multicastInterface;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Config config = (Config)o;
            if (this.bufferSize != config.bufferSize) {
                return false;
            }
            if (this.port != config.port) {
                return false;
            }
            if (this.ttl != config.ttl) {
                return false;
            }
            if (this.group != null ? !this.group.equals(config.group) : config.group != null) {
                return false;
            }
            return !(this.multicastInterface != null ? !this.multicastInterface.equals(config.multicastInterface) : config.multicastInterface != null);
        }

        public int hashCode() {
            int result = this.port;
            result = 31 * result + (this.group != null ? this.group.hashCode() : 0);
            result = 31 * result + this.bufferSize;
            result = 31 * result + this.ttl;
            result = 31 * result + (this.multicastInterface != null ? this.multicastInterface.hashCode() : 0);
            return result;
        }
    }
}

