/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.taskmanager;

import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.time.Duration;
import java.util.Arrays;
import javax.annotation.Nullable;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.ConfigurationUtils;
import org.apache.flink.configuration.IllegalConfigurationException;
import org.apache.flink.configuration.MemorySize;
import org.apache.flink.configuration.NettyShuffleEnvironmentOptions;
import org.apache.flink.core.memory.MemoryType;
import org.apache.flink.runtime.io.network.netty.NettyConfig;
import org.apache.flink.runtime.io.network.partition.BoundedBlockingSubpartitionType;
import org.apache.flink.runtime.util.ConfigurationParserUtils;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NettyShuffleEnvironmentConfiguration {
    private static final Logger LOG = LoggerFactory.getLogger(NettyShuffleEnvironmentConfiguration.class);
    private final int numNetworkBuffers;
    private final int networkBufferSize;
    private final int partitionRequestInitialBackoff;
    private final int partitionRequestMaxBackoff;
    private final int networkBuffersPerChannel;
    private final int floatingNetworkBuffersPerGate;
    private final Duration requestSegmentsTimeout;
    private final boolean isCreditBased;
    private final boolean isNetworkDetailedMetrics;
    private final NettyConfig nettyConfig;
    private final String[] tempDirs;
    private final BoundedBlockingSubpartitionType blockingSubpartitionType;
    private final boolean forcePartitionReleaseOnConsumption;

    public NettyShuffleEnvironmentConfiguration(int numNetworkBuffers, int networkBufferSize, int partitionRequestInitialBackoff, int partitionRequestMaxBackoff, int networkBuffersPerChannel, int floatingNetworkBuffersPerGate, Duration requestSegmentsTimeout, boolean isCreditBased, boolean isNetworkDetailedMetrics, @Nullable NettyConfig nettyConfig, String[] tempDirs, BoundedBlockingSubpartitionType blockingSubpartitionType, boolean forcePartitionReleaseOnConsumption) {
        this.numNetworkBuffers = numNetworkBuffers;
        this.networkBufferSize = networkBufferSize;
        this.partitionRequestInitialBackoff = partitionRequestInitialBackoff;
        this.partitionRequestMaxBackoff = partitionRequestMaxBackoff;
        this.networkBuffersPerChannel = networkBuffersPerChannel;
        this.floatingNetworkBuffersPerGate = floatingNetworkBuffersPerGate;
        this.requestSegmentsTimeout = (Duration)Preconditions.checkNotNull((Object)requestSegmentsTimeout);
        this.isCreditBased = isCreditBased;
        this.isNetworkDetailedMetrics = isNetworkDetailedMetrics;
        this.nettyConfig = nettyConfig;
        this.tempDirs = (String[])Preconditions.checkNotNull((Object)tempDirs);
        this.blockingSubpartitionType = (BoundedBlockingSubpartitionType)((Object)Preconditions.checkNotNull((Object)((Object)blockingSubpartitionType)));
        this.forcePartitionReleaseOnConsumption = forcePartitionReleaseOnConsumption;
    }

    public int numNetworkBuffers() {
        return this.numNetworkBuffers;
    }

    public int networkBufferSize() {
        return this.networkBufferSize;
    }

    public int partitionRequestInitialBackoff() {
        return this.partitionRequestInitialBackoff;
    }

    public int partitionRequestMaxBackoff() {
        return this.partitionRequestMaxBackoff;
    }

    public int networkBuffersPerChannel() {
        return this.networkBuffersPerChannel;
    }

    public int floatingNetworkBuffersPerGate() {
        return this.floatingNetworkBuffersPerGate;
    }

    public Duration getRequestSegmentsTimeout() {
        return this.requestSegmentsTimeout;
    }

    public NettyConfig nettyConfig() {
        return this.nettyConfig;
    }

    public boolean isCreditBased() {
        return this.isCreditBased;
    }

    public boolean isNetworkDetailedMetrics() {
        return this.isNetworkDetailedMetrics;
    }

    public String[] getTempDirs() {
        return this.tempDirs;
    }

    public BoundedBlockingSubpartitionType getBlockingSubpartitionType() {
        return this.blockingSubpartitionType;
    }

    public boolean isForcePartitionReleaseOnConsumption() {
        return this.forcePartitionReleaseOnConsumption;
    }

    public static NettyShuffleEnvironmentConfiguration fromConfiguration(Configuration configuration, long maxJvmHeapMemory, boolean localTaskManagerCommunication, InetAddress taskManagerAddress) {
        int dataport = NettyShuffleEnvironmentConfiguration.getDataport(configuration);
        int pageSize = ConfigurationParserUtils.getPageSize(configuration);
        int numberOfNetworkBuffers = NettyShuffleEnvironmentConfiguration.calculateNumberOfNetworkBuffers(configuration, maxJvmHeapMemory);
        NettyConfig nettyConfig = NettyShuffleEnvironmentConfiguration.createNettyConfig(configuration, localTaskManagerCommunication, taskManagerAddress, dataport);
        int initialRequestBackoff = configuration.getInteger(NettyShuffleEnvironmentOptions.NETWORK_REQUEST_BACKOFF_INITIAL);
        int maxRequestBackoff = configuration.getInteger(NettyShuffleEnvironmentOptions.NETWORK_REQUEST_BACKOFF_MAX);
        int buffersPerChannel = configuration.getInteger(NettyShuffleEnvironmentOptions.NETWORK_BUFFERS_PER_CHANNEL);
        int extraBuffersPerGate = configuration.getInteger(NettyShuffleEnvironmentOptions.NETWORK_EXTRA_BUFFERS_PER_GATE);
        boolean isCreditBased = nettyConfig != null && configuration.getBoolean(NettyShuffleEnvironmentOptions.NETWORK_CREDIT_MODEL);
        boolean isNetworkDetailedMetrics = configuration.getBoolean(NettyShuffleEnvironmentOptions.NETWORK_DETAILED_METRICS);
        String[] tempDirs = ConfigurationUtils.parseTempDirectories((Configuration)configuration);
        Duration requestSegmentsTimeout = Duration.ofMillis(configuration.getLong(NettyShuffleEnvironmentOptions.NETWORK_EXCLUSIVE_BUFFERS_REQUEST_TIMEOUT_MILLISECONDS));
        BoundedBlockingSubpartitionType blockingSubpartitionType = NettyShuffleEnvironmentConfiguration.getBlockingSubpartitionType(configuration);
        boolean forcePartitionReleaseOnConsumption = configuration.getBoolean(NettyShuffleEnvironmentOptions.FORCE_PARTITION_RELEASE_ON_CONSUMPTION);
        return new NettyShuffleEnvironmentConfiguration(numberOfNetworkBuffers, pageSize, initialRequestBackoff, maxRequestBackoff, buffersPerChannel, extraBuffersPerGate, requestSegmentsTimeout, isCreditBased, isNetworkDetailedMetrics, nettyConfig, tempDirs, blockingSubpartitionType, forcePartitionReleaseOnConsumption);
    }

    @VisibleForTesting
    public static long calculateNewNetworkBufferMemory(Configuration config, long maxJvmHeapMemory) {
        long jvmHeapNoNet;
        MemoryType memoryType = ConfigurationParserUtils.getMemoryType(config);
        if (memoryType == MemoryType.HEAP) {
            jvmHeapNoNet = maxJvmHeapMemory;
        } else if (memoryType == MemoryType.OFF_HEAP) {
            long configuredMemory = ConfigurationParserUtils.getManagedMemorySize(config) << 20;
            jvmHeapNoNet = configuredMemory > 0L ? maxJvmHeapMemory + configuredMemory : (long)((double)maxJvmHeapMemory / (1.0 - (double)ConfigurationParserUtils.getManagedMemoryFraction(config)));
        } else {
            throw new RuntimeException("No supported memory type detected.");
        }
        float networkBufFraction = config.getFloat(NettyShuffleEnvironmentOptions.NETWORK_BUFFERS_MEMORY_FRACTION);
        long networkBufSize = (long)((double)jvmHeapNoNet / (1.0 - (double)networkBufFraction) * (double)networkBufFraction);
        return NettyShuffleEnvironmentConfiguration.calculateNewNetworkBufferMemory(config, networkBufSize, maxJvmHeapMemory);
    }

    public static long calculateNetworkBufferMemory(long totalJavaMemorySize, Configuration config) {
        long networkBufBytes;
        int segmentSize = ConfigurationParserUtils.getPageSize(config);
        if (NettyShuffleEnvironmentConfiguration.hasNewNetworkConfig(config)) {
            float networkBufFraction = config.getFloat(NettyShuffleEnvironmentOptions.NETWORK_BUFFERS_MEMORY_FRACTION);
            long networkBufSize = (long)((float)totalJavaMemorySize * networkBufFraction);
            networkBufBytes = NettyShuffleEnvironmentConfiguration.calculateNewNetworkBufferMemory(config, networkBufSize, totalJavaMemorySize);
        } else {
            int numNetworkBuffers = config.getInteger(NettyShuffleEnvironmentOptions.NETWORK_NUM_BUFFERS);
            networkBufBytes = (long)numNetworkBuffers * (long)segmentSize;
            NettyShuffleEnvironmentConfiguration.checkOldNetworkConfig(numNetworkBuffers);
            ConfigurationParserUtils.checkConfigParameter(networkBufBytes < totalJavaMemorySize, networkBufBytes, NettyShuffleEnvironmentOptions.NETWORK_NUM_BUFFERS.key(), "Network buffer memory size too large: " + networkBufBytes + " >= " + totalJavaMemorySize + " (total JVM memory size)");
        }
        return networkBufBytes;
    }

    private static long calculateNewNetworkBufferMemory(Configuration config, long networkBufSize, long maxJvmHeapMemory) {
        float networkBufFraction = config.getFloat(NettyShuffleEnvironmentOptions.NETWORK_BUFFERS_MEMORY_FRACTION);
        long networkBufMin = MemorySize.parse((String)config.getString(NettyShuffleEnvironmentOptions.NETWORK_BUFFERS_MEMORY_MIN)).getBytes();
        long networkBufMax = MemorySize.parse((String)config.getString(NettyShuffleEnvironmentOptions.NETWORK_BUFFERS_MEMORY_MAX)).getBytes();
        int pageSize = ConfigurationParserUtils.getPageSize(config);
        NettyShuffleEnvironmentConfiguration.checkNewNetworkConfig(pageSize, networkBufFraction, networkBufMin, networkBufMax);
        long networkBufBytes = Math.min(networkBufMax, Math.max(networkBufMin, networkBufSize));
        ConfigurationParserUtils.checkConfigParameter(networkBufBytes < maxJvmHeapMemory, "(" + networkBufFraction + ", " + networkBufMin + ", " + networkBufMax + ")", "(" + NettyShuffleEnvironmentOptions.NETWORK_BUFFERS_MEMORY_FRACTION.key() + ", " + NettyShuffleEnvironmentOptions.NETWORK_BUFFERS_MEMORY_MIN.key() + ", " + NettyShuffleEnvironmentOptions.NETWORK_BUFFERS_MEMORY_MAX.key() + ")", "Network buffer memory size too large: " + networkBufBytes + " >= " + maxJvmHeapMemory + " (maximum JVM memory size)");
        return networkBufBytes;
    }

    private static void checkOldNetworkConfig(int numNetworkBuffers) {
        ConfigurationParserUtils.checkConfigParameter(numNetworkBuffers > 0, numNetworkBuffers, NettyShuffleEnvironmentOptions.NETWORK_NUM_BUFFERS.key(), "Must have at least one network buffer");
    }

    private static void checkNewNetworkConfig(int pageSize, float networkBufFraction, long networkBufMin, long networkBufMax) throws IllegalConfigurationException {
        ConfigurationParserUtils.checkConfigParameter(networkBufFraction > 0.0f && networkBufFraction < 1.0f, Float.valueOf(networkBufFraction), NettyShuffleEnvironmentOptions.NETWORK_BUFFERS_MEMORY_FRACTION.key(), "Network buffer memory fraction of the free memory must be between 0.0 and 1.0");
        ConfigurationParserUtils.checkConfigParameter(networkBufMin >= (long)pageSize, networkBufMin, NettyShuffleEnvironmentOptions.NETWORK_BUFFERS_MEMORY_MIN.key(), "Minimum memory for network buffers must allow at least one network buffer with respect to the memory segment size");
        ConfigurationParserUtils.checkConfigParameter(networkBufMax >= (long)pageSize, networkBufMax, NettyShuffleEnvironmentOptions.NETWORK_BUFFERS_MEMORY_MAX.key(), "Maximum memory for network buffers must allow at least one network buffer with respect to the memory segment size");
        ConfigurationParserUtils.checkConfigParameter(networkBufMax >= networkBufMin, networkBufMax, NettyShuffleEnvironmentOptions.NETWORK_BUFFERS_MEMORY_MAX.key(), "Maximum memory for network buffers must not be smaller than minimum memory (" + NettyShuffleEnvironmentOptions.NETWORK_BUFFERS_MEMORY_MAX.key() + ": " + networkBufMin + ")");
    }

    @VisibleForTesting
    public static boolean hasNewNetworkConfig(Configuration config) {
        return config.contains(NettyShuffleEnvironmentOptions.NETWORK_BUFFERS_MEMORY_FRACTION) || config.contains(NettyShuffleEnvironmentOptions.NETWORK_BUFFERS_MEMORY_MIN) || config.contains(NettyShuffleEnvironmentOptions.NETWORK_BUFFERS_MEMORY_MAX) || !config.contains(NettyShuffleEnvironmentOptions.NETWORK_NUM_BUFFERS);
    }

    private static int getDataport(Configuration configuration) {
        int dataport = configuration.getInteger(NettyShuffleEnvironmentOptions.DATA_PORT);
        ConfigurationParserUtils.checkConfigParameter(dataport >= 0, dataport, NettyShuffleEnvironmentOptions.DATA_PORT.key(), "Leave config parameter empty or use 0 to let the system choose a port automatically.");
        return dataport;
    }

    private static int calculateNumberOfNetworkBuffers(Configuration configuration, long maxJvmHeapMemory) {
        int numberOfNetworkBuffers;
        if (!NettyShuffleEnvironmentConfiguration.hasNewNetworkConfig(configuration)) {
            numberOfNetworkBuffers = configuration.getInteger(NettyShuffleEnvironmentOptions.NETWORK_NUM_BUFFERS);
            NettyShuffleEnvironmentConfiguration.checkOldNetworkConfig(numberOfNetworkBuffers);
        } else {
            long networkMemorySize;
            long numberOfNetworkBuffersLong;
            if (configuration.contains(NettyShuffleEnvironmentOptions.NETWORK_NUM_BUFFERS)) {
                LOG.info("Ignoring old (but still present) network buffer configuration via {}.", (Object)NettyShuffleEnvironmentOptions.NETWORK_NUM_BUFFERS.key());
            }
            if ((numberOfNetworkBuffersLong = (networkMemorySize = NettyShuffleEnvironmentConfiguration.calculateNewNetworkBufferMemory(configuration, maxJvmHeapMemory)) / (long)ConfigurationParserUtils.getPageSize(configuration)) > Integer.MAX_VALUE) {
                throw new IllegalArgumentException("The given number of memory bytes (" + networkMemorySize + ") corresponds to more than MAX_INT pages.");
            }
            numberOfNetworkBuffers = (int)numberOfNetworkBuffersLong;
        }
        return numberOfNetworkBuffers;
    }

    @Nullable
    private static NettyConfig createNettyConfig(Configuration configuration, boolean localTaskManagerCommunication, InetAddress taskManagerAddress, int dataport) {
        NettyConfig nettyConfig;
        if (!localTaskManagerCommunication) {
            InetSocketAddress taskManagerInetSocketAddress = new InetSocketAddress(taskManagerAddress, dataport);
            nettyConfig = new NettyConfig(taskManagerInetSocketAddress.getAddress(), taskManagerInetSocketAddress.getPort(), ConfigurationParserUtils.getPageSize(configuration), ConfigurationParserUtils.getSlot(configuration), configuration);
        } else {
            nettyConfig = null;
        }
        return nettyConfig;
    }

    private static BoundedBlockingSubpartitionType getBlockingSubpartitionType(Configuration config) {
        String transport;
        switch (transport = config.getString(NettyShuffleEnvironmentOptions.NETWORK_BOUNDED_BLOCKING_SUBPARTITION_TYPE)) {
            case "mmap": {
                return BoundedBlockingSubpartitionType.FILE_MMAP;
            }
            case "file": {
                return BoundedBlockingSubpartitionType.FILE;
            }
        }
        return BoundedBlockingSubpartitionType.AUTO;
    }

    public int hashCode() {
        int result = 1;
        result = 31 * result + this.numNetworkBuffers;
        result = 31 * result + this.networkBufferSize;
        result = 31 * result + this.partitionRequestInitialBackoff;
        result = 31 * result + this.partitionRequestMaxBackoff;
        result = 31 * result + this.networkBuffersPerChannel;
        result = 31 * result + this.floatingNetworkBuffersPerGate;
        result = 31 * result + this.requestSegmentsTimeout.hashCode();
        result = 31 * result + (this.isCreditBased ? 1 : 0);
        result = 31 * result + (this.nettyConfig != null ? this.nettyConfig.hashCode() : 0);
        result = 31 * result + Arrays.hashCode(this.tempDirs);
        result = 31 * result + (this.forcePartitionReleaseOnConsumption ? 1 : 0);
        return result;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null || this.getClass() != obj.getClass()) {
            return false;
        }
        NettyShuffleEnvironmentConfiguration that = (NettyShuffleEnvironmentConfiguration)obj;
        return this.numNetworkBuffers == that.numNetworkBuffers && this.networkBufferSize == that.networkBufferSize && this.partitionRequestInitialBackoff == that.partitionRequestInitialBackoff && this.partitionRequestMaxBackoff == that.partitionRequestMaxBackoff && this.networkBuffersPerChannel == that.networkBuffersPerChannel && this.floatingNetworkBuffersPerGate == that.floatingNetworkBuffersPerGate && this.requestSegmentsTimeout.equals(that.requestSegmentsTimeout) && this.isCreditBased == that.isCreditBased && (this.nettyConfig != null ? this.nettyConfig.equals(that.nettyConfig) : that.nettyConfig == null) && Arrays.equals(this.tempDirs, that.tempDirs) && this.forcePartitionReleaseOnConsumption == that.forcePartitionReleaseOnConsumption;
    }

    public String toString() {
        return "NettyShuffleEnvironmentConfiguration{, numNetworkBuffers=" + this.numNetworkBuffers + ", networkBufferSize=" + this.networkBufferSize + ", partitionRequestInitialBackoff=" + this.partitionRequestInitialBackoff + ", partitionRequestMaxBackoff=" + this.partitionRequestMaxBackoff + ", networkBuffersPerChannel=" + this.networkBuffersPerChannel + ", floatingNetworkBuffersPerGate=" + this.floatingNetworkBuffersPerGate + ", requestSegmentsTimeout=" + this.requestSegmentsTimeout + ", isCreditBased=" + this.isCreditBased + ", nettyConfig=" + this.nettyConfig + ", tempDirs=" + Arrays.toString(this.tempDirs) + ", forcePartitionReleaseOnConsumption=" + this.forcePartitionReleaseOnConsumption + '}';
    }
}

